/**
* This file is a part of the Cairo-Dock project
*
* Copyright : (C) see the 'copyright' file.
* E-mail    : see the 'copyright' file.
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 3
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef __CAIRO_DOCK_APPLET_FACILITY__
#define  __CAIRO_DOCK_APPLET_FACILITY__

#include "cairo-dock-struct.h"
#include "cairo-dock-modules.h"
#include "cairo-dock-menu.h"
#include "cairo-dock-applet-factory.h"
G_BEGIN_DECLS

/**
*@file cairo-dock-applet-facility.h A collection of useful macros for applets.
* Macros provides a normalized API that will :
*  - lets you perform complex operations with a minimum amount of code
*  - ensures a bug-free functioning
*  - masks the internal complexity
*  - allows a normalized and easy-to-maintain code amongst all the applets.
*/

/** Apply a surface on a context, with a zoom and a transparency factor. The context is cleared beforehand with the default icon background.
*@param pIconContext the drawing context; is not altered by the function.
*@param pSurface the surface to apply.
*@param fScale zoom fastor.
*@param fAlpha transparency in [0,1].
*@param pIcon the icon.
*@param pContainer the container of the icon.
*/
void cairo_dock_set_icon_surface_full (cairo_t *pIconContext, cairo_surface_t *pSurface, double fScale, double fAlpha, Icon *pIcon, CairoContainer *pContainer);

/** Apply a surface on a context. The context is cleared beforehand with the default icon background..
*@param pIconContext the drawing context; is not altered by the function.
*@param pSurface the surface to apply.
*/
#define cairo_dock_set_icon_surface(pIconContext, pSurface) cairo_dock_set_icon_surface_full (pIconContext, pSurface, 1, 1, NULL, NULL)

/** Draw a bar at the bottom of an Icon, with a gradation from red to green and a given length.
*@param pIconContext the drawing context; is not altered by the function.
*@param fValue the value representing a percentage, in [-1,1]. if negative, the gradation is inverted, and the absolute value is used.
*@param pIcon the icon.
*@param pContainer the container of the icon.
*/
void cairo_dock_draw_bar_on_icon (cairo_t *pIconContext, double fValue, Icon *pIcon, CairoContainer *pContainer);

void cairo_dock_set_icon_surface_with_bar (cairo_t *pIconContext, cairo_surface_t *pSurface, double fValue, Icon *pIcon, CairoContainer *pContainer);

/** Apply a surface on the context of an icon, clearing it beforehand, and adding the reflect.
*@param pIconContext the drawing context; is not altered by the function.
*@param pSurface the surface to apply.
*@param pIcon the icon.
*@param pContainer the container of the icon.
*/
void cairo_dock_set_icon_surface_with_reflect (cairo_t *pIconContext, cairo_surface_t *pSurface, Icon *pIcon, CairoContainer *pContainer);

/** Apply an image on the context of an icon, clearing it beforehand, and adding the reflect.
*@param pIconContext the drawing context; is not altered by the function.
*@param cImagePath path of an image to apply on the icon.
*@param pIcon the icon.
*@param pContainer the container of the icon.
*/
void cairo_dock_set_image_on_icon (cairo_t *pIconContext, const gchar *cImagePath, Icon *pIcon, CairoContainer *pContainer);


void cairo_dock_set_hours_minutes_as_quick_info (cairo_t *pSourceContext, Icon *pIcon, CairoContainer *pContainer, int iTimeInSeconds);
void cairo_dock_set_minutes_secondes_as_quick_info (cairo_t *pSourceContext, Icon *pIcon, CairoContainer *pContainer, int iTimeInSeconds);

/** Convert a size in bytes into a readable format.
*@param iSizeInBytes size in bytes.
*@return a newly allocated string.
*/
gchar *cairo_dock_get_human_readable_size (long long int iSizeInBytes);
void cairo_dock_set_size_as_quick_info (cairo_t *pSourceContext, Icon *pIcon, CairoContainer *pContainer, long long int iSizeInBytes);

/// type of possible display on a Icon.
typedef enum {
	/// don't display anything.
	CAIRO_DOCK_INFO_NONE = 0,
	/// display info on the icon (as quick-info).
	CAIRO_DOCK_INFO_ON_ICON,
	/// display on the label of the icon.
	CAIRO_DOCK_INFO_ON_LABEL,
	CAIRO_DOCK_NB_INFO_DISPLAY
} CairoDockInfoDisplay;


gchar *cairo_dock_get_theme_path_for_module (GKeyFile *pKeyFile, gchar *cGroupName, gchar *cKeyName, gboolean *bFlushConfFileNeeded, gchar *cDefaultThemeName, const gchar *cShareThemesDir, const gchar *cExtraDirName);

GtkWidget *cairo_dock_create_sub_menu (const gchar *cLabel, GtkWidget *pMenu, const gchar *cImage);


/** Play a sound, through Alsa or PulseAudio.
*@param cSoundPath path to an audio file.
*/
void cairo_dock_play_sound (const gchar *cSoundPath);


/** Get the Gnome's version.
*@param iMajor pointer to the major.
*@param iMinor pointer to the minor.
*@param iMicro pointer to the micro.
*/
void cairo_dock_get_gnome_version (int *iMajor, int *iMinor, int *iMicro);


void cairo_dock_pop_up_about_applet (GtkMenuItem *menu_item, CairoDockModuleInstance *pModuleInstance);

void cairo_dock_open_module_config_on_demand (int iClickedButton, GtkWidget *pInteractiveWidget, CairoDockModuleInstance *pModuleInstance, CairoDialog *pDialog);



  ////////////
 // CONFIG //
////////////

/** Get the value of a 'boolean' from the conf file.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@param bDefaultValue default value if the group/key is not found (typically if the key is new).
*@return a gboolean.
*/
#define CD_CONFIG_GET_BOOLEAN_WITH_DEFAULT(cGroupName, cKeyName, bDefaultValue) cairo_dock_get_boolean_key_value (pKeyFile, cGroupName, cKeyName, &bFlushConfFileNeeded, bDefaultValue, NULL, NULL)
/** Get the value of a 'boolean' from the conf file, with TRUE as default value.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@return a gboolean.
*/
#define CD_CONFIG_GET_BOOLEAN(cGroupName, cKeyName) CD_CONFIG_GET_BOOLEAN_WITH_DEFAULT (cGroupName, cKeyName, TRUE)

/** Get the value of an 'integer' from the conf file.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@param iDefaultValue default value if the group/key is not found (typically if the key is new).
*@return an integer.
*/
#define CD_CONFIG_GET_INTEGER_WITH_DEFAULT(cGroupName, cKeyName, iDefaultValue) cairo_dock_get_integer_key_value (pKeyFile, cGroupName, cKeyName, &bFlushConfFileNeeded, iDefaultValue, NULL, NULL)
/** Get the value of a 'entier' from the conf file, with 0 as default value.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@return an integer.
*/
#define CD_CONFIG_GET_INTEGER(cGroupName, cKeyName) CD_CONFIG_GET_INTEGER_WITH_DEFAULT (cGroupName, cKeyName, 0)

/** Get the value of a 'double' from the conf file.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@param fDefaultValue default value if the group/key is not found (typically if the key is new).
*@return a double.
*/
#define CD_CONFIG_GET_DOUBLE_WITH_DEFAULT(cGroupName, cKeyName, fDefaultValue) cairo_dock_get_double_key_value (pKeyFile, cGroupName, cKeyName, &bFlushConfFileNeeded, 0., NULL, NULL)
/** Get the value of a 'double' from the conf file, with 0. as default value.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@return a double.
*/
#define CD_CONFIG_GET_DOUBLE(cGroupName, cKeyName) CD_CONFIG_GET_DOUBLE_WITH_DEFAULT (cGroupName, cKeyName, 0.)

/** Get the value of an 'integers list' from the conf file.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@param iNbElements number of elements to get from the conf file.
*@param iValueBuffer buffer to fill with the values.
*/
#define CD_CONFIG_GET_INTEGER_LIST(cGroupName, cKeyName, iNbElements, iValueBuffer) \
cairo_dock_get_integer_list_key_value (pKeyFile, cGroupName, cKeyName, &bFlushConfFileNeeded, iValueBuffer, iNbElements, NULL, NULL, NULL)

/** Get the value of a 'string' from the conf file.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@param cDefaultValue default value if the group/key is not found (typically if the key is new). can be NULL.
*@return a newly allocated string.
*/
#define CD_CONFIG_GET_STRING_WITH_DEFAULT(cGroupName, cKeyName, cDefaultValue) cairo_dock_get_string_key_value (pKeyFile, cGroupName, cKeyName, &bFlushConfFileNeeded, cDefaultValue, NULL, NULL)
/** Get the value of a 'string' from the conf file, with NULL as default value.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@return a newly allocated string.
*/
#define CD_CONFIG_GET_STRING(cGroupName, cKeyName) CD_CONFIG_GET_STRING_WITH_DEFAULT (cGroupName, cKeyName, NULL)

/** Get the value of a 'file' from the conf file, with NULL as default value. If the value is a file name (not a path), it is supposed to be in the Cairo-Dock's current theme folder. If the value is NULL, the default file is used, taken at the applet's data folder, but the conf file is not updated with this value.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@param cDefaultFileName defaul tfile if none is specified in the conf file.
*@return a newly allocated string giving the complete path of the file.
*/
#define CD_CONFIG_GET_FILE_PATH(cGroupName, cKeyName, cDefaultFileName) cairo_dock_get_file_path_key_value (pKeyFile, cGroupName, cKeyName, &bFlushConfFileNeeded, NULL, NULL, MY_APPLET_SHARE_DATA_DIR, cDefaultFileName)

/** Get the value of a 'strings list' from the conf file.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@param length pointer to the number of strings that were extracted from the conf file.
*@param cDefaultValues default value if the group/key is not found (typically if the key is new). It is a string with words separated by ';'. It can be NULL.
*@return a table of strings, to be freeed with 'g_strfreev'.
*/
#define CD_CONFIG_GET_STRING_LIST_WITH_DEFAULT(cGroupName, cKeyName, length, cDefaultValues) cairo_dock_get_string_list_key_value (pKeyFile, cGroupName, cKeyName, &bFlushConfFileNeeded, length, cDefaultValues, NULL, NULL)
/** Get the value of a 'strings list' from the conf file, with NULL as default value.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@param length pointer to the number of strings that were extracted from the conf file.
*@return a table of strings, to be freeed with 'g_strfreev'.
*/
#define CD_CONFIG_GET_STRING_LIST(cGroupName, cKeyName, length) CD_CONFIG_GET_STRING_LIST_WITH_DEFAULT(cGroupName, cKeyName, length, NULL)

/** Get the value of a 'color' in the RGBA format from the conf file.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@param pColorBuffer a table of 4 'double' already allocated, that will be filled with the color components.
*@param pDefaultColor default value if the group/key is not found (typically if the key is new). It is a table of 4 'double'. It can be NULL.
*/
#define CD_CONFIG_GET_COLOR_WITH_DEFAULT(cGroupName, cKeyName, pColorBuffer, pDefaultColor) cairo_dock_get_double_list_key_value (pKeyFile, cGroupName, cKeyName, &bFlushConfFileNeeded, (double*)pColorBuffer, 4, pDefaultColor, NULL, NULL)
/** Get the value of a 'color' in the RGBA format from the conf file, with NULL as default value.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@param pColorBuffer a table of 4 'double' already allocated, that will be filled with the color components.
*/
#define CD_CONFIG_GET_COLOR(cGroupName, cKeyName, pColorBuffer) CD_CONFIG_GET_COLOR_WITH_DEFAULT(cGroupName, cKeyName, pColorBuffer, NULL)
/** Get the value of a 'color' in the RGB format from the conf file.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@param pColorBuffer a table of 3 'double' already allocated, that will be filled with the color components.
*@param pDefaultColor default value if the group/key is not found (typically if the key is new). It is a table of 3 'double'. It can be NULL.
*/
#define CD_CONFIG_GET_COLOR_RVB_WITH_DEFAULT(cGroupName, cKeyName, pColorBuffer, pDefaultColor) cairo_dock_get_double_list_key_value (pKeyFile, cGroupName, cKeyName, &bFlushConfFileNeeded, pColorBuffer, 3, pDefaultColor, NULL, NULL)
/** Get the value of a 'color' in the RGB format from the conf file, with NULL as default value.
*@param cGroupName name of the group in the conf file.
*@param cKeyName name of the key in the conf file.
*@param pColorBuffer a table of 3 'double' already allocated, that will be filled with the color components.
*/
#define CD_CONFIG_GET_COLOR_RVB(cGroupName, cKeyName, pColorBuffer) CD_CONFIG_GET_COLOR_RVB_WITH_DEFAULT(cGroupName, cKeyName, pColorBuffer, NULL)

/** Get the complete path of a theme in the conf file.
*@param cGroupName name of the group (in the conf file).
*@param cKeyName name of the key (in the conf file).
*@param cThemeDirName name of the folder containing the local, user, and distant themes.
*@param cDefaultThemeName default value, if the key/group/theme doesn't exist.
*@return Path to the folder of the theme, in a newly allocated string.
*/
#define CD_CONFIG_GET_THEME_PATH(cGroupName, cKeyName, cThemeDirName, cDefaultThemeName) \
	__extension__ ({\
	gchar *_cThemePath = cairo_dock_get_theme_path_for_module (pKeyFile, cGroupName, cKeyName, &bFlushConfFileNeeded, cDefaultThemeName, MY_APPLET_SHARE_DATA_DIR"/"cThemeDirName, MY_APPLET_USER_DATA_DIR);\
	if (_cThemePath == NULL) {\
		const gchar *_cMessage = _("the theme couldn't be found; the default theme will be used instead.\n You can change this by opening the configuration of this module; do you want to do it now ?");\
		Icon *_pIcon = cairo_dock_get_dialogless_icon ();\
		gchar *_cQuestion = g_strdup_printf ("%s : %s", myApplet->pModule->pVisitCard->cModuleName, _cMessage);\
		cairo_dock_show_dialog_with_question (_cQuestion, _pIcon, CAIRO_CONTAINER (g_pMainDock), MY_APPLET_SHARE_DATA_DIR"/"MY_APPLET_ICON_FILE, (CairoDockActionOnAnswerFunc) cairo_dock_open_module_config_on_demand, myApplet, NULL);\
		g_free (_cQuestion); }\
	_cThemePath; })

/** Get the complete path of a Gauge theme in the conf file.
*@param cGroupName name of the group (in the conf file).
*@param cKeyName name of the key (in the conf file).
*/
#define CD_CONFIG_GET_GAUGE_THEME(cGroupName, cKeyName) \
	__extension__ ({\
	gchar *_cThemePath = cairo_dock_get_gauge_key_value(CD_APPLET_MY_CONF_FILE, pKeyFile, cGroupName, cKeyName, &bFlushConfFileNeeded, "turbo-night-fuel");\
	if (_cThemePath == NULL) {\
		const gchar *_cMessage = _("the gauge theme couldn't be found; a default gauge will be used instead.\n You can change this by opening the configuration of this module; do you want to do it now ?");\
		Icon *_pIcon = cairo_dock_get_dialogless_icon ();\
		gchar *_cQuestion = g_strdup_printf ("%s : %s", myApplet->pModule->pVisitCard->cModuleName, _cMessage);\
		cairo_dock_show_dialog_with_question (_cQuestion, _pIcon, CAIRO_CONTAINER (g_pMainDock), MY_APPLET_SHARE_DATA_DIR"/"MY_APPLET_ICON_FILE, (CairoDockActionOnAnswerFunc) cairo_dock_open_module_config_on_demand, myApplet, NULL);\
		g_free (_cQuestion); }\
	_cThemePath; })


  //////////
 // MENU //
//////////
/** Create and add a sub-menu to a given menu.
*@param cLabel name of the sub-menu.
*@param pMenu GtkWidget of the menu we will add the sub-menu to..
*@param cImage name of an image (can be a path or a GtkStock).
*@return the sub-menu, newly created and attached to the menu.
*/
#define CD_APPLET_ADD_SUB_MENU_WITH_IMAGE(cLabel, pMenu, cImage) \
	cairo_dock_create_sub_menu (cLabel, pMenu, cImage)

/** Create and add a sub-menu to a given menu.
*@param cLabel name of the sub-menu.
*@param pMenu GtkWidget of the menu we will add the sub-menu to..
*@return the sub-menu, newly created and attached to the menu.
*/
#define CD_APPLET_ADD_SUB_MENU(cLabel, pMenu) CD_APPLET_ADD_SUB_MENU_WITH_IMAGE(cLabel, pMenu, NULL)
#define CD_APPLET_CREATE_AND_ADD_SUB_MENU CD_APPLET_ADD_SUB_MENU

/** Create and add the default sub-menu of an applet to the main menu. This sub-menu is named according to the name of the applet, and is represented by the default icon of the applet.
*@return the sub-menu, newly created and attached to the main menu.
*/
#define CD_APPLET_CREATE_MY_SUB_MENU(...) CD_APPLET_ADD_SUB_MENU_WITH_IMAGE (myApplet->pModule->pVisitCard->cModuleName, CD_APPLET_MY_MENU, MY_APPLET_SHARE_DATA_DIR"/"MY_APPLET_ICON_FILE)

/** Create and add an entry to a menu.
*@param cLabel name of the entry.
*@param pFunction function called when the user selects this entry.
*@param pMenu menu to add the entry to.
*@param pData data passed as parameter of the callback.
*/
#define CD_APPLET_ADD_IN_MENU_WITH_DATA(cLabel, pFunction, pMenu, pData) do { \
	pMenuItem = gtk_menu_item_new_with_label (cLabel); \
	gtk_menu_shell_append  (GTK_MENU_SHELL (pMenu), pMenuItem); \
	g_signal_connect (G_OBJECT (pMenuItem), "activate", G_CALLBACK (pFunction), pData); } while (0)

/** Create and add an entry to a menu. 'myApplet' will be passed to the callback.
*@param cLabel name of the entry.
*@param pFunction function called when the user selects this entry.
*@param pMenu menu to add the entry to.
*/
#define CD_APPLET_ADD_IN_MENU(cLabel, pFunction, pMenu) CD_APPLET_ADD_IN_MENU_WITH_DATA(cLabel, pFunction, pMenu, myApplet)

/** Create and add an entry to a menu, with an icon.
*@param cLabel name of the entry.
*@param gtkStock name of a GTK icon or path to an image.
*@param pCallBack function called when the user selects this entry.
*@param pSubMenu menu to add the entry to.
*@param pData data passed as parameter of the callback.
*/
#define CD_APPLET_ADD_IN_MENU_WITH_STOCK_AND_DATA(cLabel, gtkStock, pCallBack, pSubMenu, pData) cairo_dock_add_in_menu_with_stock_and_data (cLabel, gtkStock, (GFunc)pCallBack, pSubMenu, pData)

/** Create and add an entry to a menu, with an icon. 'myApplet' will be passed to the callback.
*@param cLabel name of the entry.
*@param gtkStock name of a GTK icon or path to an image.
*@param pFunction function called when the user selects this entry.
*@param pMenu menu to add the entry to.
*/
#define CD_APPLET_ADD_IN_MENU_WITH_STOCK(cLabel, gtkStock, pFunction, pMenu) CD_APPLET_ADD_IN_MENU_WITH_STOCK_AND_DATA(cLabel, gtkStock, pFunction, pMenu, myApplet)

/** Create and add a separator to a menu.
 */
#define CD_APPLET_ADD_SEPARATOR_IN_MENU(pMenu) do { \
	pMenuItem = gtk_separator_menu_item_new (); \
	gtk_menu_shell_append(GTK_MENU_SHELL (pMenu), pMenuItem); } while (0)
#define CD_APPLET_ADD_SEPARATOR CD_APPLET_ADD_SEPARATOR_IN_MENU

/** Create and add an entry to a menu for the 'about' function.
*@param pMenu menu to add the entry to.
*/
#define CD_APPLET_ADD_ABOUT_IN_MENU(pMenu) CD_APPLET_ADD_IN_MENU_WITH_STOCK (_("About this applet"), GTK_STOCK_ABOUT, cairo_dock_pop_up_about_applet, pMenu)


  /////////////////////////
 // AVAILABLE VARIABLES //
/////////////////////////

//\______________________ init, config, reload.
/** Path of the applet's instance's conf file.
*/
#define CD_APPLET_MY_CONF_FILE myApplet->cConfFilePath
/** Key file of the applet instance, availale during the init, config, and reload.
*/
#define CD_APPLET_MY_KEY_FILE pKeyFile

//\______________________ reload.
/** TRUE if the conf file has changed before the reload.
*/
#define CD_APPLET_MY_CONFIG_CHANGED (pKeyFile != NULL)

/** TRUE if the container type has changed.
*/
#define CD_APPLET_MY_CONTAINER_TYPE_CHANGED (myApplet->pContainer == NULL || myApplet->pContainer->iType != pOldContainer->iType)

/** The previous Container.
*/
#define CD_APPLET_MY_OLD_CONTAINER pOldContainer


//\______________________ clic droit, clic milieu, clic gauche.
/** The clicked Icon.
*/
#define CD_APPLET_CLICKED_ICON pClickedIcon
/** The clicked Container.
*/
#define CD_APPLET_CLICKED_CONTAINER pClickedContainer

//\______________________ clic droit
/**  TRUE if the 'SHIFT' key was pressed during the click.
*/
#define CD_APPLET_SHIFT_CLICK (iButtonState & GDK_SHIFT_MASK)
/**  TRUE if the 'CTRL' key was pressed during the click.
*/
#define CD_APPLET_CTRL_CLICK (iButtonState & GDK_CONTROL_MASK)
/**  TRUE if the 'ALT' key was pressed during the click.
*/
#define CD_APPLET_ALT_CLICK (iButtonState & GDK_MOD1_MASK)

//\______________________ construction du menu.
/** Main menu of the applet.
*/
#define CD_APPLET_MY_MENU pAppletMenu

//\______________________ drop.
/** Data received after a drop occured (string).
*/
#define CD_APPLET_RECEIVED_DATA cReceivedData

//\______________________ scroll
#define CD_APPLET_SCROLL_DIRECTION iDirection
/** TRUE if the user scrolled up.
*/
#define CD_APPLET_SCROLL_UP (CD_APPLET_SCROLL_DIRECTION == GDK_SCROLL_UP)
/** TRUE if the user scrolled down.
*/
#define CD_APPLET_SCROLL_DOWN (CD_APPLET_SCROLL_DIRECTION == GDK_SCROLL_DOWN)


  /////////////////////
 // DRAWING SURFACE //
/////////////////////

/** Redraw the applet's icon (as soon as the main loop is available).
*/
#define CD_APPLET_REDRAW_MY_ICON \
	cairo_dock_redraw_icon (myIcon, myContainer)
/** Redraw the applet's container (as soon as the main loop is available).
*/
#define CAIRO_DOCK_REDRAW_MY_CONTAINER \
	cairo_dock_redraw_container (myContainer)
/** Reload the reflect of the applet's icon (do nothing in OpenGL mode).
*/
#define CD_APPLET_UPDATE_REFLECT_ON_MY_ICON \
	if (myContainer->bUseReflect) cairo_dock_add_reflection_to_icon (myDrawContext, myIcon, myContainer)

/** Load an image into a surface, at the same size as the applet's icon. If the image is given by its sole name, it is searched inside the current theme root folder. 
*@param cImagePath path or name of an image.
*@return the newly allocated surface.
*/
#define CD_APPLET_LOAD_SURFACE_FOR_MY_APPLET(cImagePath) \
	cairo_dock_create_surface_from_image_simple (cImagePath, myDrawContext, myIcon->fWidth * (myDock ? (1 + g_fAmplitude) / myDock->container.fRatio : 1), myIcon->fHeight* (myDock ? (1 + g_fAmplitude) / myDock->container.fRatio : 1))

/** Load a user image into a surface, at the same size as the applet's icon, or a default image taken in the installed folder of the applet if the first one is NULL. If the user image is given by its sole name, it is searched inside the current theme root folder.
*@param cUserImageName name or path of an user image.
*@param cDefaultLocalImageName default image
*@return the newly allocated surface.
*/
#define CD_APPLET_LOAD_SURFACE_FOR_MY_APPLET_WITH_DEFAULT(cUserImageName, cDefaultLocalImageName) \
	__extension__ ({\
	cairo_surface_t *pSurface; \
	if (cUserImageName != NULL) \
		pSurface = CD_APPLET_LOAD_SURFACE_FOR_MY_APPLET (cUserImageName); \
	else if (cDefaultLocalImageName != NULL)\
		pSurface = CD_APPLET_LOAD_SURFACE_FOR_MY_APPLET (MY_APPLET_SHARE_DATA_DIR"/"cDefaultLocalImageName); \
	else\
		pSurface = NULL;\
	pSurface; })

/** Apply a surface on the applet's icon, and redraw it.
*@param pSurface the surface to draw on your icon.
*/
#define CD_APPLET_SET_SURFACE_ON_MY_ICON(pSurface) do { \
	cairo_dock_set_icon_surface_with_reflect (myDrawContext, pSurface, myIcon, myContainer); \
	cairo_dock_redraw_icon (myIcon, myContainer); } while (0)

/** Apply a surface on the applet's icon, with a zoom factor and centered, and redraw it.
*@param pSurface the surface to draw on your icon.
*@param fScale zoom factor (at 1 the surface will fill all the icon).
*/
#define CD_APPLET_SET_SURFACE_ON_MY_ICON_WITH_ZOOM(pSurface, fScale) do { \
	cairo_dock_set_icon_surface_full (myDrawContext, pSurface, fScale, 1., myIcon, myContainer); \
	cairo_dock_add_reflection_to_icon (myDrawContext, myIcon, myContainer); \
	cairo_dock_redraw_icon (myIcon, myContainer); } while (0)

/** Apply a surface on the applet's icon with a transparency factor, and redraw it.
*@param pSurface the surface to draw on your icon.
*@param fAlpha transparency (in [0,1]).
*/
#define CD_APPLET_SET_SURFACE_ON_MY_ICON_WITH_ALPHA(pSurface, fAlpha) do { \
	cairo_dock_set_icon_surface_full (myDrawContext, pSurface, 1., fAlpha, myIcon, myContainer); \
	cairo_dock_add_reflection_to_icon (myDrawContext, myIcon, myContainer); \
	cairo_dock_redraw_icon (myIcon, myContainer); } while (0)

/** Apply a surface on the applet's icon with add a bar at the bottom, and redraw it. The bar is drawn at the bottom of the icon with a gradation from red to green and a given length.
*@param pSurface the surface to draw on your icon.
*@param fValue the value representing a percentage, in [-1,1]. If negative, the gradation is inverted, and the absolute value is used.
*/
#define CD_APPLET_SET_SURFACE_ON_MY_ICON_WITH_BAR(pSurface, fValue) do { \
	cairo_dock_set_icon_surface_with_bar (myDrawContext, pSurface, fValue, myIcon, myContainer); \
	cairo_dock_add_reflection_to_icon (myDrawContext, myIcon, myContainer); \
	cairo_dock_redraw_icon (myIcon, myContainer); } while (0)

/** Apply an image on the applet's icon. The image is resized at the same size as the icon. Does not trigger the icon refresh.
*@param cImagePath path to an image.
*/
#define CD_APPLET_SET_IMAGE_ON_MY_ICON(cImagePath)  \
	cairo_dock_set_image_on_icon (myDrawContext, cImagePath, myIcon, myContainer)

/** Apply an image, taken inside the installation folder of the applet, on the applet's icon. The image is resized at the same size as the icon. Does not trigger the icon refresh.
*@param cImageName name of an image 
*/
#define CD_APPLET_SET_LOCAL_IMAGE_ON_MY_ICON(cImageName) do { \
	gchar *_cImageFilePath = g_strconcat (MY_APPLET_SHARE_DATA_DIR, "/", cImageName, NULL); \
	CD_APPLET_SET_IMAGE_ON_MY_ICON (_cImageFilePath); \
	g_free (_cImageFilePath); } while (0)

/** Apply an image on the applet's icon at the same size as the applet's icon, or a default image taken in the installed folder of the applet if the first one is NULL. If the user image is given by its sole name, it is searched inside the current theme root folder.
*@param cUserImageName nom du fichier of l'image cote utilisateur.
*@param cDefaultLocalImageName image locale par defaut cote installation.
*/
#define CD_APPLET_SET_USER_IMAGE_ON_MY_ICON(cUserImageName, cDefaultLocalImageName) do { \
	gchar *cImagePath; \
	if (cUserImageName != NULL) \
		cImagePath = cairo_dock_generate_file_path (cUserImageName); \
	else \
		cImagePath = g_strdup_printf ("%s/%s", MY_APPLET_SHARE_DATA_DIR, cDefaultLocalImageName); \
	CD_APPLET_SET_IMAGE_ON_MY_ICON (cImagePath); \
	g_free (cImagePath); } while (0)

/** Apply the default icon on the applet's icon if there is no image yet.
*/
#define CD_APPLET_SET_DEFAULT_IMAGE_ON_MY_ICON_IF_NONE do { \
	if (myIcon->cFileName == NULL) { \
		CD_APPLET_SET_IMAGE_ON_MY_ICON (MY_APPLET_SHARE_DATA_DIR"/"MY_APPLET_ICON_FILE); } } while (0)


  ///////////
 // LABEL //
///////////
/** Set a new label on the applet's icon.
*@param cIconName the label.
*/
#define CD_APPLET_SET_NAME_FOR_MY_ICON(cIconName) \
	cairo_dock_set_icon_name (myDrawContext, cIconName, myIcon, myContainer)
/** Set a new label on the applet's icon.
*@param cIconNameFormat the label, in a 'printf'-like format.
*@param ... values to be written in the string.
*/
#define CD_APPLET_SET_NAME_FOR_MY_ICON_PRINTF(cIconNameFormat, ...) \
	cairo_dock_set_icon_name_full (myDrawContext, myIcon, myContainer, cIconNameFormat, ##__VA_ARGS__)


  ///////////////
 // QUICK-INFO//
///////////////
/** Set a quick-info on the applet's icon.
*@param cQuickInfo the quick-info. This is a small text (a few characters) that is superimposed on the icon.
*/
#define CD_APPLET_SET_QUICK_INFO_ON_MY_ICON(cQuickInfo) \
	cairo_dock_set_quick_info (myDrawContext, cQuickInfo, myIcon, myDock ? (1 + myIcons.fAmplitude) / 1 : 1)
/** Set a quick-info on the applet's icon.
*@param cQuickInfoFormat the label, in a 'printf'-like format.
*@param ... values to be written in the string.
*/
#define CD_APPLET_SET_QUICK_INFO_ON_MY_ICON_PRINTF(cQuickInfoFormat, ...) \
	cairo_dock_set_quick_info_full (myDrawContext, myIcon, myContainer, cQuickInfoFormat, ##__VA_ARGS__)

/** Write the time in hours-minutes as a quick-info on the applet's icon.
*@param iTimeInSeconds the time in seconds.
*/
#define CD_APPLET_SET_HOURS_MINUTES_AS_QUICK_INFO(iTimeInSeconds) \
	cairo_dock_set_hours_minutes_as_quick_info (myDrawContext, myIcon, myContainer, iTimeInSeconds)
/** Write the time in minutes-secondes as a quick-info on the applet's icon.
*@param iTimeInSeconds the time in seconds.
*/
#define CD_APPLET_SET_MINUTES_SECONDES_AS_QUICK_INFO(iTimeInSeconds) \
	cairo_dock_set_minutes_secondes_as_quick_info (myDrawContext, myIcon, myContainer, iTimeInSeconds)
/** Write a size in bytes as a quick-info on the applet's icon.
*@param iSizeInBytes the size in bytes, converted into a readable format.
*/
#define CD_APPLET_SET_SIZE_AS_QUICK_INFO(iSizeInBytes) \
	cairo_dock_set_size_as_quick_info (myDrawContext, myIcon, myContainer, iSizeInBytes)


  ///////////////
 // ANIMATION //
///////////////
/** Prevent the applet's icon to be animated when the mouse hovers it (call it once at init).
*/
#define CD_APPLET_SET_STATIC_ICON cairo_dock_set_icon_static (myIcon)

/** Launch an animation on the applet's icon.
*@param cAnimationName name of the animation.
*@param iAnimationLength number of rounds the animation should be played.
*/
#define CD_APPLET_ANIMATE_MY_ICON(cAnimationName, iAnimationLength) \
	cairo_dock_request_icon_animation (myIcon, myDock, cAnimationName, iAnimationLength)

/** Stop any animation on the applet's icon.
*/
#define CD_APPLET_STOP_ANIMATING_MY_ICON \
	cairo_dock_stop_icon_animation (myIcon)


/** Get the dimension allocated to the surface/texture of the applet's icon.
*@param iWidthPtr pointer to the width.
*@param iHeightPtr pointer to the height.
*/
#define CD_APPLET_GET_MY_ICON_EXTENT(iWidthPtr, iHeightPtr) cairo_dock_get_icon_extent (myIcon, myContainer, iWidthPtr, iHeightPtr)

/** Initiate an OpenGL drawing session on the applet's icon.
*/
#define CD_APPLET_START_DRAWING_MY_ICON cairo_dock_begin_draw_icon (myIcon, myContainer)

/** Initiate an OpenGL drawing session on the applet's icon, or quit the function if failed.
*@param ... value to return in case of failure.
*/
#define CD_APPLET_START_DRAWING_MY_ICON_OR_RETURN(...) \
	if (! cairo_dock_begin_draw_icon (myIcon, myContainer)) \
		return __VA_ARGS__

/** Terminate an OpenGL drawing session on the applet's icon. Does not trigger the icon's redraw.
*/
#define CD_APPLET_FINISH_DRAWING_MY_ICON cairo_dock_end_draw_icon (myIcon, myContainer)


/** Add a Data Renderer the applet's icon.
*@param pAttr the attributes of the Data Renderer. They allow you to define its properties.
*/
#define CD_APPLET_ADD_DATA_RENDERER_ON_MY_ICON(pAttr) cairo_dock_add_new_data_renderer_on_icon (myIcon, myContainer, myDrawContext, pAttr)

/** Reload the Data Renderer of the applet's icon. Pass NULL as the attributes to simply reload the current data renderer without changing any of its parameters. Previous values are kept.
*@param pAttr the attributes of the Data Renderer, or NULL to simply reload the Data Renderer as it it.
*/
#define CD_APPLET_RELOAD_MY_DATA_RENDERER(pAttr) cairo_dock_reload_data_renderer_on_icon (myIcon, myContainer, myDrawContext, pAttr)

/** Add new values to the Data Renderer of the applet's icon. Values are a table of 'double', having the same size as defined when the data renderer was created (1 by default). It also triggers the redraw of the icon.
*@param pValues the values, a table of double of the correct size.
*/
#define CD_APPLET_RENDER_NEW_DATA_ON_MY_ICON(pValues) cairo_dock_render_new_data_on_icon (myIcon, myContainer, myDrawContext, pValues)

/** Completely remove the Data Renderer of the applet's icon, including the values associated with.
*/
#define CD_APPLET_REMOVE_MY_DATA_RENDERER cairo_dock_remove_data_renderer_on_icon (myIcon)


/** Set the history size of the Data Renderer of the applet's icon to the maximum size, that is to say 1 value per pixel.
*/
#define CD_APPLET_SET_MY_DATA_RENDERER_HISTORY_TO_MAX cairo_dock_resize_data_renderer_history (myIcon, myIcon->fWidth)


#define CD_APPLET_GET_MY_ICON_DATA(pIcon) cairo_dock_get_icon_data (pIcon, myApplet)
#define CD_APPLET_GET_MY_CONTAINER_DATA(pContainer) cairo_dock_get_container_data (pContainer, myApplet)
#define CD_APPLET_GET_MY_DOCK_DATA(pDock) CD_APPLET_GET_MY_CONTAINER_DATA (CAIRO_CONTAINER (pDock))
#define CD_APPLET_GET_MY_DESKLET_DATA(pDesklet) CD_APPLET_GET_MY_CONTAINER_DATA (CAIRO_CONTAINER (pDesklet))

#define CD_APPLET_SET_MY_ICON_DATA(pIcon, pData) cairo_dock_set_icon_data (pIcon, myApplet, pData)
#define CD_APPLET_SET_MY_CONTAINER_DATA(pContainer, pData) cairo_dock_set_container_data (pContainer, myApplet, pData)
#define CD_APPLET_SET_MY_DOCK_DATA(pDock, pData) CD_APPLET_SET_MY_CONTAINER_DATA (CAIRO_CONTAINER (pDock), pData)
#define CD_APPLET_SET_MY_DESKLET_DATA(pDesklet, pData) CD_APPLET_SET_MY_CONTAINER_DATA (CAIRO_CONTAINER (pDesklet), pData)

#define CD_APPLET_LOAD_LOCAL_TEXTURE(cImageName) cairo_dock_create_texture_from_image (MY_APPLET_SHARE_DATA_DIR"/"cImageName)

#define CD_APPLET_LOAD_TEXTURE_WITH_DEFAULT(cUserImageName, cDefaultLocalImageName) \
	__extension__ ({\
	GLuint iTexture; \
	if (cUserImageName != NULL) \
		iTexture = cairo_dock_create_texture_from_image (cUserImageName); \
	else if (cDefaultLocalImageName != NULL)\
		iTexture = cairo_dock_create_texture_from_image (MY_APPLET_SHARE_DATA_DIR"/"cDefaultLocalImageName); \
	else\
		iTexture = 0;\
	iTexture; })

/** Say if the applet's container currently supports OpenGL.
*/
#define CD_APPLET_MY_CONTAINER_IS_OPENGL (g_bUseOpenGL && ((myDock && myDock->pRenderer->render_opengl) || (myDesklet && myDesklet->pRenderer && myDesklet->pRenderer->render_opengl)))

#define CD_APPLET_SET_TRANSITION_ON_MY_ICON(render_step_cairo, render_step_opengl, bFastPace, iDuration, bRemoveWhenFinished) \
	cairo_dock_set_transition_on_icon (myIcon, myContainer, myDrawContext,\
		(CairoDockTransitionRenderFunc) render_step_cairo,\
		(CairoDockTransitionGLRenderFunc) render_step_opengl,\
		bFastPace,\
		iDuration,\
		bRemoveWhenFinished,\
		myApplet)
#define CD_APPLET_GET_TRANSITION_FRACTION(...) \
	(cairo_dock_has_transition (myIcon) ? cairo_dock_get_transition_fraction (myIcon) : 1.)
#define CD_APPLET_REMOVE_TRANSITION_ON_MY_ICON \
	cairo_dock_remove_transition_on_icon (myIcon)


//\_________________________________ DESKLETS et SOUS-DOCKS

/** Set a renderer to the applet's desklet and create myDrawContext. Call it at the beginning of init and also reload, to take into account the desklet's resizing.
*@param cRendererName name of the renderer.
*@param pConfig configuration data for the renderer, or NULL.
*/
#define CD_APPLET_SET_DESKLET_RENDERER_WITH_DATA(cRendererName, pConfig) do { \
	cairo_dock_set_desklet_renderer_by_name (myDesklet, cRendererName, NULL, CAIRO_DOCK_LOAD_ICONS_FOR_DESKLET, (CairoDeskletRendererConfigPtr) pConfig); \
	myDrawContext = cairo_create (myIcon->pIconBuffer); } while (0)

/** Set a renderer to the applet's desklet and create myDrawContext. Call it at the beginning of init and also reload, to take into account the desklet's resizing.
*@param cRendererName name of the renderer.
*/
#define CD_APPLET_SET_DESKLET_RENDERER(cRendererName) CD_APPLET_SET_DESKLET_RENDERER_WITH_DATA (cRendererName, NULL)

/** Prevent the desklet from being rotated. Use it if your desklet has some static GtkWidget inside.
*/
#define CD_APPLET_SET_STATIC_DESKLET cairo_dock_set_static_desklet (myDesklet)

#define CD_APPLET_CREATE_MY_SUBDOCK(pIconsList, cRenderer) do { \
	if (myIcon->cName == NULL) { \
		CD_APPLET_SET_NAME_FOR_MY_ICON (myApplet->pModule->pVisitCard->cModuleName); } \
	if (cairo_dock_check_unique_subdock_name (myIcon)) { \
		CD_APPLET_SET_NAME_FOR_MY_ICON (myIcon->cName); } \
	myIcon->pSubDock = cairo_dock_create_subdock_from_scratch (pIconsList, myIcon->cName, myDock); \
	cairo_dock_set_renderer (myIcon->pSubDock, cRenderer); \
	cairo_dock_update_dock_size (myIcon->pSubDock); } while (0)

#define CD_APPLET_DESTROY_MY_SUBDOCK do { \
	cairo_dock_destroy_dock (myIcon->pSubDock, myIcon->cName, NULL, NULL); \
	myIcon->pSubDock = NULL; } while (0)

#define CD_APPLET_LOAD_ICONS_IN_MY_SUBDOCK(pIconsList) do { \
	if (myIcon->cName == NULL) { \
		CD_APPLET_SET_NAME_FOR_MY_ICON (myIcon->pModuleInstance->pModule->pVisitCard->cModuleName); }\
	else { \
		Icon *icon;\
		GList *ic;\
		for (ic = pIconsList; ic != NULL; ic = ic->next) {\
			icon = ic->data;\
			if (icon->cParentDockName == NULL)\
				icon->cParentDockName = g_strdup (myIcon->cName); } } \
	myIcon->pSubDock->icons = pIconsList; \
	myIcon->pSubDock->pFirstDrawnElement = pIconsList; \
	cairo_dock_load_buffers_in_one_dock (myIcon->pSubDock); \
	cairo_dock_update_dock_size (myIcon->pSubDock); } while (0)

/** Delete the list of icons of an applet (keep the subdock in dock mode).
*/
#define CD_APPLET_DELETE_MY_ICONS_LIST do {\
	if (myDesklet && myDesklet->icons != NULL) {\
		g_list_foreach (myDesklet->icons, (GFunc) cairo_dock_free_icon, NULL);\
		g_list_free (myDesklet->icons);\
		myDesklet->icons = NULL; }\
	if (myIcon->pSubDock != NULL) {\
		if (myDesklet) {\
			CD_APPLET_DESTROY_MY_SUBDOCK; }\
		else {\
			g_list_foreach (myIcon->pSubDock->icons, (GFunc) cairo_dock_free_icon, NULL);\
			g_list_free (myIcon->pSubDock->icons);\
			myIcon->pSubDock->icons = NULL;\
			myIcon->pSubDock->pFirstDrawnElement = NULL; } } } while (0)

/** Load a list of icons into an applet, with the given renderer for the sub-dock or the desklet).
*@param pIconList a list of icons. It will belong to the applet's container after that.
*@param cDockRendererName name of a renderer in case the applet is in dock mode.
*@param cDeskletRendererName name of a renderer in case the applet is in desklet mode.
*@param pDeskletRendererConfig possible configuration parameters for the desklet renderer.
*/
#define CD_APPLET_LOAD_MY_ICONS_LIST(pIconList, cDockRendererName, cDeskletRendererName, pDeskletRendererConfig) do {\
	if (myDock) {\
		if (myIcon->pSubDock == NULL) {\
			if (pIconList != NULL) {\
				CD_APPLET_CREATE_MY_SUBDOCK (pIconList, cDockRendererName); } }\
		else {\
			if (pIconList == NULL) {\
				CD_APPLET_DESTROY_MY_SUBDOCK; }\
			else {\
				cairo_dock_set_renderer (myIcon->pSubDock, cDockRendererName); \
				CD_APPLET_LOAD_ICONS_IN_MY_SUBDOCK (pIconList); } } }\
	else {\
		if (myIcon->pSubDock != NULL) {\
			CD_APPLET_DESTROY_MY_SUBDOCK; }\
		myDesklet->icons = pIconList;\
		CD_APPLET_SET_DESKLET_RENDERER_WITH_DATA (cDeskletRendererName, pDeskletRendererConfig);\
		CAIRO_DOCK_REDRAW_MY_CONTAINER;\
	} } while (0)

/** Gets the list of icons of your applet. It is either the icons of your sub-dock or of your desklet.
*/
#define CD_APPLET_MY_ICONS_LIST (myDock ? (myIcon->pSubDock ? myIcon->pSubDock->icons : NULL) : myDesklet->icons)
/** Gets the container of the icons of your applet. It is either your sub-dock or your desklet.
*/
#define CD_APPLET_MY_ICONS_LIST_CONTAINER (myDock && myIcon->pSubDock ? CAIRO_CONTAINER (myIcon->pSubDock) : myContainer)


//\_________________________________ TASKBAR
/** Lets your applet control the window of an external program, instead of the Taskbar.
 *\param cApplicationClass the class of the application you wish to control.
 *\param bStealTaskBarIcon TRUE to manage the application, FALSE to stop managing it.
*/
#define CD_APPLET_MANAGE_APPLICATION(cApplicationClass, bStealTaskBarIcon) do {\
	if (myIcon->cClass != NULL && (cairo_dock_strings_differ (myIcon->cClass, cApplicationClass) || ! bStealTaskBarIcon))\
		cairo_dock_deinhibate_class (cApplicationClass, myIcon);\
	if (myIcon->cClass == NULL && bStealTaskBarIcon)\
		cairo_dock_inhibate_class (cApplicationClass, myIcon); } while (0)

//\_________________________________ INTERNATIONNALISATION
/** Macro for gettext, similar to _() et N_(), but with the domain of the applet. Surround all your strings with this, so that 'xgettext' can find them and automatically include them in the translation files.
*/
#define D_(message) dgettext (MY_APPLET_GETTEXT_DOMAIN, message)
#define _D D_

G_END_DECLS
#endif
