# Copyright (C) 2009-2010 Aaron Bentley
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA


from cStringIO import StringIO
import difflib
import itertools

from bzrlib.branch import Branch, BranchReferenceFormat
from bzrlib.transport import get_transport
from bzrlib import (
    diff,
    errors,
    merge,
    shelf,
    shelf_ui,
    switch,
    trace,
    transform as _mod_transform,
    urlutils,
    workingtree,
)
from bzrlib.plugins.pipeline import is_pipe_alias

class BranchInTheWay(errors.BzrCommandError):
    _fmt = 'Branch in the way at %(url)s.'

    def __init__(self, branch):
        errors.BzrCommandError.__init__(self, url=branch.base)


class NoSuchPipe(errors.BzrCommandError):
    _fmt = """There is no pipe with nick "%(nick)s"."""

    def __init__(self, nick):
        errors.BzrCommandError.__init__(self, nick=nick)


class ChangesAlreadyStored(errors.BzrCommandError):

    _fmt = ('Cannot store uncommitted changes because this pipe already stores'
            ' uncommitted changes.')


class DuplicatePipe(errors.BzrCommandError):

    _fmt = 'There is already a pipe named "%(pipe_name)s."'

    def __init__(self, pipe_name):
        errors.BzrCommandError.__init__(self, pipe_name=pipe_name)


class DuplicatePipeLocation(errors.BzrCommandError):

    _fmt = 'Branch already in pipeline: "%(pipe_name)s."'

    def __init__(self, pipe_name):
        errors.BzrCommandError.__init__(self, pipe_name=pipe_name)


class DivergedPipes(errors.DivergedBranches):
    _fmt = 'Pipe "%(nick)s" has diverged.'

    def __init__(self, branch1, branch2):
        errors.DivergedBranches.__init__(self, branch1, branch2)
        self.nick = branch1.nick


class UnknownRemotePipe(errors.BzrCommandError):

    _fmt = 'Pipeline has no pipe named "%(nick)s".'

    def __init__(self, nick):
        errors.BzrCommandError.__init__(self, nick=nick)


class SwitchWithConflicts(errors.BzrCommandError):

    _fmt = 'Cannot switch while conflicts are present.'


class PipeStorage(object):
    """Store and retrieve pipe data."""

    def __init__(self, branch):
        self.branch = branch

    def get_next(self):
        """Return the location of the next pipe."""
        next = self.branch.get_config().get_user_option('next_pipe')
        if next == '':
            next = None
        return next

    def get_prev(self):
        """Return the location of the previous pipe."""
        prev = self.branch.get_config().get_user_option('prev_pipe')
        if prev == '':
            prev = None
        return prev

    def _sibling_pipe(self, url):
        """Return a sibling pipe for a URL."""
        if url is None:
            return None
        transports = [self.branch.bzrdir.root_transport]
        url = urlutils.join(self.branch.base, url)
        return Branch.open(url, possible_transports=transports)

    def get_next_pipe(self):
        """Return the next pipe."""
        return self._sibling_pipe(self.get_next())

    def get_prev_pipe(self):
        """Return the previous pipe."""
        return self._sibling_pipe(self.get_prev())

    def _pipe_url(self, url):
        """Determine the url to use for a pipe.

        Where possible, these urls are relative to the current pipe.
        If url is None, the result is ''.
        """
        if url is None:
            return ''
        return urlutils.relative_url(self.branch.base, url)

    def _set_next(self, next):
        """Set the next pipe location.

        None may be supplied, to set the location to None.
        """
        next = self._pipe_url(next)
        self.branch.get_config().set_user_option('next_pipe', next)

    def _set_prev(self, prev):
        """Set the previous pipe location.

        None may be supplied, to set the location to None.
        """
        prev = self._pipe_url(prev)
        self.branch.get_config().set_user_option('prev_pipe', prev)

    def _get_sibling_transport(self, name):
        t = self.branch.bzrdir.root_transport.clone('..').clone(name)
        return get_transport(t.base)

    @staticmethod
    def connect(previous, next):
        """Connect two pipes together.

        :param previous: The previous pipe, as a branch.
        :param next: The next pipe, as a branch.
        """
        if previous is None:
            prev_url = None
        else:
            prev_url = previous.base
        if next is None:
            next_url = None
        else:
            PipeStorage(next)._set_prev(prev_url)
            next_url = next.base
        if previous is not None:
            PipeStorage(previous)._set_next(next_url)

    def disconnect(self):
        """Remove this pipe from the pipeline."""
        next_pipe = self.get_next_pipe()
        prev_pipe = self.get_prev_pipe()
        self.connect(prev_pipe, next_pipe)
        self._set_prev(None)
        self._set_next(None)

    def _open_sibling_branch(self, name):
        t = self._get_sibling_transport(name)
        return Branch.open_from_transport(t)

    def _create_sibling_branch(self, name, source, revision_id=None):
        t = self._get_sibling_transport(name)
        if revision_id is None:
            revision_id = source.last_revision()
        ctrl = source.bzrdir.clone_on_transport(t, revision_id)
        return ctrl.open_branch()

    def insert_pipe(self, name, revision_id=None, before=False):
        """Insert a pipe in this pipeline by nick.

        The pipe is inserted after this pipe by default.
        :param name: The name of the pipe to insert
        :param revision_id: The revision to insert the pipe at.
        :param before: If True, insert the pipe before this pipe.
        :return: The new pipe.
        """
        try:
            self.find_pipe(name)
        except NoSuchPipe:
            pass
        else:
            raise DuplicatePipe(name)
        new_branch = self._create_sibling_branch(name, self.branch,
                                                 revision_id)
        self.insert_branch(new_branch, before)
        return new_branch

    def insert_branch(self, pipe, before=False):
        urls = [b.base for b in self.list_pipes()]
        if pipe.base in urls:
            raise DuplicatePipeLocation(pipe.nick)
        if not before:
            cur_pipe = self.branch
            next_pipe = self.get_next_pipe()
        else:
            next_pipe = self.branch
            cur_pipe = self.get_prev_pipe()
        if cur_pipe is not None:
            self.connect(cur_pipe, pipe)
        if next_pipe is not None:
            self.connect(pipe, next_pipe)

    def iter_pipes(self, prev=True):
        """Iterate from this pipe through the list of pipes.

        :param prev: If True, iterate into the previous pipes.  If false,
            iterate into the next pipes.
        """
        storage = self
        while True:
            if prev:
                next_branch = storage.get_prev_pipe()
            else:
                next_branch = storage.get_next_pipe()
            if next_branch is None:
                return
            yield next_branch
            storage = PipeStorage(next_branch)

    def list_pipes(self):
        """Return a list of the Branches for this pipeline."""
        pipes = list(self.iter_pipes(True))
        pipes.reverse()
        pipes.append(self.branch)
        pipes.extend(self.iter_pipes(False))
        return pipes

    def find_pipe(self, nick):
        """Find a pipe in the pipeline according to its nickname."""
        if self.branch.nick == nick:
            return self.branch
        for pipe in itertools.chain(self.iter_pipes(True),
                                    self.iter_pipes(False)):
            if pipe.nick == nick:
                return pipe
        else:
            raise NoSuchPipe(nick)

    def _get_transform(self):
        """Retrieve a TreeTransform in serialized form.

        :return: a file-like object.
        """
        try:
            return self.branch._transport.get('stored-transform')
        except errors.NoSuchFile:
            return None

    def _put_transform(self, input):
        """Store a TreeTransform in serialized form.

        :param input: a file-like object.
        """
        if input is None:
            try:
                self.branch._transport.delete('stored-transform')
            except errors.NoSuchFile:
                pass
        else:
            self.branch._transport.put_file('stored-transform', input)

    def has_stored_changes(self):
        """If true, the pipe has stored, uncommitted changes in it."""
        return self._get_transform() is not None


class ShelfReporter(shelf_ui.ShelfReporter):

    def __init__(self, nick): self.nick = nick

    def no_changes(self):
        pass

    def shelved_id(self, shelved_id):
        trace.note('Uncommitted changes stored in pipe "%s".', self.nick)

    def selected_changes(self, transform):
        pass


def guess_nick(location):
    return urlutils.unescape(urlutils.basename(location))


class PipeManager(object):

    def __init__(self, branch, checkout):
        self.storage = PipeStorage(branch)
        # A checkout connected to this pipeline.  May be None.
        self.checkout = checkout

    @classmethod
    def from_checkout(klass, checkout):
        return klass(checkout.branch, checkout)

    def _open_disconnected_pipe(self, name, local_pipes):
        remote = self.storage._open_sibling_branch(name)
        remote.lock_read()
        try:
            local = dict(local_pipes)[name]
            local.lock_read()
            try:
                heads = self.get_heads(local, remote, local.last_revision(),
                                       remote.last_revision())
            finally:
                local.unlock()
        finally:
            remote.unlock()
        if len(heads) > 1:
            raise BranchInTheWay(remote)
        remote_storage = PipeStorage(remote)
        if (remote_storage.get_next() is not None or
            remote_storage.get_prev() is not None):
            raise BranchInTheWay(remote)
        return remote

    def get_next_pipe(self):
        return self.storage.get_next_pipe()

    def get_prev_pipe(self):
        return self.storage.get_prev_pipe()

    def _get_terminal_pipe(self, first=True):
        pipe = self.storage.branch
        for pipe in self.storage.iter_pipes(prev=first):
            pass
        return pipe

    def get_first_pipe(self):
        return self._get_terminal_pipe()

    def get_last_pipe(self):
        return self._get_terminal_pipe(False)

    def list_pipes(self):
        return self.storage.list_pipes()

    def rename_pipe(self, new_name):
        root_transport = self.storage.branch.bzrdir.root_transport
        containing_transport = root_transport.clone('..')
        old_name = urlutils.split(root_transport.base)[1]
        containing_transport.rename(old_name, new_name)
        new_transport = containing_transport.clone(new_name)
        self.storage.branch = Branch.open_from_transport(new_transport)
        prev_pipe = self.get_prev_pipe()
        if prev_pipe is not None:
            PipeStorage.connect(prev_pipe, self.storage.branch)
        next_pipe = self.get_next_pipe()
        if next_pipe is not None:
            PipeStorage.connect(self.storage.branch, next_pipe)
        switch._set_branch_location(self.checkout.bzrdir, self.storage.branch)

    def sync_pipeline(self, location, remote_branch=None):
        """Synchronize this pipeline with another, mirror-style.

        Any missing remote pipes will be created, and the two pipelines
        will be updated to match each other by pushing or pulling pipes as
        appropriate.
        :location: The location of one of the remote pipes.
        """
        possible_transports = [self.storage.branch.bzrdir.root_transport]
        local_pipes = [(p.nick, p) for p in self.list_pipes()]
        local_nicks = [n for n, p in local_pipes]
        try:
            if remote_branch is None:
                b = Branch.open(location,
                                possible_transports=possible_transports)
            else:
                b = remote_branch
        except errors.NotBranchError:
            remote_nick = guess_nick(location)
            if remote_nick not in local_nicks:
                raise UnknownRemotePipe(remote_nick)
            trace.note('Creating new pipe at %s', location)
            t = get_transport(location,
                              possible_transports=possible_transports)
            local = self.storage.branch
            revision_id = local.last_revision()
            b = local.bzrdir.clone_on_transport(t, revision_id).open_branch()
            nick = b.nick
            remote_pipes = [(nick, b)]
            remote_manager = PipeManager(b, None)
            new_nicks = set([nick])
        else:
            remote_manager = PipeManager(b, None)
            remote_pipes = [(b.nick, b) for b in remote_manager.list_pipes()]
            new_nicks = set()
        remote_nicks = [n for n, p in remote_pipes]
        remote_map = dict(remote_pipes)
        desired_remote = self._merge_nick_order(local_nicks, remote_nicks)
        for nick, pipe in reversed(local_pipes):
            if nick in remote_nicks:
                continue
            trace.note('Creating new pipe "%s"', nick)
            try:
                remote_map[nick] = remote_manager._open_disconnected_pipe(
                    nick, local_pipes)
            except errors.NotBranchError:
                storage = remote_manager.storage
                remote_map[nick] = storage._create_sibling_branch(nick, pipe)
            pipe.set_push_location(remote_map[nick].base)
            new_nicks.add(nick)
        for prev, next in self._connections_to_create(desired_remote,
                                                      remote_nicks):
            pipe = remote_map[nick]
            self.storage.connect(remote_map[prev], remote_map[next])
        updatable = self._updatable_pipes(local_pipes, remote_map, new_nicks)
        self._push_pull(updatable)
        for nick, pipe in local_pipes:
            if pipe.get_push_location() is None:
                pipe.set_push_location(remote_map[nick].base)

    @staticmethod
    def list_connections(nicks):
        return zip(nicks[:-1], nicks[1:])

    @classmethod
    def _connections_to_create(cls, desired, existing):
        existing_map = dict(cls.list_connections(existing))
        connections = []
        for prev, next in cls.list_connections(desired):
            if existing_map.get(prev) != next:
                connections.append((prev, next))
        return connections

    @staticmethod
    def _updatable_pipes(local_pipes, remote_pipes, new_nicks):
        """Iterable of pairs of pipes that may need to be updated."""
        for nick, pipe in local_pipes:
            if nick in new_nicks:
                continue
            yield nick, pipe, remote_pipes[nick]

    def _push_pull(self, updatable_pipes):
        """Push or pull branches to update them, depending on what's newer."""
        for nick, pipe, remote_pipe in updatable_pipes:
            local_revision = pipe.last_revision()
            remote_revision = remote_pipe.last_revision()
            if remote_revision == local_revision:
                trace.note('%s is already up-to-date.', nick)
                continue
            pipe.lock_write()
            remote_pipe.lock_write()
            try:
                heads = self.get_heads(pipe, remote_pipe, local_revision,
                                       remote_revision)
            finally:
                remote_pipe.unlock()
                pipe.unlock()
            if len(heads) == 2:
                raise DivergedPipes(pipe, remote_pipe)
            head = heads.pop()
            if head == local_revision:
                trace.note('Pushing %s %s ', nick, remote_pipe.base)
                pipe.push(remote_pipe)
            else:
                trace.note('Pulling %s ', nick)
                if self.checkout.branch.base == pipe.base:
                    self.checkout.pull(remote_pipe)
                else:
                    pipe.pull(remote_pipe)

    @staticmethod
    def get_heads(pipe, remote_pipe, local_revision, remote_revision):
        graph = pipe.repository.get_graph(remote_pipe.repository)
        return graph.heads([local_revision, remote_revision])

    @staticmethod
    def _merge_nick_order(local_nick, remote_nick):
        """Merge two orderings of nicknames into one containing both.

        Order of the source is preserved within matching sections and
        non-matching sections.  Unmatched sequences are sorted according to
        their first element.
        """
        matcher = difflib.SequenceMatcher(None, local_nick, remote_nick)
        prev_i = 0
        prev_j = 0
        result = []
        for i, j, n in matcher.get_matching_blocks():
            local_unique = (local_nick[pos] for pos in range(prev_i, i))
            remote_unique = (remote_nick[pos] for pos in range(prev_j, j))
            if (i != prev_i and j != prev_j
                and remote_nick[prev_j] < local_nick[prev_i]):
                result.extend(remote_unique)
                result.extend(local_unique)
            else:
                result.extend(local_unique)
                result.extend(remote_unique)
            for nick_pos in range(i, i+n):
                result.append(local_nick[nick_pos])
            prev_i = i + n
            prev_j = j + n
        return result

    def shelve_changes(self, creator, message=None):
        # provided for interface-compatibility with Shelver
        if self.storage.has_stored_changes():
            raise ChangesAlreadyStored
        transform = StringIO()
        creator.write_shelf(transform, message)
        try:
            transform.seek(0)
            self.storage._put_transform(transform)
            creator.transform()
        finally:
            creator.finalize()
        return 0

    def has_stored_changes(self):
        return self.storage.has_stored_changes()

    def store_uncommitted(self, interactive=False):
        """Store uncommitted changes from this working tree in the pipe."""
        if not interactive:
            return self.store_all()
        self.checkout.lock_write()
        try:
            target_tree = self.checkout.basis_tree()
            nick = self.storage.branch.nick
            shelver = shelf_ui.Shelver(self.checkout, target_tree,
                                       auto=not interactive,
                                       auto_apply=True,
                                       manager=self,
                                       reporter=ShelfReporter(nick))
            shelver.run()
        finally:
            self.checkout.unlock()

    def store_all(self):
        self.checkout.lock_write()
        try:
            target_tree = self.checkout.basis_tree()
            shelf_creator = shelf.ShelfCreator(self.checkout, target_tree)
            try:
                change = None
                for change in shelf_creator.iter_shelvable():
                    shelf_creator.shelve_change(change)
                if change is None:
                    return
                self.shelve_changes(shelf_creator)
            finally:
                shelf_creator.finalize()
        finally:
            self.checkout.unlock()
        trace.note('Uncommitted changes stored in pipe "%s".',
                   self.storage.branch.nick)

    def make_uncommitted_merger(self, cleaner):
        metadata, base_tree, tt = self.get_transform_data()
        if tt is None:
            return None
        cleaner.add_cleanup(tt.finalize)
        return self.get_unshelve_merger(base_tree, tt, metadata)

    def get_transform_data(self):
        transform = self.storage._get_transform()
        if transform is None:
            return None, None, None
        records = shelf.Unshelver.iter_records(transform)
        metadata = shelf.Unshelver.parse_metadata(records)
        base_revision_id = metadata['revision_id']
        try:
            base_tree = self.checkout.revision_tree(base_revision_id)
        except errors.NoSuchRevisionInTree:
            repo = self.storage.branch.repository
            base_tree = repo.revision_tree(base_revision_id)
        tt = _mod_transform.TransformPreview(base_tree)
        tt.deserialize(records)
        return metadata, base_tree, tt

    def get_unshelve_merger(self, base_tree, tt, metadata):
        unshelver = shelf.Unshelver(self.checkout, base_tree, tt,
                                    metadata.get('message'))
        merger = unshelver.make_merger()
        merger.ignore_zero = True
        return merger

    def restore_uncommitted(self, delete=True):
        """Restore uncommitted changes from this pipe into the working tree."""
        self.checkout.lock_tree_write()
        try:
            metadata, base_tree, tt = self.get_transform_data()
            if tt is None:
                return None
            try:
                merger = self.get_unshelve_merger(base_tree, tt, metadata)
                merger.do_merge()
            finally:
                tt.finalize()
        finally:
            self.checkout.unlock()
        trace.note('Uncommitted changes restored from pipe "%s".',
                   self.storage.branch.nick)
        if delete:
            self.storage._put_transform(None)

    def _switch_to_tree(self, to_branch, to_tree):
        merge.Merge3Merger(self.checkout, self.checkout, self.checkout,
                           to_tree)
        self.checkout.set_last_revision(to_branch.last_revision())
        switch._set_branch_location(self.checkout.bzrdir, to_branch)

    def switch_to_pipe(self, pipe):
        if len(self.checkout.conflicts()) > 0:
            raise SwitchWithConflicts
        pipe = PipeManager(pipe, self.checkout)
        self.store_uncommitted()
        self.checkout.lock_write()
        try:
            metadata, base_tree, tt = pipe.get_transform_data()
            try:
                direct = (tt is not None and base_tree.get_revision_id() ==
                    pipe.storage.branch.last_revision())
                # do only one transform in the common case.
                if direct:
                    preview = tt.get_preview_tree()
                    switch_target = preview
                else:
                    switch_target = pipe.storage.branch.basis_tree()
                self._switch_to_tree(pipe.storage.branch, switch_target)
                if not direct and tt is not None:
                    merger = self.get_unshelve_merger(base_tree, tt, metadata)
                    merger.do_merge()
                if tt is not None:
                    pipe.storage._put_transform(None)
            finally:
                if tt is not None:
                    tt.finalize()
        finally:
            self.checkout.unlock()
        return pipe

    def _make_old_merger(self, old_branch, tree, tree_branch, merge_config):
        old_revision_id = old_branch.last_revision()
        merger = merge.Merger.from_revision_ids(None, tree,
            old_revision_id, other_branch=old_branch, tree_branch=tree_branch)
        merger.merge_type = merge_config.merge_type
        merger.ignore_zero = True
        merger.show_base = merge_config.show_base
        merger.reprocess = merge_config.reprocess
        return merger

    def _merge_commit_branch(self, branch, old_branch, merge_config):
        branch.lock_write()
        old_branch.lock_read()
        try:
            graph = branch.repository.get_graph(old_branch.repository)
            old_last_revision = old_branch.last_revision()
            if is_lefthand_ancestor(graph, branch.last_revision(),
                                    old_last_revision):
                branch.pull(old_branch, stop_revision=old_last_revision)
                return True
            merge_controller = self._make_old_merger(
                old_branch, branch.basis_tree(), branch, merge_config)
            merger = merge_controller.make_merger()
            def null_warning(message):
                pass
            old_trace_warning = trace.warning
            trace.warning = null_warning
            try:
                tt = merger.make_preview_transform()
            finally:
                trace.warning = old_trace_warning
            try:
                if len(merger.cooked_conflicts) > 0:
                    return False
                message = 'Merged %s into %s.' % (old_branch.nick,
                                                  branch.nick)
                parents = [old_branch.last_revision()]
                tt.commit(branch, message, parents)
                return True
            finally:
                tt.finalize()
        finally:
            old_branch.unlock()
            branch.unlock()

    def _merge_tree(self, tree, old_branch, merge_config):
        """Merge the old_branch into the tree and commit the result."""
        tree.lock_write()
        try:
            merger = self._make_old_merger(old_branch, tree, tree.branch,
                                           merge_config)
            conflicts = merger.do_merge()
            merger.set_pending()
        finally:
            tree.unlock()
        return conflicts

    @staticmethod
    def _skip_merge(old_branch, branch):
        old_branch.lock_read()
        branch.lock_read()
        try:
            old_revision_id = old_branch.last_revision()
            graph = old_branch.repository.get_graph(branch.repository)
            new_revision_id = branch.last_revision()
            return graph.is_ancestor(old_revision_id, new_revision_id)
        finally:
            branch.unlock()
            old_branch.unlock()

    def _mergeables(self, from_branch=None):
        """Iterable of new_branch, old_branch pairs to merge."""
        if from_branch is not None:
            if not self._skip_merge(from_branch, self.storage.branch):
                yield self.storage.branch, from_branch
        old_branch = self.storage.branch
        for branch in self.storage.iter_pipes(False):
            if not self._skip_merge(old_branch, branch):
                yield branch, old_branch
            old_branch = branch

    @staticmethod
    def _refresh_tree(tree):
        return workingtree.WorkingTree.open(tree.bzrdir.root_transport.base)

    def _merge_commit(self, tree, new_branch, old_branch, merge_config):
        if self._merge_commit_branch(new_branch, old_branch, merge_config):
            return []
        trace.note('Switching to %s for conflict resolution.' %
                   new_branch.nick)
        self.store_uncommitted()
        self._switch_to_tree(new_branch, new_branch.basis_tree())
        tree = self._refresh_tree(tree)
        return self._merge_tree(tree, old_branch, merge_config)

    def pipeline_merge(self, from_branch=None, merge_config=None):
        """Use the supplied tree to merge along the pipeline.

        Each pipe is merged into the next one, and committed, until conflicts
        are encountered or the end of the pipeline is reached.

        :param from_submit: If true, merge the pipe's submit branch into the
            pipe before merging the pipe into subsequent pipes.
        """
        checkout_branch_changed = False
        if merge_config is None:
            merge_config = MergeConfig()
        for new_branch, old_branch in self._mergeables(from_branch):
            if self._merge_commit(self.checkout, new_branch, old_branch,
                                  merge_config) != []:
                return False
            if new_branch.base == self.checkout.branch.base:
                checkout_branch_changed = True
        else:
            if checkout_branch_changed:
                self.checkout.update()
            return True

    def get_merge_branch(self):
        previous_loc = self.storage.branch.get_submit_branch()
        if previous_loc is None:
            previous_loc = self.storage.branch.get_parent()
        if previous_loc is None:
            return None
        return Branch.open(previous_loc)

    def _patch_previous_branch(self):
        prev_pipe = self.get_prev_pipe()
        if prev_pipe is not None:
            return prev_pipe
        else:
            return self.get_merge_branch()

    def get_prev_revision_id(self):
        prev = self.get_prev_pipe()
        if prev is not None:
            return prev.last_revision()
        merge_branch = self.get_merge_branch()
        if merge_branch is None:
            return None
        branch = self.storage.branch
        merge_branch.lock_read()
        try:
            branch.lock_read()
            try:
                graph = branch.repository.get_graph(merge_branch.repository)
                return graph.find_unique_lca(merge_branch.last_revision(),
                                             branch.last_revision())
            finally:
                branch.unlock()
        finally:
            merge_branch.unlock()

    def get_patch(self):
        """Get a patch representing this pipe.

        For the first pipe, the patch is against the submit branch, falling
        back to the parent branch.  If there is neither a submit or parent
        branch, None is returned.

        For subsequent pipes, the patch is against the previous pipe.
        """
        from_branch = self._patch_previous_branch()
        if from_branch is None:
            return None
        to_branch = self.storage.branch
        to_tree = to_branch.basis_tree()
        from_branch.lock_read()
        try:
            graph = from_branch.repository.get_graph(to_branch.repository)
            lca = graph.find_unique_lca(from_branch.last_revision(),
                                        to_branch.last_revision())
            from_tree = from_branch.repository.revision_tree(lca)
            output = StringIO()
            diff.DiffTree(from_tree, to_tree, output).show_diff(None)
            return output.getvalue()
        finally:
            from_branch.unlock()


def look_up(self, name, url):
    """Look up a branch location from a pipe location alias."""
    manager = PipeManager(Branch.open_containing('.')[0], None)
    return look_up_pipe(manager, name).base


def look_up_pipe(manager, name):
    if name.startswith('pipe:'):
        return manager.storage.find_pipe(name[len('pipe:'):])
    elif name == 'first':
        result = manager.get_first_pipe()
    elif name == 'last':
        result = manager.get_last_pipe()
    elif name == 'next':
        result = manager.get_next_pipe()
        if result is None:
            raise errors.DirectoryLookupFailure('No next pipe.')
    else:
        result = manager.get_prev_pipe()
        if result is None:
            raise errors.DirectoryLookupFailure('No previous pipe.')
    return result


def dwim_pipe(manager, pipe):
    """Convert a pipe name or alias into a branch."""
    if pipe is not None and pipe[0] == ':' and is_pipe_alias(pipe[1:]):
        return look_up_pipe(manager, pipe[1:])
    else:
        return manager.storage.find_pipe(pipe)


def tree_to_pipeline(tree):
    """Convert a colocated tree and branch for use with pipelines.

    This moves the branch of a tree into a new "pipes" subdirectory.
    It ensures that a shared repository is in use, creating one in "pipes",
    if necessary.  The previous location of the branch becomes a
    BranchReference to the new location.
    """
    if (tree.bzrdir.root_transport.base !=
        tree.branch.bzrdir.root_transport.base):
        raise errors.AlreadyLightweightCheckout(tree.bzrdir)
    pipes_transport = tree.bzrdir.root_transport.clone('.bzr/pipes')
    pipe_transport = pipes_transport.clone(tree.branch.nick)
    pipe_transport.create_prefix()
    if not tree.branch.repository.is_shared():
        format = tree.bzrdir.cloning_metadir()
        newdir = format.initialize_on_transport(pipes_transport)
        newdir.create_repository(shared=True)
    new_branch = tree.bzrdir.sprout(pipe_transport.base,
                       possible_transports=[pipe_transport],
                       create_tree_if_local=False).open_branch()
    tree.bzrdir.destroy_branch()
    BranchReferenceFormat().initialize(tree.bzrdir, target_branch=new_branch)
    return workingtree.WorkingTree.open(tree.basedir)


def is_lefthand_ancestor(graph, candidate_lefthand_ancestor,
                         candidate_descendant):
    current = candidate_descendant
    searcher = graph._make_breadth_first_searcher(
        [candidate_lefthand_ancestor])
    lefthand = set()
    while True:
        lefthand.add(current)
        if current == candidate_lefthand_ancestor:
            return True
        if len(lefthand.intersection(searcher.seen)) != 0:
            return False
        parents = graph.get_parent_map([current])
        if len(parents) == 0:
            return False
        current_parents = parents[current]
        if len(current_parents) == 0:
            return False
        current = current_parents[0]
        searcher.step()


class MergeConfig(object):
    """Configure merge behaviour."""

    def __init__(self, merge_type=None, show_base=False, reprocess=False):
        if merge_type is None:
            merge_type = merge.Merge3Merger
        self.merge_type = merge_type
        self.show_base = show_base
        self.reprocess = reprocess
