# Copyright (C) 2009 Canonical Ltd
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test the BranchMapper methods."""

from bzrlib import tests

from bzrlib.plugins.fastimport import (
    branch_mapper,
    )


class TestBranchMapper(tests.TestCase):

    def test_git_to_bzr(self):
        m = branch_mapper.BranchMapper()
        for git, bzr in {
            'refs/heads/master':                'trunk',
            'refs/heads/foo':                   'foo',
            'refs/tags/master':                 'trunk.tag',
            'refs/tags/foo':                    'foo.tag',
            'refs/remotes/origin/master':       'trunk.remote',
            'refs/remotes/origin/foo':          'foo.remote',
            }.items():
            self.assertEqual(m.git_to_bzr(git), bzr)

    def test_git_to_bzr_with_slashes(self):
        m = branch_mapper.BranchMapper()
        for git, bzr in {
            'refs/heads/master/slave':              'master/slave',
            'refs/heads/foo/bar':                   'foo/bar',
            'refs/tags/master/slave':               'master/slave.tag',
            'refs/tags/foo/bar':                    'foo/bar.tag',
            'refs/remotes/origin/master/slave':     'master/slave.remote',
            'refs/remotes/origin/foo/bar':          'foo/bar.remote',
            }.items():
            self.assertEqual(m.git_to_bzr(git), bzr)

    def test_git_to_bzr_for_trunk(self):
        # As 'master' in git is mapped to trunk in bzr, we need to handle
        # 'trunk' in git in a sensible way.
        m = branch_mapper.BranchMapper()
        for git, bzr in {
            'refs/heads/trunk':             'git-trunk',
            'refs/tags/trunk':              'git-trunk.tag',
            'refs/remotes/origin/trunk':    'git-trunk.remote',
            'refs/heads/git-trunk':         'git-git-trunk',
            'refs/tags/git-trunk':          'git-git-trunk.tag',
            'refs/remotes/origin/git-trunk':'git-git-trunk.remote',
            }.items():
            self.assertEqual(m.git_to_bzr(git), bzr)
