#include "config.h"
#if HAVE_MAC_PARTITION

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <asm/types.h>
#include "byteorder.h"

#include "fdisk.h"


#define MAC_PARTITION_MAGIC	0x504d

/* type field value for A/UX or other Unix partitions */
#define APPLE_AUX_TYPE	"Apple_UNIX_SVR2"

struct mac_partition {
	__u16	signature;	/* expected to be MAC_PARTITION_MAGIC */
	__u16	res1;
	__u32	map_count;	/* # blocks in partition map */
	__u32	start_block;	/* absolute starting block # of partition */
	__u32	block_count;	/* number of blocks in partition */
	char	name[32];	/* partition name */
	char	type[32];	/* string type description */
	__u32	data_start;	/* rel block # of first data block */
	__u32	data_count;	/* number of data blocks */
	__u32	status;		/* partition status bits */
	__u32	boot_start;
	__u32	boot_size;
	__u32	boot_load;
	__u32	boot_load2;
	__u32	boot_entry;
	__u32	boot_entry2;
	__u32	boot_cksum;
	char	processor[16];	/* identifies ISA of boot */
	/* there is more stuff after this that we don't need */
};

#define MAC_STATUS_BOOTABLE	8	/* partition is bootable */

#define MAC_DRIVER_MAGIC	0x4552

/* Driver descriptor structure, in block 0 */
struct mac_driver_desc {
	__u16	signature;	/* expected to be MAC_DRIVER_MAGIC */
	__u16	block_size;
	__u32	block_count;
    /* ... more stuff */
};

static struct str2ptype {
    char *str;
    unsigned int ptype;
} str2ptype[] = {
    { "Apple_partition_map",	PTYPE_MAC_PMAP },
    { "Apple_Driver",		PTYPE_MAC_DRIVER },
    { "Apple_Driver43",		PTYPE_MAC_DRIVER43 },
    { "Apple_Driver_ATA",       PTYPE_MAC_ATADRIVER },
    { "Apple_FWDriver",         PTYPE_MAC_FWDRIVER },
    { "Apple_Driver_IOKit",     PTYPE_MAC_IODRIVER },
    { "Apple_Patches",          PTYPE_MAC_PATCH },
    { "Apple_Boot",             PTYPE_MAC_OSXBOOT },
    { "Apple_Loader",           PTYPE_MAC_OSXLOADER },
    { "Apple_UFS",              PTYPE_MAC_UFS },
    { "Apple_HFS",		PTYPE_MAC_HFS },
    { "Apple_MFS",		PTYPE_MAC_MFS },
    { "Apple_Scratch",		PTYPE_MAC_SCRATCH },
    { "Apple_PRODOS",		PTYPE_MAC_PRODOS },
    { "Apple_Free",		PTYPE_MAC_FREE },
    { "Apple_Bootstrap",	PTYPE_MAC_BOOT },
};

static unsigned int mac_ptype(char *type_str, char *name_str)
{
    int i;
    struct str2ptype *p;

    if (strcmp( type_str, "Apple_UNIX_SVR2" ) == 0) {
	/* for this type, need to differentiate by name */
	if (strcasecmp( name_str, "swap" ) == 0)
	    return PTYPE_MAC_SWAP;
	else if (strncmp( name_str, "A/UX", 4 ) == 0)
	    return PTYPE_MAC_AUX;
	else if (strcasecmp( name_str, "msdos" ) == 0)
	    return PTYPE_MAC_MSDOS;
	else if (strcasecmp( name_str, "minix" ) == 0)
	    return PTYPE_MAC_MINIX;
	else if (strcasecmp( name_str, "affs" ) == 0)
	    return PTYPE_MAC_AFFS;
	else
	    /* in default case, assume "Linux native" */
	    return PTYPE_MAC_EXT2;
    }
    
    for(i = 0,p = str2ptype; i<sizeof(str2ptype)/sizeof(*str2ptype); ++i,++p) {
	if (strcmp(p->str, type_str) == 0)
	    return p->ptype;
    }
    return PTYPE_UNKNOWN;
}

int parse_mac_partition(char *device, int fd)
{
    int blk, blocks_in_map;
    int dev_pos, pos, minor = 1;
    unsigned secsize;
    struct mac_partition *part;
    struct mac_driver_desc *md;
    unsigned char data[512];

    dev_pos = 0;
    /* Get 0th block and look at the first partition map entry. */
    if (!sread( fd, 0, data ))
	return -1;
    md = (struct mac_driver_desc *) data;
    if (__be16_to_cpu(md->signature) != MAC_DRIVER_MAGIC)
	return 0;
    secsize = __be16_to_cpu(md->block_size);
    if (secsize >= 512) {
	dev_pos = secsize;
	if (!sread(fd, secsize/512, data))
	    return -1;
    }
    part = (struct mac_partition *) (data + secsize - dev_pos);
    if (__be16_to_cpu(part->signature) != MAC_PARTITION_MAGIC)
	return 0;		/* not a MacOS disk */
    blocks_in_map = __be32_to_cpu(part->map_count);
    for (blk = 1; blk <= blocks_in_map; ++blk) {
	pos = blk * secsize;
	if (pos >= dev_pos + 512) {
	    dev_pos = pos;
	    if (!sread(fd, pos/512, data))
		return -1;
	}
	part = (struct mac_partition *) (data + pos - dev_pos);
	if (__be16_to_cpu(part->signature) != MAC_PARTITION_MAGIC)
	    break;
	blocks_in_map = __be32_to_cpu(part->map_count);
	fdisk_add_partition(device, minor,
			    mac_ptype(part->type, part->name),
			    __be32_to_cpu(part->block_count)*secsize/1024);
	
	++minor;
    }
    return 1;
}

#endif /* HAVE_MAC_PARTITION */
