#include "config.h"
#if HAVE_ACORN_PARTITION

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/fs.h>
#include <asm/types.h>
#include "byteorder.h"

#include "fdisk.h"


/* Acorn partitioning recognition by Peter Naulls.  Based upon kernel versions by
   Russell King */

#define PARTITION_RISCIX_MFM	1
#define PARTITION_RISCIX_SCSI	2
#define PARTITION_LINUX		9

#define LINUX_NATIVE_MAGIC 0xdeafa1de
#define LINUX_SWAP_MAGIC   0xdeafab1e


/*
 * Disc Record at disc address 0xc00
 */
struct adfs_discrecord {
    __u8  log2secsize;
    __u8  secspertrack;
    __u8  heads;
    __u8  density;
    __u8  idlen;
    __u8  log2bpmb;
    __u8  skew;
    __u8  bootoption;
    __u8  lowsector;
    __u8  nzones;
    __u16 zone_spare;
    __u32 root;
    __u32 disc_size;
    __u16 disc_id;
    __u8  disc_name[10];
    __u32 disc_type;
    __u32 disc_size_high;
    __u8  log2sharesize:4;
    __u8  unused40:4;
    __u8  big_flag:1;
    __u8  unused41:1;
    __u8  nzones_high;
    __u32 format_version;
    __u32 root_size;
    __u8  unused52[60 - 52];
};


struct linux_part {
    __u32 magic;
    __u32 start_sect;
    __u32 nr_sects;
};


inline int adfs_checkbblk(unsigned char *ptr)
{
  unsigned int result = 0;
  unsigned char *p = ptr + 511;

  do {
    result = (result & 0xff) + (result >> 8);
    result = result + *--p;
  } while (p != ptr);

  return (result & 0xff) != ptr[511];
}

static int linux_partition(char *device, int fd, unsigned long first_sect, unsigned long nr_sects)
{
  struct linux_part *linuxp;
  unsigned char data[512];
  unsigned int partnum = 3;

  if (!sread(fd, first_sect, data))
    return -1;

  linuxp = (struct linux_part *)data;

  while (linuxp->magic == __cpu_to_le32(LINUX_NATIVE_MAGIC) ||
      linuxp->magic == __cpu_to_le32(LINUX_SWAP_MAGIC)) {

    fdisk_add_partition(device, partnum,
     linuxp->magic == __cpu_to_le32(LINUX_NATIVE_MAGIC) ?  PTYPE_ACORN_EXT2 : PTYPE_ACORN_SWAP,
     linuxp->nr_sects / 2);

    linuxp++;
    partnum++;
  }

  return 1;
}



int parse_acorn_partition(char *device, int fd)
{
    unsigned long start_sect, nr_sects, sectscyl, heads;
    struct adfs_discrecord *dr;
    unsigned char data[512];

    if (!sread( fd, 0xc00 / 512, data ))
       return -1;

    if (adfs_checkbblk(data))
        return 0;

    dr = (struct adfs_discrecord *) (data + 0x1c0);

    if (dr->disc_size == 0 && dr->disc_size_high == 0)
      return 0;

    nr_sects = (__le32_to_cpu(dr->disc_size_high) << 23) |
               (__le32_to_cpu(dr->disc_size) >> 9);

    if (nr_sects)
    {
      unsigned int size;
      fdisk_add_partition(device, 1, PTYPE_ACORN_ADFS, nr_sects / 2);

      heads = dr->heads + ((dr->lowsector >> 6) & 1);
      sectscyl = dr->secspertrack * heads;

      /*
       * Work out start of non-adfs partition.
       */
      start_sect = ((data[0x1fe] << 8) + data[0x1fd]) * sectscyl;

      /* Get size in Kb */
      if (ioctl(fd, BLKGETSIZE, &size))
        return 1;

      nr_sects = size / 2 - start_sect;

      if (start_sect)
      {
        switch (data[0x1fc] & 15)
        {
          case PARTITION_LINUX:
            return linux_partition(device, fd, start_sect, nr_sects);
            break;
        }
      }

      return 1;
    }

    return 0;
}

#endif /* HAVE_ACORN_PARTITION */
