/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-2001  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *  Copyright (C) 1999-2001  Haavard Kvaalen
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "beep.h"
#include "beep/skin.h"
#include "skin.h"
#include "na.xpm"
#include "beep_mini.xpm"

GtkWidget *skinwin = NULL, *skinwin_close, *scrolled_win;
GList *skinlist = NULL;
//static pthread_mutex_t skinwin_mutex = PTHREAD_MUTEX_INITIALIZER;

void create_skin_window(void);

GdkPixbuf *skin_get_thumbnail(const gchar * path)
{

    GdkPixbuf *scaled = NULL;
    GdkPixbuf *buf = NULL;

    const gchar *dec_path;
    gchar *f_path = NULL;
    gboolean is_archive = FALSE;


    if (file_is_archive(path)) {
	dec_path = decompress_archive(path);
	if (dec_path == NULL) {
	    buf = gdk_pixbuf_new_from_xpm_data((const char **) na_xpm);
	    return buf;
	}
	is_archive = TRUE;
    } else {
	dec_path = path;
    }

    f_path = find_file_recursively(dec_path, "main.bmp");
    buf = gdk_pixbuf_new_from_file(f_path, NULL);

    if (is_archive) {
	del_directory(dec_path);
    }

    if (buf != NULL) {
	scaled = gdk_pixbuf_scale_simple(buf,
					 137, 58, GDK_INTERP_BILINEAR);
	g_object_unref(buf);
	return scaled;
    } else {
	buf = gdk_pixbuf_new_from_xpm_data((const char **) na_xpm);
	return buf;
    }

    return NULL;
}

gint skinwin_delete_event(GtkWidget * widget, GdkEvent * event,
			  gpointer data)
{
    gtk_widget_hide(skinwin);
    return (TRUE);
}


static gboolean skinwin_keypress_cb(GtkWidget * widget,
				    GdkEventKey * event, gpointer userdata)
{
    if (!event)
	return FALSE;

    switch (event->keyval) {
    case GDK_Escape:
	gtk_widget_hide(skinwin);
	break;
    default:
	return FALSE;
    }

    return TRUE;
}


void change_skin_event(GtkTreeSelection * selection, gpointer data)
{

    char *name;
    GtkTreeIter iter;
    GtkTreeView *treeview = GTK_TREE_VIEW(data);
    GtkTreeModel *model;
    GtkTreeSelection *l_selection;
    guint max;
    guint loop;
    gchar *comp = NULL;


    /* FIXME:
     *  This is all alittle of bad mojo here, the gtk-doc says about the "changed" signal for GtkTreeSelection:
     *  "[...] [This] signal is mostly a hint. It may only be emitted once when a range of rows are selected, 
     *  and it may occasionally be emitted when nothing has happened."
     * Still it seems to be the most reasonable to connect. Ahwell...
     */
    l_selection = gtk_tree_view_get_selection(treeview);
    if (!gtk_tree_selection_get_selected(GTK_TREE_SELECTION(l_selection),
					 &model, &iter))
	return;
    gtk_tree_model_get(model, &iter, 1, &name, -1);

    max = g_list_length(skinlist);

    for (loop = 0; loop < max; loop++) {
	comp =
	    ((struct SkinNode *) g_list_nth(skinlist, loop)->data)->path;
	if (g_strrstr(comp, name) != NULL)
	    break;
    }
    if (comp)
	load_skin(comp);
}

static void enable_random_skin_event(GtkWidget * widget, gpointer data)
{
    cfg.random_skin_on_play =
	gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
}


static void add_skin(gchar * skin)
{
    struct SkinNode *node =
	(struct SkinNode *) g_malloc(sizeof(struct SkinNode));
    gchar *tmp = NULL;

    node->path = skin;

    if (!tmp)
	tmp = skin;
    node->name = (char *) g_malloc(strlen(tmp + 1) + 1);
    strcpy(node->name, tmp + 1);
    if (tmp) {
	tmp = strrchr(node->name, '.');
    }
/*	g_print ("nn: \"%s\" np:\"%s\"\n", node->name, node->path);*/
    skinlist = g_list_prepend(skinlist, node);
}

static void scan_skindir(char *path)
{
    GDir *dir2;
    GError *dir_open_gerror = NULL;
    gchar *file;
    const gchar *dir_ent;

    dir2 = g_dir_open(path, 0, &dir_open_gerror);

    if (!dir2) {
	g_warning("error:\n%s", dir_open_gerror->message);
	g_error_free(dir_open_gerror);
	return;
    }

    while ((dir_ent = g_dir_read_name(dir2)) != NULL) {
	file = g_build_filename(path, dir_ent, NULL);
	if (g_file_test(file, G_FILE_TEST_EXISTS)) {
	    if (g_file_test(file, G_FILE_TEST_IS_DIR)) {
		add_skin(file);
		file = NULL;
	    } else if (g_file_test(file, G_FILE_TEST_IS_REGULAR)) {
		if (file_is_archive(file)) {
		    add_skin(file);
		    file = NULL;	/* the g_free of file will simply */
		    /* return if the param is NULL */
		}
	    }
	}
	g_free(file);

    }
    g_dir_close(dir2);
    return;
}

static gint skinlist_compare_func(gconstpointer a, gconstpointer b)
{
    return strcasecmp(((struct SkinNode *) a)->name,
		      ((struct SkinNode *) b)->name);
}

static void skin_free_func(gpointer data, gpointer user_data)
{
    g_free(((struct SkinNode *) data)->name);
    g_free(((struct SkinNode *) data)->path);
    g_free(data);
}

void update_skins(GtkWidget * w, gpointer data)
{

    GList *entry;
    int i;
    char *str, *skinsdir;
    gchar **list;

    gtk_widget_set_sensitive(scrolled_win, FALSE);
    GtkTreeView *treeview = GTK_TREE_VIEW(data);
    GtkTreeSelection *selection = NULL;
    GtkListStore *store;

    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
    store = GTK_LIST_STORE(gtk_tree_view_get_model(treeview));

    gtk_list_store_clear(store);

    if (skinlist) {
	g_list_foreach(skinlist, skin_free_func, NULL);
	g_list_free(skinlist);
    }
    skinlist = NULL;

    str = g_build_filename(g_get_home_dir(), BMP_RCPATH, "Skins", NULL);	//g_strconcat(g_get_home_dir(), "/", BMP_RCPATH,"/Skins", NULL);
    scan_skindir(str);
    g_free(str);
    str = g_strconcat(DATA_DIR, "/Skins", NULL);
    scan_skindir(str);
    g_free(str);
    skinlist = g_list_sort(skinlist, skinlist_compare_func);
    skinsdir = getenv("SKINSDIR");
    if (skinsdir) {
	list = g_strsplit(skinsdir, ":", 0);
	i = 0;
	while (list[i])
	    scan_skindir(list[i++]);
    }

    gtk_widget_show_all(skinwin);

    for (i = 0; i < g_list_length(skinlist); i++) {
	GdkPixbuf *buf;
	GtkTreeIter iter;

	gchar *name, *tmp;

	entry = g_list_nth(skinlist, i);
	name =
	    g_strconcat(G_DIR_SEPARATOR_S,
			((struct SkinNode *) entry->data)->name, NULL);
	buf = skin_get_thumbnail(name);
	tmp = g_path_get_basename(name);
	g_free(name);

	if (tmp) {
	    if (file_is_archive(tmp)) {
		gchar *tmptmp;
		tmptmp = archive_basename(tmp);
		g_free(tmp);
		tmp = tmptmp;

	    }
	}


	while (gtk_events_pending())
	    gtk_main_iteration();

	/* Just in case we get deleted */
	if (GTK_WIDGET_VISIBLE(skinwin)) {
	    gtk_list_store_append(GTK_LIST_STORE(store), &iter);
	    gtk_list_store_set(GTK_LIST_STORE(store), &iter,
			       0, buf, 1, tmp, -1);
	    if (skin->path)
		if (g_strstr_len(skin->path, strlen(skin->path), tmp) !=
		    NULL) {
		    /* If we don't block, the skin could be reloaded unneccessarily again */
		    g_signal_handlers_disconnect_by_func(selection,
							 change_skin_event,
							 NULL);

		    gtk_tree_selection_select_iter(selection, &iter);

		    g_signal_connect(G_OBJECT(selection), "changed",
				     G_CALLBACK(change_skin_event),
				     treeview);
		}

	    g_object_unref(buf);
	}
	g_free(tmp);

    }
    gtk_widget_set_sensitive(scrolled_win, TRUE);
}
static void skinwin_set_icon(GtkWidget * win)
{
    static GdkPixmap *icon;
    static GdkBitmap *mask;
    GdkAtom icon_atom;
    glong data[2];

    if (!icon)
	icon =
	    gdk_pixmap_create_from_xpm_d(win->window, &mask,
					 &win->style->bg[GTK_STATE_NORMAL],
					 beep_mini_xpm);
    data[0] = GDK_WINDOW_XWINDOW(icon);
    data[1] = GDK_WINDOW_XWINDOW(mask);

    icon_atom = gdk_atom_intern((const gchar *) "KWM_WIN_ICON", FALSE);
    gdk_property_change(win->window, icon_atom, icon_atom, 32,
			GDK_PROP_MODE_REPLACE, (guchar *) data, 2);
    gdk_window_set_icon(win->window, NULL, icon, mask);
    gdk_window_set_icon_name(win->window, "bmp: skin browser");
    gdk_window_set_title(win->window, "bmp: skin browser");
    gdk_window_set_group(win->window, mainwin->window);
}
void create_skin_window(void)
{
    GtkWidget *vbox, *hbox, *main_hbox, *separator, *checkbox;
    char *none;

    GtkTreeViewColumn *column;
    GtkCellRenderer *renderer;
    GtkWidget *treeview;
    GtkListStore *store;
    GtkTreeModel *model;

    GObject *selection;

    skinwin = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_widget_set_usize(skinwin, 500, 380);
    gtk_window_set_position(GTK_WINDOW(skinwin), GTK_WIN_POS_CENTER);
    gtk_window_set_title(GTK_WINDOW(skinwin), _("Skin Browser"));
    gtk_window_set_transient_for(GTK_WINDOW(skinwin), GTK_WINDOW(mainwin));

    g_signal_connect(G_OBJECT(skinwin), "delete_event",
		     G_CALLBACK(skinwin_delete_event), NULL);
    gtk_container_border_width(GTK_CONTAINER(skinwin), 10);

    vbox = gtk_vbox_new(FALSE, 5);
    gtk_container_add(GTK_CONTAINER(skinwin), vbox);

    scrolled_win = gtk_scrolled_window_new(NULL, NULL);

    gtk_container_border_width(GTK_CONTAINER(scrolled_win), 5);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_win),
				   GTK_POLICY_AUTOMATIC,
				   GTK_POLICY_ALWAYS);
    gtk_box_pack_start(GTK_BOX(vbox), scrolled_win, TRUE, TRUE, 0);


    separator = gtk_hseparator_new();
    gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, TRUE, 0);

    main_hbox = gtk_hbox_new(FALSE, 5);
    gtk_box_set_spacing(GTK_BOX(main_hbox), 5);
    gtk_box_pack_start(GTK_BOX(vbox), main_hbox, FALSE, FALSE, 0);

    checkbox =
	gtk_check_button_new_with_label(_("Select random skin on play"));
    gtk_box_pack_start(GTK_BOX(main_hbox), checkbox, FALSE, FALSE, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox),
				 cfg.random_skin_on_play);
    g_signal_connect(G_OBJECT(checkbox), "toggled",
		     G_CALLBACK(enable_random_skin_event), NULL);

    hbox = gtk_hbutton_box_new();
    gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
    gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 5);
    gtk_box_pack_start(GTK_BOX(main_hbox), hbox, TRUE, TRUE, 0);
    skinwin_close = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
    GTK_WIDGET_SET_FLAGS(skinwin_close, GTK_CAN_DEFAULT);
    g_signal_connect(G_OBJECT(skinwin_close), "clicked",
		     G_CALLBACK(skinwin_delete_event), NULL);
    gtk_box_pack_start(GTK_BOX(hbox), skinwin_close, FALSE, FALSE, 0);
    gtk_widget_grab_default(skinwin_close);
    gtk_scrolled_window_set_shadow_type
	(GTK_SCROLLED_WINDOW(scrolled_win), GTK_SHADOW_ETCHED_IN);
    gtk_widget_show_all(scrolled_win);

    none = _("(none)");

    store = gtk_list_store_new(2, GDK_TYPE_PIXBUF, G_TYPE_STRING);
    model = GTK_TREE_MODEL(store);
    treeview = gtk_tree_view_new_with_model(model);
    g_object_unref(G_OBJECT(model));

    gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(treeview), TRUE);
    renderer = gtk_cell_renderer_pixbuf_new();
    column = gtk_tree_view_column_new();
    gtk_tree_view_column_set_sizing(GTK_TREE_VIEW_COLUMN(column),
				    GTK_TREE_VIEW_COLUMN_AUTOSIZE);

    gtk_tree_view_column_pack_start(GTK_TREE_VIEW_COLUMN(column), renderer,
				    FALSE);
    gtk_tree_view_column_set_attributes(GTK_TREE_VIEW_COLUMN(column),
					renderer, "pixbuf", 0, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview),
				GTK_TREE_VIEW_COLUMN(column));
    renderer = gtk_cell_renderer_text_new();

/*	g_object_set(renderer,"weight",PANGO_WEIGHT_BOLD,NULL);
	g_object_set(renderer,"weight-set",TRUE,NULL);
*/

    gtk_tree_view_column_pack_start(column, renderer, TRUE);
    gtk_tree_view_column_set_spacing(column, 16);
    gtk_tree_view_column_set_attributes(column, renderer, "text", 1, NULL);

    selection =
	G_OBJECT(gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview)));
    g_signal_connect(G_OBJECT(selection), "changed",
		     G_CALLBACK(change_skin_event), treeview);
    g_signal_connect(G_OBJECT(skinwin), "key_press_event",
		     G_CALLBACK(skinwin_keypress_cb), NULL);
    g_signal_connect(G_OBJECT(skinwin), "show", G_CALLBACK(update_skins),
		     treeview);
    g_object_set(G_OBJECT(treeview), "headers-visible", FALSE, NULL);
    gtk_widget_show(treeview);
    gtk_container_add(GTK_CONTAINER(scrolled_win), treeview);
    gtk_widget_realize(skinwin);
    skinwin_set_icon(skinwin);
}

void make_skin_window(void)
{
    create_skin_window();
}

void show_skin_window(void)
{
    gtk_widget_show_all(skinwin);
}
