/*
 * Copyright 2000 Murray Cumming
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <bakery_gnomeui/App/App_WithDoc_GnomeUI.h>
#include <bakery/App/GtkDialogs.h>
#include <libgnome/gnome-i18n.h>

#define BAKERY_GCONF_DIRECTORY_SESSIONDOCS "session_docs"

namespace Bakery
{

App_WithDoc_GnomeUI::App_WithDoc_GnomeUI(const Glib::ustring& appname, const Glib::ustring& title)
: App_WithDoc(appname),
  App_GnomeUI(appname, title)
{
  m_iMenuPos_File = 0;
  m_iMenuPos_File_Save = 0;
  m_pMenuItem_File_Save = 0;
  m_pToolbarItem_Save = 0;
  m_bMenuOpenRecentCreated = false;
}

App_WithDoc_GnomeUI::~App_WithDoc_GnomeUI()
{
}

void App_WithDoc_GnomeUI::init()
{
  //Repeat some stuff from App_GnomeUI::init(): TODO: Maybe have a bool bMenusAndToolbarsInited.
  set_statusbar(m_Status); //We must do this before creating the menus.

    
  App_WithDoc::init(); //Create document and ask to show it in the UI. (also creates the menus and toolbars indirectly)
  
  //If the app just started, then reload any previously-open documents.
  static bool bAlreadyReopening = false; //prevent endless loop when this instance is re-used.
  if(bAlreadyReopening == false)
  {
    if(m_AppInstanceManager.get_app_count() == 1)
    {
      bAlreadyReopening = true;
      session_load_documents_open();
    }
  }
  
  show();
}

void App_WithDoc_GnomeUI::init_toolbars()
{
  using namespace Gnome::UI::Items;

  m_toolbar_UI_Infos.push_back(Item(Icon(Gtk::Stock::NEW),
    N_("New "),
    SigC::slot(*this, &App_WithDoc::on_menu_file_new),
    N_("Create a new " + m_strAppName)));

  m_toolbar_UI_Infos.push_back(Item(Icon(Gtk::Stock::OPEN),
    N_("Open "),
    SigC::slot(*this, &App_WithDoc::on_menu_file_open),
    N_("Open a " + m_strAppName)));

  m_toolbar_UI_Infos.push_back(Item(Icon(Gtk::Stock::SAVE),
    N_("Save "),
    SigC::slot(*this, &App_WithDoc::on_menu_file_save),
    N_("Save this " + m_strAppName)));
  guint iToolbarPos_Save = m_toolbar_UI_Infos.size() - 1; //Remember for later.

  Gnome::UI::Items::Array<Gnome::UI::Items::Info>& arrayInfp = create_toolbar(m_toolbar_UI_Infos);

  //Get Toolbar 'Save' widget:
  m_pToolbarItem_Save = arrayInfp[iToolbarPos_Save].get_widget();
}


void App_WithDoc_GnomeUI::init_menus_file()
{
  // File menu
  type_vecGnome_UI_Info menu_file;

  //Build menu:
  menu_file.push_back(Gnome::UI::MenuItems::New(_("New Document"), _("Create a new document"), SigC::slot(*this, &App_WithDoc::on_menu_file_new)));
  menu_file.push_back(Gnome::UI::MenuItems::Open(SigC::slot(*this, &App_WithDoc::on_menu_file_open)));
  //'Open Recent' added later.
  menu_file.push_back(Gnome::UI::MenuItems::Save(SigC::slot(*this, &App_WithDoc::on_menu_file_save)));
  m_iMenuPos_File_Save = menu_file.size() - 1; //Remember for later.  
  menu_file.push_back(Gnome::UI::MenuItems::SaveAs(SigC::slot(*this, &App_WithDoc::on_menu_file_saveas)));

  menu_file.push_back(Gnome::UI::Items::Separator());
  menu_file.push_back(Gnome::UI::MenuItems::Close(SigC::slot(*this, &App_WithDoc::on_menu_file_close)));
  menu_file.push_back(Gnome::UI::MenuItems::Exit(SigC::slot(*this, &App_WithDoc::on_menu_file_exit)));

  //Add menu:
  m_menu_UI_Infos.push_back(Gnome::UI::Menus::File(menu_file));
  m_iMenuPos_File = m_menu_UI_Infos.size() - 1; //Remember for later.
}

void App_WithDoc_GnomeUI::init_menus()
{
  //Overriden to get widget pointers:

  init_menus_file();
  init_menus_edit();
  init_menus_help();

  Gnome::UI::Items::Array<Gnome::UI::Items::SubTree>& arraySubTrees = create_menus(m_menu_UI_Infos);

  Gnome::UI::Items::Array<Gnome::UI::Items::Info>& arrayFileMenu = arraySubTrees[m_iMenuPos_File].get_uitree();
  m_pMenuItem_File_Save = arrayFileMenu[m_iMenuPos_File_Save].get_widget();

  install_menu_hints();

  //Override this to add more menus.
}

void App_WithDoc_GnomeUI::update_window_title()
{
  //Set application's main window title:
  
  Glib::ustring strTitle = m_strAppName;
  Document* pDoc = get_document();
  if(pDoc)
  {
    strTitle += " - " + pDoc->get_name();

    //Indicate unsaved changes:
    if(pDoc->get_modified())
      strTitle += " *";

    set_title(strTitle);
  }
}


bool App_WithDoc_GnomeUI::on_session_save_yourself(int phase, Gnome::UI::SaveStyle save_style, bool is_shutdown, Gnome::UI::InteractStyle interact_style, bool is_fast)
{
  session_save_document_history();
  session_save_documents_open();

  return true; //return App::on_session_save_yourself(phase, save_style, is_shutdown, interact_style, is_fast);
}

void App_WithDoc_GnomeUI::show_document_history()
{
  //Update File menu in all instances:
  AppInstanceManager::type_listAppInstances apps = m_AppInstanceManager.get_instances();
  for(AppInstanceManager::type_listAppInstances::iterator iter = apps.begin(); iter != apps.end(); iter++)
  {
    App_WithDoc_GnomeUI* pApp = dynamic_cast<App_WithDoc_GnomeUI*>(*iter);
    if(pApp)
    {
      if(pApp->m_bMenuOpenRecentCreated)
      {
        //Remove current "Open Recent" sub-menu:
        pApp->remove_menus( "File/Open Recent", 1 );
      }

      //Add new sub menu:
      type_vecGnome_UI_Info vecUI_Info;
      guint index = 0;
      for(type_vecStrings::iterator iter = m_vecDocHistory.begin(); iter != m_vecDocHistory.end(); iter++)
      {
        Glib::ustring strName = *iter;
        if(!strName.empty())
        {
          strName = Document::util_filepath_get_name(strName);
          vecUI_Info.push_back(Gnome::UI::Items::Item(strName,
                               SigC::bind<guint>(SigC::slot(*pApp, &App_WithDoc::on_menu_file_open_recent), index),
                               _("Open ") + strName));
        }

        index++;
      }

      type_vecGnome_UI_Info vecUI_Info_SubTree;
      vecUI_Info_SubTree.push_back(Gnome::UI::Items::SubTree(_("Open Recent"), vecUI_Info, _("Open a recently-used document.")));            ;
      pApp->insert_menus("File/Open...", vecUI_Info_SubTree);
      pApp->m_bMenuOpenRecentCreated = true;
    }
  }

}

void App_WithDoc_GnomeUI::session_save_documents_open()
{
  //Save list of open documents:
   
  //Build list of documents:
  typedef std::list<Glib::ustring> type_listStrings;
  type_listStrings listStrings;

  const AppInstanceManager::type_listAppInstances apps = m_AppInstanceManager.get_instances();
  for(AppInstanceManager::type_listAppInstances::const_iterator iter = apps.begin(); iter != apps.end(); iter++)
  {
    Glib::ustring strFilepath;
    const App_WithDoc_GnomeUI* pApp = dynamic_cast<const App_WithDoc_GnomeUI*>(*iter);
    if(pApp)
      strFilepath = pApp->get_document()->get_filepath();

    if(!strFilepath.empty())
       listStrings.push_back(strFilepath);
  }

  //Store the list:
  Glib::RefPtr<Gnome::Conf::Client> refConfClient = Gnome::Conf::Client::get_default_client();

  try
  {
    refConfClient->set_string_list(get_conf_fullkey(BAKERY_GCONF_DIRECTORY_SESSIONDOCS), listStrings);
  }
  catch(const Gnome::Conf::Error& ex)
  {
    std::cerr << "Bakery: App_WithDoc_GnomeUI::session_save_documents_open(): " << ex.what() << std::endl;
  }
}

void App_WithDoc_GnomeUI::session_load_documents_open()
{
  //Load previously-open documents:
  
  Glib::RefPtr<Gnome::Conf::Client> refConfClient = Gnome::Conf::Client::get_default_client();

  typedef std::list<Glib::ustring> type_listStrings;
  type_listStrings listStrings;
  try
  {
    listStrings = refConfClient->get_string_list(get_conf_fullkey(BAKERY_GCONF_DIRECTORY_SESSIONDOCS));
  }
  catch(const Gnome::Conf::Error& ex)
  {
    //Maybe they were never saved. That's not a problem.
    //std::cerr << "Bakery: App_WithDoc_GnomeUI::session_load_documents_open(): " << ex.what() << std::endl;
  }
  

  //Load items:
  for(type_listStrings::iterator iter = listStrings.begin(); iter != listStrings.end(); ++iter)
  {
    open_document(*iter);
  }
}

void App_WithDoc_GnomeUI::ui_warning(const Glib::ustring& text)
{
  GtkDialogs::ui_warning(text);
}

Glib::ustring App_WithDoc_GnomeUI::ui_file_select_open()
{
  return GtkDialogs::ui_file_select_open();
}

Glib::ustring App_WithDoc_GnomeUI::ui_file_select_save(const Glib::ustring& old_filepath)
{
  return GtkDialogs::ui_file_select_save(old_filepath);
}

App_WithDoc::enumSaveChanges App_WithDoc_GnomeUI::ui_offer_to_save_changes()
{
  return GtkDialogs::ui_offer_to_save_changes(m_pDocument->get_filepath());
}

bool App_WithDoc_GnomeUI::ui_ask_overwrite(const std::string& filepath)
{
  return GtkDialogs::ui_ask_overwrite(m_pDocument->get_filepath());
}

void App_WithDoc_GnomeUI::ui_show_modification_status()
{
  bool modified = m_pDocument->get_modified();

  //Enable Save and SaveAs menu items:
  if(m_pMenuItem_File_Save)
    m_pMenuItem_File_Save->set_sensitive(modified);

  if(m_pToolbarItem_Save)
    m_pToolbarItem_Save->set_sensitive(modified);
}

  

} //namespace
