/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import nz.net.catalyst.ELog;
import nz.net.catalyst.Log;
import nz.net.catalyst.StringPair;
//import nz.net.catalyst.Util;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;

/**
 * Execute a Lucene Index request
 */

public class Index implements IPackage, Constants
{
  /**
   * The following headers are special and therefore do not describe
   * document fields.
   */

  private static final Set SPECIAL_HEADERS = new HashSet();
  static
  {
    SPECIAL_HEADERS.add(ID);
    SPECIAL_HEADERS.add(DOMAIN);
    SPECIAL_HEADERS.add(APPLICATION);
    SPECIAL_HEADERS.add(STOP_LIST);
    SPECIAL_HEADERS.add(FIELD_NAME);
    SPECIAL_HEADERS.add(FIELD_TYPE);
    SPECIAL_HEADERS.add(FIELD_INDEXED);
    SPECIAL_HEADERS.add(FIELD_STORED);
    SPECIAL_HEADERS.add(CONTENT_LENGTH);
    SPECIAL_HEADERS.add(BODY);
    SPECIAL_HEADERS.add(SERIAL);
  }

  private final Transmission input;

  private Application application;

  private String domain;
  private String id;

  Index(Transmission transmission)
  {
    input = transmission;
  }

  /**
   * Index a document
   */

  Transmission execute()
  {
    long indexStart = System.currentTimeMillis();
    Transmission response = new Transmission(ECommand.INDEX_RESPONSE);
    response.setSerial(input.getSerial());

    id = input.get(ID, NO_APP);
    if (id == null)
      return error("Mandatory '" + ID + "' header is missing");

    String appName = input.get(APPLICATION, NO_APP);
    try
    {
      application = Application.getAppOrDefault(appName);
    }
    catch (ApplicationMissingException e)
    {
      return error(e.getMessage());
    }

    input.setApplication(application);
    domain = input.get(DOMAIN);

    if (domain == null)
      return error("'" + DOMAIN +
                   "' header is missing and is not in Application defaults");

    Analyzer analyzer = Application.getAnalyzer(input); // Decide upon an analyzer

    // Get List of FieldDef field definitions.
    List fields = input.getFields();

    File luceneStoreDir = Application.getIndexDirectory(application);

    // Create the Lucene document and add the fields to it.
    Document document = createDocument(fields);

    boolean succeed = false;

    long beforeLock = System.currentTimeMillis();
    long afterLock;
    long afterReaderOpen;
    long afterDelete;
    long afterReaderClose;
    long afterIndex;
    long afterWriterOpen;

    boolean create = false;

    if (!luceneStoreDir.isDirectory())
    {
      // If the lucene directory does not exist, then we need to
      // make a brand new index

        luceneStoreDir.mkdirs();

        if (!luceneStoreDir.isDirectory())
          return error("Cannot access Lucene Index in directory: " +
                       luceneStoreDir.getPath());

        create = true;
    }
    afterReaderOpen = afterDelete = afterReaderClose = System.currentTimeMillis();
    WriterControl writerControl = null;
    String stage = "";

    try
    {
      if (!create)
      {
        // Delete the current instance of the document from the index.

        IndexReader reader = null;
        succeed = false;
        stage = "opening IndexReader";

        try
        {
          writerControl = WriterControl.closeWriter(luceneStoreDir, true);
          reader = IndexReader.open(luceneStoreDir);
          afterReaderOpen = System.currentTimeMillis();
          Term keyField = new Term(DOCUMENT_KEY, domain + "\u0000" + id);
          int count = reader.delete(keyField);
          afterDelete = System.currentTimeMillis();
          Log.debug("Deleted " + count + " documents");

          //THESE LINES NO LONGER NEEDED! Can delete without searching.
          // Query keyQuery = new TermQuery(keyField);
          // Searcher searcher = new IndexSearcher(reader);
          // Hits hits = searcher.search(keyQuery);
          //
          // if (hits.length() > 0)
          // {
          //   int count = reader.delete(keyField);
          //   afterDelete = System.currentTimeMillis();
          //   Log.debug("Deleted " + count + " documents");
          // }

          succeed = true;
        }
        catch (FileNotFoundException e)
        {
          // Maybe the index is brand new and contains no files!
          Log.debug("Maybe the index is brand new and contains no files!");
          Log.log(ELog.ERROR, e.toString());
        }
        catch (IOException e)
        {
          String message = "Error while deleting previous document " + DOMAIN +
            '=' + domain + ", " + ID +'=' + id + ": " + e.toString();
          Log.log(ELog.ERROR, message);
          return error(message);
        }
        finally
        {
          if (reader != null)
          {
            try
            {
              // We must always close the IndexReader!
              reader.close();
            }
            catch (Throwable e)
            {
              String message = "Error while closing IndexReader: " + e.toString();
              Log.log(ELog.ERROR, message);

              if (succeed)
                return error(message);
            }
          }
        }
        afterReaderClose = System.currentTimeMillis();
      }

      succeed = false;
      stage = "opening IndexWriter";

      writerControl = WriterControl.getWriterControl(luceneStoreDir, analyzer);
      IndexWriter writer = writerControl.getIndexWriter();
      afterWriterOpen = System.currentTimeMillis();
      stage = "adding document";
      writer.addDocument(document);
      afterIndex = System.currentTimeMillis();
      succeed = true;
      stage = "";
    }
    catch (IOException e)
    {
      String message = "Error while " + stage + ", " + DOMAIN + '=' +
        domain + ", " + ID +'=' + id + ": " + e.toString();
      Log.log(ELog.ERROR, message);
      return error(message);
    }
    finally
    {
      if (writerControl != null)
      {
        try
        {
          // We must always free the IndexWriter for other users.
          writerControl.release();
        }
        catch (Throwable e)
        {
          String message = "Error while closing IndexWriter: " + e.toString();
          Log.log(ELog.ERROR, message);
          if (succeed)
            return error(message);
        }
      }
    }
    long afterWriterClose = System.currentTimeMillis();

    /*
    response.add("OpenReader", String.valueOf(afterReaderOpen - beforeLock));
    response.add("DeleteOld",  String.valueOf(afterDelete - afterReaderOpen));
    response.add("ReaderClose",String.valueOf(afterReaderClose - afterDelete));
    response.add("WriterOpen", String.valueOf(afterWriterOpen - afterReaderClose));
    response.add("IndexDoc",   String.valueOf(afterIndex - afterWriterOpen));
    response.add("WriterClose",String.valueOf(afterWriterClose - afterIndex));
	*/
  	response.add("Status","Document indexed successfully");
	
    // Single-threaded: Get the Index Writer and create the document.
    return response;
  }

  /**
   * Create a Lucene Document from a List of FieldDef definitions
   *
   * @param fields a List of FieldDef objects defining the fields to
   * be in the document
   *
   * Return a new Lucene Document object.
   */

  private Document createDocument(List fields)
  {
    // make a new, empty document
    Document document = new Document();

    // Add the special composite-key
    document.add(Field.Keyword(DOCUMENT_KEY, domain + "\u0000" + id));

    Set processed = new HashSet();

    // Add all the pre-defined fields...
  addPredefinedFields:
    for (Iterator it = fields.iterator(); it.hasNext(); )
    {
      FieldDef field = (FieldDef)it.next();
      String value;

      if (field.name.equals(ID))
        value = id;
      else if (field.name.equals(DOMAIN))
        value = domain;
      else
      {
        value = input.get(field.name);
        if (value == null)
          continue addPredefinedFields;
      }
      Log.debug("Adding predefined field \"" + field.name + "\" to document: Value is: \"" + nz.net.catalyst.Util.clean(value) + '"');
    	//Log.debug("Adding predefined field \"" + field.name + "\" to document: Value is: \"" + value + '"');

      processed.add(field.name);
      if (field.date)
      {
        try
        {
          value = DateField.dateToString(Application.makeDate(value));
          Log.debug("  =" + value);
        }
        catch (IllegalArgumentException e)
        {
          Log.log(ELog.ERROR, "Can't parse \"" + field.name + '=' + value + "\" as a date");
          continue addPredefinedFields;
        }
      }

      document.add(new Field(field.name, value, field.store, field.index, field.token));
    }

    // Now add any other fields that may be lurking

  addExtraFields:
    for (Iterator it = input.getHeadersView().iterator();
         it.hasNext(); )
    {
      StringPair header = (StringPair)it.next();
      String key = header.getKeyString();

      if (SPECIAL_HEADERS.contains(key) || processed.contains(key))
        continue addExtraFields;

      FieldDef field = new FieldDef(key);
      String value = input.get(field.name);

      Log.debug("Adding undefined field \"" + field.name +
                         "\" to document");
      processed.add(field.name);
      document.add(new Field(field.name, value, field.store, field.index, field.token));
    }

    return document;
  }


  /**
   * Build an error response for sending back to the client.
   *
   * @param message The text of the error message
   * @return An INDEX-RESPONSE Transmission
   */

  private Transmission error(String message)
  {
    Transmission response = new Transmission(ECommand.INDEX_RESPONSE);
    response.setSerial(input.getSerial());
    response.add(ERROR, message);
    return response;
  }
}