/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;

import nz.net.catalyst.ELog;
import nz.net.catalyst.Log;
import nz.net.catalyst.Pair;
import nz.net.catalyst.Util;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Field;


/**
 * Maintain Application defaults.  This class will monitor properties
 * files for changes and reread them after a change. It will also
 * rewrite to the file any changes made to the application properties.  <p>
 *
 * The name of the directory which contains the properties files is
 * stored in the "nz.net.catalyst.lucene.server.ApplicationDirectory"
 * System Property.  <p>
 *
 * This directory is read at startup and any files matching
 * *.properties are loaded.  These files are assumed to be correctly
 * constructed Java properties files.  For a given property file,
 * xxxx.properties, all the properties it contains become the default
 * values for Application xxxx.  <p>
 *
 * The directory is rescanned periodically and checked for changes to
 * the modified time of the files.  If any files have been touched,
 * then there properties are re-read.  The property which controls the
 * rescan interval is
 * "nz.net.catalyst.lucene.server.ApplicationRescan".  This should
 * contain a number of minutes between directory rescans.  The default
 * if this property is missing or invalid is one minute.
 *
 * When changes are rewritten to the file, any comments recorded in
 * the file are lost.
 */

public class Application implements IPackage, Constants
{
  /**
   * The property which defines the name of the Directory where the
   * Application Properties files are kept.
   */
  private static final String DIRECTORY_PROPERTY = PACKAGE + "ApplicationDirectory";
  private static final String RESCAN_PROPERTY = PACKAGE + "ApplicationRescan";
  private static final String UNDEFINED_APP_PROP = PACKAGE + "MissingAppIsError";


  private static final String PROP_FILE_SUFFIX = ".properties";

  private static final String DEFAULT_INDEX_PROPERTY = PACKAGE + LUCENE_INDEX_DIRECTORY_PROPERTY;

  /**
   * Date formats supported by the application to convert to lucene dates.
   * <p>
   * Currently:
   * <br><pre>
   *yyyy/MM/dd HH:mm:ss
   *yyyy-MM-dd HH:mm:ss
   *yyyy/MM/dd
   *yyyy-MM-dd</pre>
   *and a unix timestamp
   */
  private static final SimpleDateFormat [] sdf = {
    new SimpleDateFormat("yyyy/MM/dd HH:mm:ss"),
    new SimpleDateFormat("yyyy-MM-dd HH:mm:ss"),
    new SimpleDateFormat("yyyy/MM/dd"),
    new SimpleDateFormat("yyyy-MM-dd"),
  };

  static
  {
    for (int i = 0; i < sdf.length; ++i)
      sdf[i].setLenient(false);
  }

  private Analyzer analyzer = null;

  /**
   * A List of FieldDef objects, which is the result of parsing the
   * Field-Definition property.  Each field is uniquely identified by
   * its name and appears no more than once in the list.
   */
  private final List fields = new LinkedList();

  /**
   * An unmodifiable view of the fields List that can be released to
   * clients.  Note that the contained FieldDef objects are not
   * unmodifiable!
   */
  private final List fieldsView = Collections.unmodifiableList(fields);

  /**
   * A cross-reference by name of all the fields stored in the <fields> List.
   */
  private final Map fieldMap = new HashMap();

  /**
   * Cache of all Application defaults.
   * Key is String(AppName), Value is Application()
   */
  private static final Map cache = new TreeMap();

  private final Properties properties = new Properties();

  private final String name;

  // Use the file last-modified-time to keep track of whether the file
  // needs to be loaded.  If the file last-modified-time is less than
  // the Application lastModTime, then the Application data is written
  // back to the disk and the times are set equal.  If the file
  // last-modified-time time is greater than the Application
  // lastModTime, then the Application is discarded and read afresh
  // from the file.

  private long lastModTime = -1L;

  public Application(String name)
  {
    this.name = name;

    synchronized(cache)
    {
      cache.put(name, this);
    }
    organise();
  }

  /**
   * Get (a copy of) the definition of a named field
   */
  FieldDef getFieldDef(String name)
  {
    FieldDef field = (FieldDef)fieldMap.get(name);
    if (field == null)
      field = new FieldDef(name);
    else
      field = field.copy();

    return field;
  }

  /**
   * Get an unmodifiable view of the List of all field definitions.
   *
   * @return a List containing FieldDef objects.  The List cannot be
   * manipulated but the FieldDefs can!  Don't do it!!
   */
  List getAllFieldDefs()
  {
    return fieldsView;
  }

  /**
   * Build the Indexer field using the appropriate field attributes
   */
  public Field makeField(String name, String value)
  {
    FieldDef field = getFieldDef(name);
    if (field.date)
      value = DateField.dateToString(makeDate(value));

    return new Field(name, value, field.store, field.index, field.token);
  }


  /**
   * Obtain a Lucene Analyzer instance for based on the Stop-List
   * provided in the Transmission and the Default Stop-List.
   * This will be the default Analyzer for the Application unless the
   * Stop-List has been overridden, in which case we build a new one.
   *
   * @return An Analyzer
   */

  public static Analyzer getAnalyzer(Transmission tr)
  {
    String stopList = tr.get(STOP_LIST, NO_APP);
    Application app = tr.getApplication();
    
    if (stopList == null && app != null)
      return app.getAnalyzer();

    return AnalyzerPool.get(stopList);
  }

  /**
   * Obtain the directory of the Lucene index.  This will generally
   * have a system default but may be overridden on a per-application
   * level.
   *
   * @param application The Application instance to check for a
   * directory override, or null
   *
   * @return The Directory of the Lucene Index to use.
   */

  public static File getIndexDirectory(Application application)
  {
    File luceneStoreDir = null;

    if (application != null)
    {
      String luceneStore = application.getProperty(LUCENE_INDEX_DIRECTORY_PROPERTY);
      if (luceneStore != null)
      {
        luceneStoreDir = new File(luceneStore);
        if (!luceneStoreDir.isDirectory())
        {
          luceneStoreDir.mkdirs();

          if (!luceneStoreDir.isDirectory())
            luceneStoreDir = null;
        }
      }
    }

    if (luceneStoreDir == null)
    {
      String luceneStore = System.getProperty(DEFAULT_INDEX_PROPERTY);
      if (luceneStore == null)
        luceneStore = DEFAULT_INDEX_DIRECTORY;
      else
        luceneStore = luceneStore.trim();

      luceneStoreDir = new File(luceneStore);
    }
    return luceneStoreDir;
  }

  /**
   * Try to convert a String into a Date.  Use either the number of
   * seconds since the epoch (standard Unix time) or a parse against a
   * number of date format options (all specifying local time).
   *
   * @param value The string representation of a date (trimmed).
   *
   * @throws IllegalArgumentException if date cannot be parsed.
   */

  public static Date makeDate(String value)
  {
    try
    {
      return new Date(Integer.parseInt(value) * 1000L);
    }
    catch (NumberFormatException e)
    {
      // Ignore
    }

    synchronized(sdf)
    {
      for (int i = 0; i < sdf.length; ++i)
      {
        try
        {
          return sdf[i].parse(value);
        }
        catch (ParseException e)
        {
          // Ignore
        }
      }
    }
    throw new IllegalArgumentException("Cannot parse date: " + value);
  }

  public static String format(Date d)
  {
    synchronized(sdf)
    {
      return sdf[0].format(d);
    }
  }

  public synchronized String remove(String key)
  {
    String oldValue = (String) properties.remove(key);
    long now = System.currentTimeMillis();
    lastModTime = lastModTime < now ? now : lastModTime + 1;
    return oldValue;
  }

  public String appendProperty(String key, String value)
  {
    return appendProperty(key, value, " ");
  }

  public synchronized String appendProperty(String key, String value, String sep)
  {
    String oldValue;
    synchronized(properties)
    {
      oldValue = (String) properties.setProperty(key, value);
      if (oldValue != null)
        properties.setProperty(key, oldValue + sep + value);
    }

    long now = System.currentTimeMillis();
    lastModTime = lastModTime < now ? now : lastModTime + 1;
    return oldValue;
  }

  public synchronized String setProperty(String key, String value)
  {
    String oldValue = (String) properties.setProperty(key, value);
    long now = System.currentTimeMillis();
    lastModTime = lastModTime < now ? now : lastModTime + 1;
    return oldValue;
  }

  public synchronized String getProperty(String key)
  {
    return properties.getProperty(key);
  }

  public synchronized Properties getProperties()
  {
    return properties;
  }

  /**
   * This is only ever called for a brand-new Application object that
   * isn't even in the cache yet.
   */

  private synchronized void load(File file) throws IOException
  {
    Log.log(ELog.INFO, "Reading Application properties from: " + file.getPath());
    InputStream in = new BufferedInputStream(new FileInputStream(file));

    try
    {
      synchronized (properties)
      {
        properties.clear();
        properties.load(in);
      }
    }
    finally
    {
      in.close();
    }
    lastModTime = file.lastModified();
    organise();
  }

  private synchronized void store(File file) throws IOException
  {
    Log.log(ELog.INFO, "Writing Application properties to: " + file.getPath());
    OutputStream out = new BufferedOutputStream(new FileOutputStream(file));
    try
    {
      synchronized (properties)
      {
        properties.store(out, name);
      }
    }
    finally
    {
      out.close();
    }
    lastModTime = file.lastModified();
  }

  /**
   * Construct a number of useful objects based on the data in the
   * Application and try to write the Application defaults to a disk
   * file.  If the write to the disk fails, then the defaults are not
   * persisted, but can still be used in this session.
   */

  public void store()
  {
    organise();

    File directory = new File(System.getProperty(DIRECTORY_PROPERTY).trim());
    if (!directory.isDirectory())
    {
      directory.mkdirs();
      if (!directory.isDirectory())
      {
        Log.log(ELog.ERROR, "Cannot store Application defaults into directory: "+
                           Util.getFullPath(directory));
        return;
      }
    }

    File file = new File(directory, name + PROP_FILE_SUFFIX);
    try
    {
      store(file);
    }
    catch (IOException e)
    {
      Log.log(ELog.ERROR, "Cannot store Application defaults into file: "+
                         Util.getFullPath(file) + ": " + e);
      return;
      // Log the fact that the properties could not be stored!
    }
  }

  /**
   * Use the details stored in the properties to generate a bunch of
   * structures that will be needed for indexing.
   */

  private synchronized void organise()
  {
    buildAnalyzer();
    buildFieldDefs();
  }

  /**
   * Generate a Standard Lucene Analyzer based on the current set of
   * Stop words.
   */

  private void buildAnalyzer()
  {
    analyzer = AnalyzerPool.get(properties.getProperty(STOP_LIST));
  }

  Analyzer getAnalyzer()
  {
    return analyzer;
  }

  /**
   * Build a list and Map of field definitions based on the property
   * string.  The property string consists of a comma-separated list
   * of field definitions.  Each field definition is a space-separated
   * list of four terms: <ol>
   *
   * <li>The field name (case sensitive)
   *
   * <li>The field type: Id, Text or Date.  Id fields are not
   * tokenized, indexed and stored.  Text fields are tokenized,
   * indexed and not stored.  Date fields are specially mangled,
   * indexed and stored.  The type is optional and defaults to Text
   *
   * <li>Whether the field is indexed (Yes or No).  This is optional
   * and will be taken from the Type if not specified.  If a field is
   * not indexed, then it cannot be used as a search field.
   *
   * <li>Whether the field is stored (Yes or No).  This is optional
   * and will be taken from the Type if not specified.  If a field is
   * not stored, then it cannot be returned after a query.
   *
   * </ol>
   */

  private void buildFieldDefs()
  {
    fields.clear();
    fieldMap.clear();

    String fieldProperty = properties.getProperty(FIELD_DEFS);
    if (fieldProperty == null)
      return;


    String[] fieldDefList = Util.split(fieldProperty, ",");

  fieldParsing:
    for (int i = 0; i < fieldDefList.length; ++i)
    {
      String fieldDef = fieldDefList[i];
      String[] fieldAttributeList = Util.split(fieldDef);

      if (fieldAttributeList.length == 0)
        continue fieldParsing;

      FieldDef field = new FieldDef(fieldAttributeList[0]);

      if (fieldAttributeList.length >= 2)
      {
        String type = fieldAttributeList[1];
        field.token = type.equalsIgnoreCase(TEXT);
        field.date = type.equalsIgnoreCase(DATE);
      }

      if (fieldAttributeList.length >= 3)
      {
        String indexed = fieldAttributeList[2];
        field.index = indexed.equalsIgnoreCase(YES);
      }

      if (fieldAttributeList.length >= 4)
      {
        String stored = fieldAttributeList[3];
        field.store = stored.equalsIgnoreCase(YES);
      }

      int index;

      if (fieldMap.containsKey(field.name))
      {
        index = fields.indexOf(field);
        fields.remove(index);
      }
      else
        index = fields.size();

      fields.add(index, field);
      fieldMap.put(field.name, field);
    }
    String specials[] = new String[]{ID, DOMAIN};
    for (int i = 0; i < specials.length; ++i)
    {
      FieldDef field = (FieldDef)fieldMap.get(specials[i]);
      if (field == null)
      {
        field = new FieldDef(specials[i]);
        field.store = true;
        field.index = false;
        field.token = false;
        fields.add(0, field);
        fieldMap.put(field.name, field);
      }
    }
  }

   /**
   * Return the Application object corresponding to a name.  If the
   * Application does not exist or the name is null, then return the
   * default application.  If that does not exist either, return null.
   *
   * @param name The name of the Application object to get. May be null.
   * @return the named Application, or the default Application if there
   * is no Application with the name given.
   */

  public static Application getAppOrDefault(String name)
  {
    synchronized(cache)
    {
      Application result = name == null ? null : (Application)cache.get(name);
      if (result == null)
      {
        if (name != null && Boolean.getBoolean(UNDEFINED_APP_PROP))
          throw new ApplicationMissingException("Application \"" +
                                                name + "\" is not defined");

        result = (Application)cache.get(DEFAULT_APPLICATION);
      }

      return result;
    }
  }

   /**
   * Return the Application object corresponding to a name.
   * If the Application does not exist, then return the default
   * application.  If that does not exist either, return null.
   *
   * @param The name of the Application object to get.
   */

  public static Application get(String name)
  {
    synchronized(cache)
    {
      return (Application)cache.get(name);
    }
  }

  /**
   * Optional Start-up.  This will cause the class to start up a
   * monitor thread which will check for changes to the Application
   * properties files and reload them if changes occur.
   */

  public static void Start(String[] args)
  {
    scanDirectory();
    startMonitor();
  }

  /**
   * Scan the Application directory for changes to the properties
   * files, and write out any Applications that have changed.
   */

  private static synchronized void scanDirectory()
  {
    // Log.debug("Scanning for changed Application properties files");
    File directory = new File(System.getProperty(DIRECTORY_PROPERTY).trim());
    if (!directory.isDirectory())
      return;

    // Find all properties files in the directory.

    String[] propsFiles = directory.list(new FilenameFilter()
      {
        public boolean accept(File parent, String file)
        {
          return file.endsWith(PROP_FILE_SUFFIX);
        }
      });

    // Check whether any of the properties files need to be read.

    for (int i = 0; i < propsFiles.length; ++i)
    {
      int baseLength = propsFiles[i].length() - PROP_FILE_SUFFIX.length();
      String appName = propsFiles[i].substring(0, baseLength);

      boolean readApp = false;

      Application app = get(appName);
      File propFile = new File(directory, propsFiles[i]);

      if (app == null || app.lastModTime < propFile.lastModified())
      {
        try
        {
          new Application(appName).load(propFile);
        }
        catch (IOException e)
        {
          synchronized (cache)
          {
            cache.remove(appName);
          }
        }
      }
    }

    // List of Applications objects that need to be stored

    Collection changes = new ArrayList(cache.size());

    synchronized (cache)
    {
      // Check whether any Application objects need to be re-written.
      for (Iterator it = cache.values().iterator(); it.hasNext(); )
      {
        Application app = (Application)it.next();
        File propFile = new File(directory, app.name + PROP_FILE_SUFFIX);

        // If Application needs rewriting, save the details for later.
        if (!propFile.exists() || app.lastModTime > propFile.lastModified())
          changes.add(new Pair(propFile, app));
      }
    }

    // No longer locking cache, now rewrite the Application objects.

    for (Iterator it = changes.iterator(); it.hasNext(); )
    {
      Pair pair = (Pair)it.next();
      File propFile = (File)pair.getKey();
      Application app = (Application)pair.getValue();
      try
      {
        app.store(propFile);
      }
      catch (IOException e)
      {
        // Log an error, else ignore
      }
    }
  }

  private static void monitor()
  {
    try
    {
      for(;;)
      {
        int sleepMinutes = Integer.getInteger(RESCAN_PROPERTY, 1).intValue();
        if (sleepMinutes < 1)
          sleepMinutes = 1;

        Thread.sleep(sleepMinutes * 60L * 1000L);
        scanDirectory();
      }
    }
    catch(InterruptedException e)
    {
      // Don't know how this happened, but I'm going to kill this thread!
    }
    finally
    {
      // If thread dies for any reason, start a new instance.
      startMonitor();
    }
  }

  private static void startMonitor()
  {
    Thread monitor = new Thread(new Runnable(){
        public void run(){ monitor(); }
      }, "Application Monitor");
    monitor.setDaemon(true);  // Won't keep program alive.

    // Use reduced priority for polling loop.
    int priority = monitor.getPriority() - 1;
    if (priority < Thread.MIN_PRIORITY)
      ++priority;
    monitor.setPriority(priority);
    monitor.start();
  }
}
