/*
*  Aquamarine the KDE window decorator
*
*  Copyright (c) 2006 Dennis Kasprzyk <onestone@beryl-project.org>
*  Copyright (c) 2006 Volker Krause <vkrause@kde.org>
*  Copyright (c) 2006 David Reveman <davidr@novell.com>
*
*  Uses code of:
*      KWin window manager (www.kde.org)
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*
*/

#include <qglobal.h>

#include <dcopclient.h>
#include <kcmdlineargs.h>
#include <kconfig.h>
#include <kdebug.h>
#include <kglobal.h>
#include <kwinmodule.h>
#include <klocale.h>
#include <kcommondecoration.h>
#include <kwin.h>
#include <qwidgetlist.h>
#include <qpoint.h>

#include <X11/Xlib.h>
#include <X11/extensions/Xcomposite.h>

#include "decorator.h"
#include "options.h"
#include "utils.h"
#include "settings.h"

#include <stdio.h>

decor_context_t Aquamarine::Decorator::mDefaultContext;
decor_shadow_t *Aquamarine::Decorator::mNoBorderShadow = 0;
decor_shadow_t *Aquamarine::Decorator::mDefaultShadow  = 0;
Aquamarine::PluginManager *Aquamarine::Decorator::mPlugins = 0;
Aquamarine::Options *Aquamarine::Decorator::mOptions = 0;
NETRootInfo *Aquamarine::Decorator::mRootInfo;
WId Aquamarine::Decorator::mActiveId;
decor_shadow_options_t Aquamarine::Decorator::mShadowOptions;

QSize Aquamarine::Decorator::mDesktop;
QPoint Aquamarine::Decorator::mViewport;
QSize Aquamarine::Decorator::mRootSize;


extern Time qt_x_time;

struct _cursor cursors[3][3] = {
	{ C (top_left_corner), C (top_side), C (top_right_corner) },
	{ C (left_side), C (left_ptr), C (right_side) },
	{ C (bottom_left_corner), C (bottom_side), C (bottom_right_corner) }
};

Aquamarine::PluginManager::PluginManager (KConfig *config): KDecorationPlugins (config)
{
	// taken from KWin
	defaultPlugin = (QPixmap::defaultDepth () > 8) ? "kwin3_plastik" : "kwin3_quartz";
}

Aquamarine::Decorator::Decorator (void) : DCOPObject ("KWinInterface"),
	KApplication (),
	mConfig (0),
	mDecorNormal (0), mDecorActive (0),
	mKWinModule (new KWinModule (this, KWinModule::INFO_ALL)),
	mCompositeWindow (0)
{
	KGlobal::locale()->insertCatalogue("kdelibs");
	KGlobal::locale()->setActiveCatalogue("aquamarine");
	XSync (qt_xdisplay(), false);

	mReady = false;
	mRestart = false;

	int event, error;

	DCOPClient *client;
	int	       i, j;

	mRootInfo = new NETRootInfo (qt_xdisplay (), 0);

	mActiveId = 0;

	Atoms::init ();

	if (!XDamageQueryExtension (qt_xdisplay (), &event, &error))
	{
		fprintf (stderr, "Damage extension is missing on display \"%s\"\n",
			DisplayString (qt_xdisplay ()));
		return;
	}

	mActiveWM = XGetSelectionOwner(qt_xdisplay(),Atoms::wmSn);
	if (mActiveWM == None)
		return;
	Aquamarine::trapXError ();
	XSelectInput (qt_xdisplay(), mActiveWM, StructureNotifyMask);
	if (Aquamarine::popXError ()) return;

	KCmdLineArgs *args = KCmdLineArgs::parsedArgs ();

	Time timestamp;
	int status = decor_acquire_dm_session (qt_xdisplay (), qt_xscreen(), "Aquamarine",
					args->isSet ("replace"),&timestamp);

	if (status != DECOR_ACQUIRE_STATUS_SUCCESS)
	{
		if (status == DECOR_ACQUIRE_STATUS_OTHER_DM_RUNNING)
		{
			fprintf (stderr,
				"Could not acquire decoration manager "
				"selection on screen %d display \"%s\"\n",
				qt_xscreen(), DisplayString (qt_xdisplay ()));
		}
		else if (status == DECOR_ACQUIRE_STATUS_OTHER_DM_RUNNING)
		{
			fprintf (stderr,
				"Screen %d on display \"%s\" already "
				"has a decoration manager; try using the "
				"--replace option to replace the current "
				"decoration manager.\n",
				qt_xscreen(), DisplayString (qt_xdisplay ()));
		}
		return;
	}
	decor_set_dm_check_hint (qt_xdisplay (), qt_xscreen());

	mConfig = new KConfig ("kwinrc");
	mConfig->setGroup ("Style");

	mOptions = new Aquamarine::Options (mConfig);
	mPlugins = new PluginManager (mConfig);

	for (i = 0; i < 3; i++)
	{
	for (j = 0; j < 3; j++)
	{
		if (cursors[i][j].shape != XC_left_ptr)
		cursors[i][j].cursor =
			XCreateFontCursor (qt_xdisplay (), cursors[i][j].shape);
	}
	}

	client = dcopClient ();
	client->registerAs ("kwin", false);
	client->setDefaultObject ("KWinInterface");

	mShadowOptions.shadow_radius   = (Settings::drawShadows())?
			Settings::shadowRadius():0;
	mShadowOptions.shadow_opacity  = Settings::shadowOpacity() / 100.0;
	mShadowOptions.shadow_offset_x = Settings::shadowOffsetX();
	mShadowOptions.shadow_offset_y = Settings::shadowOffsetY();
	mShadowOptions.shadow_color[0] = Settings::shadowColor().red() * 0x101;
	mShadowOptions.shadow_color[1] = Settings::shadowColor().green() * 0x101;
	mShadowOptions.shadow_color[2] = Settings::shadowColor().blue() * 0x101;
	mShadowOptions.shadow_color[3] = 0xffff;

	mCompositeWindow = new QWidget (0, "AquamarineCompositeWidget",
					Qt::WType_TopLevel | Qt::WStyle_NoBorder |
					Qt::WX11BypassWM);

	mCompositeWindow->setGeometry (QRect (-ROOT_OFF_X, -ROOT_OFF_Y, 1, 1));
	mCompositeWindow->show ();

	XCompositeRedirectSubwindows (qt_xdisplay (), mCompositeWindow->winId (),
				CompositeRedirectManual);

	if (!enableDecorations (timestamp, event))
	{
		fprintf (stderr, "Could not enable decorations on display \"%s\"\n",
				DisplayString (qt_xdisplay ()));
		return;
	}

	updateDesktopGeometry ();

	mReady = true;
}

Aquamarine::Decorator::~Decorator (void)
{
	QMap <WId, Aquamarine::Window *>::ConstIterator it;

	for (it = mClients.begin (); it != mClients.end (); it++)
		delete (*it);

	mClients.clear ();
	mFrames.clear ();
	mWindows.clear ();

	if (mDefaultShadow)
	{
		decor_shadow_destroy (qt_xdisplay (), mDefaultShadow);
		mDefaultShadow = NULL;
	}

	if (mNoBorderShadow)
	{
		decor_shadow_destroy (qt_xdisplay (), mNoBorderShadow);
		mNoBorderShadow = NULL;
		Aquamarine::trapXError ();
		XDeleteProperty (qt_xdisplay (), qt_xrootwin (),
						 Atoms::netWindowDecorBare);
		Aquamarine::popXError ();
	}

	if (mDecorNormal)
	{
		delete mDecorNormal;
		mDecorNormal = NULL;
	}

	if (mDecorActive)
	{
		delete mDecorActive;
		mDecorActive = NULL;
	}

	if (mOptions)
	{
		delete mOptions;
		mOptions = NULL;
	}

	if (mPlugins)
	{
		delete mPlugins;
		mPlugins = NULL;
	}

	if (mConfig)
	{
		delete mConfig;
		mConfig = NULL;
	}

	if (mKWinModule)
	{
		delete mKWinModule;
		mKWinModule = NULL;
	}

	if (mRootInfo)
	{
		delete mRootInfo;
		mRootInfo = NULL;
	}

	/*
	if (mCompositeWindow)
	{
		delete mCompositeWindow;
		mCompositeWindow = 0;
	}
	*/
}

bool Aquamarine::Decorator::enableDecorations (Time timestamp, int damageEvent)
{
	QValueList <WId>::ConstIterator it;

	mDmSnTimestamp = timestamp;
	mDamageEvent   = damageEvent;

	if (!pluginManager ()->loadPlugin (""))
		return false;

	updateShadow ();

	mDecorNormal = new Aquamarine::Window (mCompositeWindow, qt_xrootwin (),
					0, Window::Default);
	mDecorActive = new Aquamarine::Window (mCompositeWindow, qt_xrootwin (),
					0, Window::DefaultActive);

	connect (mKWinModule, SIGNAL (windowAdded (WId)),
		SLOT (handleWindowAdded (WId)));
	connect (mKWinModule, SIGNAL (windowRemoved (WId)),
		SLOT (handleWindowRemoved (WId)));
	connect (mKWinModule, SIGNAL (activeWindowChanged (WId)),
		SLOT (handleActiveWindowChanged (WId)));
	connect (mKWinModule, SIGNAL (windowChanged (WId, const unsigned long *)),
		SLOT (handleWindowChanged (WId, const unsigned long *)));

	connect (&mIdleTimer, SIGNAL (timeout ()), SLOT (processDamage ()));

	mActiveId = mKWinModule->activeWindow ();

	it = mKWinModule->windows ().begin ();
	for (; it != mKWinModule->windows ().end (); it++)
		handleWindowAdded ((*it));

	connect (this, SIGNAL (appearanceChanged ()), SLOT (reconfigure ()));

	(void) QApplication::desktop (); // trigger creation of desktop widget

	// select for client messages
	XSelectInput (qt_xdisplay(), qt_xrootwin (),
		StructureNotifyMask | PropertyChangeMask);

	return true;
}

void Aquamarine::Decorator::updateDefaultShadow (Aquamarine::Window *w)
{
	bool uniqueHorzShape, uniqueVertShape;

	if (mDefaultShadow)
	{
		decor_shadow_destroy (qt_xdisplay (), mDefaultShadow);
		mDefaultShadow = NULL;
	}

	w->getShapeInfo (&uniqueHorzShape, &uniqueVertShape);

	/* only return shadow if decoration doesn't use a unique shape */
	if (uniqueHorzShape || uniqueVertShape)
		return;

	mDefaultContext = *w->context ();
	mDefaultShadow  = w->shadow ();

	if (mDefaultShadow)
		decor_shadow_reference (mDefaultShadow);
}

void Aquamarine::Decorator::updateAllShadowOptions (void)
{
	mShadowOptions.shadow_radius   = (Settings::drawShadows())?
			Settings::shadowRadius():0;
	mShadowOptions.shadow_opacity  = Settings::shadowOpacity() / 100.0;
	mShadowOptions.shadow_offset_x = Settings::shadowOffsetX();
	mShadowOptions.shadow_offset_y = Settings::shadowOffsetY();
	mShadowOptions.shadow_color[0] = Settings::shadowColor().red() * 0x101;
	mShadowOptions.shadow_color[1] = Settings::shadowColor().green() * 0x101;
	mShadowOptions.shadow_color[2] = Settings::shadowColor().blue() * 0x101;
}

void Aquamarine::Decorator::updateShadow (void)
{
	Display	    *xdisplay = qt_xdisplay ();
	Screen	    *xscreen = ScreenOfDisplay (xdisplay, qt_xscreen ());
	decor_context_t context;

	if (mDefaultShadow)
	{
		decor_shadow_destroy (xdisplay, mDefaultShadow);
		mDefaultShadow = NULL;
	}

	if (mNoBorderShadow)
		decor_shadow_destroy (xdisplay, mNoBorderShadow);

	mNoBorderShadow = decor_shadow_create (xdisplay,
					xscreen,
					1, 1,
					0,
					0,
					0,
					0,
					0, 0, 0, 0,
					&mShadowOptions,
					&context,
					decor_draw_simple,
					0);

	if (mNoBorderShadow)
	{
		decor_extents_t extents = { 0, 0, 0, 0 };
		long	        data[256];
		decor_quad_t    quads[N_QUADS_MAX];
		int	        nQuad;
		decor_layout_t  layout;

		decor_get_default_layout (&context, 1, 1, &layout);

		nQuad = decor_set_lSrStSbS_window_quads (quads, &context, &layout);

		decor_quads_to_property (data, mNoBorderShadow->pixmap,
					&extents, &extents,
					0, 0, quads, nQuad);

		Aquamarine::trapXError ();
		XChangeProperty (qt_xdisplay (), qt_xrootwin (),
				Atoms::netWindowDecorBare,
				XA_INTEGER,
				32, PropModeReplace, (unsigned char *) data,
				BASE_PROP_SIZE + QUAD_PROP_SIZE * nQuad);
		Aquamarine::popXError ();
	}
}

void Aquamarine::Decorator::processDamage (void)
{
	QMap <WId, Aquamarine::Window *>::ConstIterator it;

	mDecorNormal->processDamage ();
	mDecorActive->processDamage ();

	for (it = mClients.constBegin (); it != mClients.constEnd (); it++)
	it.data ()->processDamage ();
}

bool Aquamarine::Decorator::x11EventFilter (XEvent *xevent)
{
	Aquamarine::Window *client;
	int		status;

	switch (xevent->type) {
		case MapNotify:
		{
			XMapEvent *xme = reinterpret_cast <XMapEvent *> (xevent);

			if (mWindows.contains (xme->window))
				client = mWindows[xme->window];
			else if (mDecorNormal->winId () == xme->window)
				client = mDecorNormal;
			else if (mDecorActive->winId () == xme->window)
				client = mDecorActive;
			else
				break;

			if (client->handleMap ())
			{
				if (!mIdleTimer.isActive ())
				mIdleTimer.start (0, TRUE);
			}
			break;
		}
		case ConfigureNotify:
		{
			XConfigureEvent *xce = reinterpret_cast <XConfigureEvent *> (xevent);

			if (mFrames.contains (xce->window))
				mFrames[xce->window]->updateFrame (xce->window);

			if (mWindows.contains (xce->window))
				client = mWindows[xce->window];
			else if (mDecorNormal->winId () == xce->window)
				client = mDecorNormal;
			else if (mDecorActive->winId () == xce->window)
				client = mDecorActive;
			else
				break;

			if (client->handleConfigure ())
			{
				if (!mIdleTimer.isActive ())
				mIdleTimer.start (0, TRUE);
			}
			break;
		}
		case DestroyNotify:
			if (xevent->xdestroywindow.window == mActiveWM)
			{
				mRestart = true;
				KApplication::exit(0);
			}
			break;
		case SelectionRequest:
			decor_handle_selection_request (qt_xdisplay (), xevent, mDmSnTimestamp);
			break;
		case SelectionClear:
			status = decor_handle_selection_clear (qt_xdisplay (),
							xevent, 0);
			if (status == DECOR_SELECTION_GIVE_UP)
				KApplication::exit (0);
			break;
		case PropertyNotify:
			if (xevent->xproperty.atom == Atoms::netFrameWindow)
			{
				handleWindowAdded (xevent->xproperty.window);
			}
			else if (xevent->xproperty.atom == Atoms::switchSelectWindow)
			{
				if (!mClients.contains (xevent->xproperty.window))
				{
					handleWindowAdded (xevent->xproperty.window);
				}
				else
				{
				WId id;

				if (Aquamarine::readWindowProperty (xevent->xproperty.window,
								Atoms::switchSelectWindow,
								(long *) &id))
					mClients[xevent->xproperty.window]->updateSelected (id);
				}
			}
			else if (xevent->xproperty.atom == Atoms::netWmWindowOpacity)
			{
				if (mClients.contains (xevent->xproperty.window))
				mClients[xevent->xproperty.window]->updateOpacity ();
			}
			else if (xevent->xproperty.atom == Atoms::netWmWindowBrightness)
			{
				if (mClients.contains (xevent->xproperty.window))
					mClients[xevent->xproperty.window]->updateBrightness ();
			}
			else if (xevent->xproperty.atom == Atoms::netWmWindowSaturation)
			{
				if (mClients.contains (xevent->xproperty.window))
					mClients[xevent->xproperty.window]->updateSaturation ();
			}
			else if (xevent->xproperty.atom == Atoms::netDesktopGeometry
				|| xevent->xproperty.atom == Atoms::netDesktopViewport)
			{
				updateDesktopGeometry ();
			}
			break;
		case EnterNotify:
		{
			XCrossingEvent *xce = reinterpret_cast <XCrossingEvent *> (xevent);
			QWidget	       *widget, *child;

			if (!mFrames.contains (xce->window))
				break;

			client = mFrames[xce->window];

			widget = client->decoration ()->widget ();
			child = widget->childAt (xce->x, xce->y, true);
			if (child)
			{
				QEvent qe (QEvent::Enter);

				QApplication::sendEvent (child, &qe);

				client->setActiveChild (child);
				client->updateCursor (QPoint (xce->x, xce->y));
			}
			break;
		}
		case LeaveNotify:
		{
			XCrossingEvent *xce = reinterpret_cast <XCrossingEvent *> (xevent);

			if (mFrames.contains (xce->window))
			{
				QEvent qe (QEvent::Leave);

				client = mFrames[xce->window];

				QApplication::sendEvent (client->activeChild (), &qe);

				XUndefineCursor (qt_xdisplay (), client->frameId ());
			}
			break;
		}
		case MotionNotify:
		{
			XMotionEvent *xme = reinterpret_cast < XMotionEvent * >(xevent);
			QWidget	     *widget, *child;

			if (!mFrames.contains (xme->window))
				break;

			client = mFrames[xme->window];

			widget = client->decoration ()->widget ();
			child = widget->childAt (xme->x, xme->y, true);
			if (child)
			{
				QPoint qp (xme->x, xme->y);

				if (child != client->activeChild ())
				{
				QEvent qee (QEvent::Enter);
				QEvent qle (QEvent::Leave);

				if (client->activeChild ())
					QApplication::sendEvent (client->activeChild (), &qle);

				QApplication::sendEvent (child, &qee);

				client->setActiveChild (child);
				}

				if (widget != child)
				qp -= QPoint (child->pos ().x (), child->pos ().y ());

				QMouseEvent qme (QEvent::MouseMove, qp, Qt::NoButton, Qt::NoButton);

				QApplication::sendEvent (child, &qme);

				client->updateCursor (QPoint (xme->x, xme->y));
			}
			break;
		}
		case ButtonPress:
		case ButtonRelease:
		{
			XButtonEvent *xbe = reinterpret_cast <XButtonEvent *>(xevent);
			QWidget	     *widget, *child;

			if (!mFrames.contains (xbe->window))
				break;

			client = mFrames[xbe->window];

			widget = client->decoration ()->widget ();
			child = widget->childAt (xbe->x, xbe->y, true);

			if (child)
			{
				XButtonEvent xbe2 = *xbe;

				xbe2.window = child->winId ();
				if (widget != child)
				{
				xbe2.x = xbe->x - child->pos ().x ();
				xbe2.y = xbe->y - child->pos ().y ();
				}

				QApplication::x11ProcessEvent ((XEvent *) &xbe2);

				return true;
			}
			break;
		}
		case ClientMessage:
			if (xevent->xclient.message_type == Atoms::toolkitActionAtom)
			{
				unsigned long action;

				action = xevent->xclient.data.l[0];
				if (action == Atoms::toolkitActionWindowMenuAtom)
				{
				if (mClients.contains (xevent->xclient.window))
				{
					QPoint pos;

					client = mClients[xevent->xclient.window];

					if (xevent->xclient.data.l[2])
					{
					pos = QPoint (xevent->xclient.data.l[3],
							xevent->xclient.data.l[4]);
					}
					else
					{
					pos = client->clientGeometry ().topLeft ();
					}

					client->showWindowMenu (pos);
				}
				}
				else if (action == Atoms::toolkitActionMainMenuAtom)
				{
				dcopClient ()->send ("kicker",
							"kicker",
							"showKMenu()",
							QByteArray ());
				}
				else if (action == Atoms::toolkitActionRunDialogAtom)
				{
				dcopClient ()->send ("kdesktop",
							"KDesktopIface",
							"popupExecuteCommand()",
							QByteArray ());
				}
				else if (action == Atoms::toolkitActionForceQuitDialogAtom)
				{
				if (mClients.contains (xevent->xclient.window))
				{
					Time timestamp = xevent->xclient.data.l[1];

					client = mClients[xevent->xclient.window];

					if (xevent->xclient.data.l[2])
					client->showKillProcessDialog (timestamp);
					else
					client->hideKillProcessDialog ();
				}
				}
			}
			break;
		default:
			if (xevent->type == mDamageEvent + XDamageNotify)
			{
				XDamageNotifyEvent *xde =
				reinterpret_cast <XDamageNotifyEvent *>(xevent);

				if (mWindows.contains (xde->drawable))
				client = mWindows[xde->drawable];
				else if (mDecorNormal->winId () == xde->drawable)
				client = mDecorNormal;
				else if (mDecorActive->winId () == xde->drawable)
				client = mDecorActive;
				else
				break;

				client->addDamageRect (xde->area.x,
						xde->area.y,
						xde->area.width,
						xde->area.height);

				if (client->pixmapId ())
				{
				if (!mIdleTimer.isActive ())
					mIdleTimer.start (0, TRUE);
				}

				return true;
			}
		break;
	}

	return KApplication::x11EventFilter (xevent);
}

void
Aquamarine::Decorator::reconfigure (void)
{
	unsigned long changed;

	Settings::self()->config()->reparseConfiguration();
	Settings::self()->readConfig();

	updateAllShadowOptions();

	mConfig->reparseConfiguration ();

	changed = mOptions->updateSettings ();

	updateShadow ();

	if (mPlugins->reset (changed))
	{
		QMap < WId, Aquamarine::Window * >::ConstIterator it;

		mDecorNormal->reloadDecoration ();
		mDecorActive->reloadDecoration ();

		for (it = mClients.constBegin (); it != mClients.constEnd (); it++)
			it.data ()->reloadDecoration ();

		mPlugins->destroyPreviousPlugin ();
	}
	else
	{
		QMap < WId, Aquamarine::Window * >::ConstIterator it;

		mDecorNormal->updateShadow ();
		mDecorActive->updateShadow ();

		for (it = mClients.constBegin (); it != mClients.constEnd (); it++)
			it.data ()->updateShadow ();
	}
}

void
Aquamarine::Decorator::handleWindowAdded (WId id)
{
	QMap <WId, Aquamarine::Window *>::ConstIterator it;
	Aquamarine::Window				     *client = 0;
	WId					     select, frame = 0;
	Aquamarine::Window::Type			     type;
	unsigned int			     width, height, border, depth;
	int					     x, y;
	XID					     root;
	QWidgetList				     *widgets;

	/* avoid adding any of our own top level windows */
	widgets = QApplication::topLevelWidgets ();
	if (widgets)
	{
		for (QWidgetListIt it (*widgets); it.current (); ++it)
		{
			if (it.current ()->winId () == id)
			{
				delete widgets;
				return;
			}
		}

		delete widgets;
	}

	Aquamarine::trapXError ();
	XGetGeometry (qt_xdisplay (), id, &root, &x, &y, &width, &height,
		&border, &depth);
	if (Aquamarine::popXError ())
		return;

	Aquamarine::readWindowProperty (id, Atoms::netFrameWindow, (long *) &frame);
	if (Aquamarine::readWindowProperty (id, Atoms::switchSelectWindow,
				(long *) &select))
	{
		type = Aquamarine::Window::Switcher;
	}
	else
	{
		KWin::WindowInfo wInfo = KWin::windowInfo (id, NET::WMWindowType, 0);

		switch (wInfo.windowType (~0)) {
			case NET::Normal:
			case NET::Dialog:
			case NET::Toolbar:
			case NET::Menu:
			case NET::Utility:
			case NET::Splash:
			case NET::Unknown:
				/* decorate these window types */
				break;
			default:
				return;
		}

		type = Aquamarine::Window::Normal;
	}

	Aquamarine::trapXError ();
	XSelectInput (qt_xdisplay (), id, StructureNotifyMask | PropertyChangeMask);
	Aquamarine::popXError ();

	if (frame)
	{
		if (!mClients.contains (id))
		{
			client = new Aquamarine::Window (mCompositeWindow, id, frame, type,
						x, y,
						width + border * 2,
						height + border * 2);
			connect( this, SIGNAL(desktopGeometryChanged()), client,
					 		SLOT(handleDesktopGeometryChanged()) );

			mClients.insert (id, client);
			mWindows.insert (client->winId (), client);
			mFrames.insert (frame, client);
		}
		else
		{
			client = mClients[id];
			mFrames.remove (client->frameId ());
			mFrames.insert (frame, client);

			client->updateFrame (frame);
		}
	}
	else if (type == Aquamarine::Window::Switcher)
	{
		if (!mClients.contains (id))
		{
			client = new Aquamarine::Window (mCompositeWindow, id, 0, type,
						x, y,
						width + border * 2,
						height + border * 2);
			mClients.insert (id, client);
			mWindows.insert (client->winId (), client);
		}
	}
	else
	{
		if (mClients.contains (id))
			client = mClients[id];

		if (client)
		{
			mClients.remove (client->windowId ());
			mWindows.remove (client->winId ());
			mFrames.remove (client->frameId ());

			delete client;
		}
	}
}

void
Aquamarine::Decorator::handleWindowRemoved (WId id)
{
	Aquamarine::Window *window = 0;

	if (mClients.contains (id))
		window = mClients[id];
	else if (mFrames.contains (id))
		window = mFrames[id];

	if (window)
	{
		mClients.remove (window->windowId ());
		mWindows.remove (window->winId ());
		mFrames.remove (window->frameId ());

		delete window;
	}
}

void
Aquamarine::Decorator::handleActiveWindowChanged (WId id)
{
	if (id != mActiveId)
	{
		Aquamarine::Window *newActiveWindow = 0;
		Aquamarine::Window *oldActiveWindow = 0;

		if (mClients.contains (id))
			newActiveWindow = mClients[id];

		if (mClients.contains (mActiveId))
			oldActiveWindow = mClients[mActiveId];

		mActiveId = id;

		if (oldActiveWindow)
			oldActiveWindow->handleActiveChange ();

		if (newActiveWindow)
			newActiveWindow->handleActiveChange ();
	}
}

void
Aquamarine::Decorator::handleWindowChanged (WId		 id,
					const unsigned long *properties)
{
	Aquamarine::Window *client;

	if (!mClients.contains (id))
		return;

	client = mClients[id];

	if (properties[0] & NET::WMName)
		client->updateName ();
	if (properties[0] & NET::WMState)
		client->updateState ();
	if (properties[0] & NET::WMIcon)
		client->updateIcons ();
	if (properties[0] & NET::WMGeometry)
		client->updateWindowGeometry ();
}

void
Aquamarine::Decorator::sendClientMessage (WId  eventWid,
				WId  wid,
				Atom atom,
				Atom value,
				long data1,
				long data2,
				long data3)
{
	XEvent ev;
	long   mask = 0;

	memset (&ev, 0, sizeof (ev));

	ev.xclient.type	    = ClientMessage;
	ev.xclient.window	    = wid;
	ev.xclient.message_type = atom;
	ev.xclient.format       = 32;

	ev.xclient.data.l[0] = value;
	ev.xclient.data.l[1] = qt_x_time;
	ev.xclient.data.l[2] = data1;
	ev.xclient.data.l[3] = data2;
	ev.xclient.data.l[4] = data3;

	if (eventWid == qt_xrootwin ())
		mask = SubstructureRedirectMask | SubstructureNotifyMask;

	Aquamarine::trapXError ();
	XSendEvent (qt_xdisplay (), eventWid, false, mask, &ev);
	Aquamarine::popXError ();
}

void Aquamarine::Decorator::updateDesktopGeometry ()
{
	int x, y;
	unsigned int width, height, border, depth;
	::Window root;
	XGetGeometry (qt_xdisplay (), qt_xrootwin (), &root, &x, &y, &width,
				&height, &border, &depth);
	mRootSize = QSize (width, height);

	int	num_val;
	void *values = readXProperty (qt_xrootwin (), Atoms::netDesktopGeometry,
					XA_CARDINAL, &num_val);

	if (values)
	{
		unsigned long *
			val = (unsigned long *)values;
		mDesktop = QSize (val[0], val[1]);
		XFree (val);
	}
	else
	{
		mDesktop = QSize (mRootSize.width (), mRootSize.height ());
	}
	values = readXProperty (qt_xrootwin (), Atoms::netDesktopViewport,
					XA_CARDINAL, &num_val);

	if (values)
	{
		unsigned long *val = (unsigned long *)values;
		mViewport = QPoint (val[0], val[1]);
		XFree (val);
	}
	else
	{
		mViewport = QPoint (0, 0);
	}

	emit desktopGeometryChanged();
}


int Aquamarine::Decorator::onViewport (QPoint pos)
{
	return (((pos.y () +
			mViewport.y ()) / mRootSize.height ()) * (mDesktop.width () /
														mRootSize.
														width ())) +
		((pos.x () + mViewport.x ()) / mRootSize.width ()) + 1;
}

QSize Aquamarine::Decorator::viewports ()
{
	return QSize ((mDesktop.width () / mRootSize.width ()),
				(mDesktop.height () / mRootSize.height ()));
}

void Aquamarine::Decorator::moveToViewport (WId id, int vp)
{
	int x, y;
	unsigned int width, height, border, depth;
	::Window root;
	XGetGeometry (qt_xdisplay (), id, &root, &x, &y, &width, &height, &border,
				&depth);
	if (vp <= 0) vp+= (mDesktop.width () / mRootSize.width ()) * (mDesktop.height () / mRootSize.height ());
	vp = ((vp - 1) % ((mDesktop.width () / mRootSize.width ()) * (mDesktop.height () / mRootSize.height ()))) + 1;
	int	offsetX = mRootSize.width () * ((vp - 1) % viewports ().width ());
	int	offsetY = mRootSize.height () * ((vp - 1) / viewports ().width ());
	XMoveWindow (qt_xdisplay (), id, x + offsetX - mViewport.x (),
				y + offsetY - mViewport.y ());
}
