# Autoconf checks for pthreads.
#
# Source: http://ac-archive.sourceforge.net/Installed_Packages/acx_pthread.html

AC_DEFUN([ACX_PTHREAD], [
AC_REQUIRE([AC_CANONICAL_HOST])
AC_LANG_SAVE
AC_LANG_C
acx_pthread_ok=no

# We used to check for pthread.h first, but this fails if pthread.h
# requires special compiler flags (e.g. on True64 or Sequent).
# It gets checked for in the link test anyway.

# First of all, check if the user has set any of the PTHREAD_LIBS,
# etcetera environment variables, and if threads linking works using
# them:
if test x"$PTHREAD_LIBS$PTHREAD_CFLAGS" != x; then
        save_CFLAGS="$CFLAGS"
        CFLAGS="$CFLAGS $PTHREAD_CFLAGS"
        save_LIBS="$LIBS"
        LIBS="$PTHREAD_LIBS $LIBS"
        AC_MSG_CHECKING([for pthread_join in LIBS=$PTHREAD_LIBS with CFLAGS=$PTHREAD_CFLAGS])
        AC_TRY_LINK_FUNC(pthread_join, acx_pthread_ok=yes)
        AC_MSG_RESULT($acx_pthread_ok)
        if test x"$acx_pthread_ok" = xno; then
                PTHREAD_LIBS=""
                PTHREAD_CFLAGS=""
        fi
        LIBS="$save_LIBS"
        CFLAGS="$save_CFLAGS"
fi

# We must check for the threads library under a number of different
# names; the ordering is very important because some systems
# (e.g. DEC) have both -lpthread and -lpthreads, where one of the
# libraries is broken (non-POSIX).

# Create a list of thread flags to try.  Items starting with a "-" are
# C compiler flags, and other items are library names, except for "none"
# which indicates that we try without any flags at all, and "pthread-config"
# which is a program returning the flags for the Pth emulation library.

acx_pthread_flags="pthreads none -Kthread -kthread lthread -pthread -pthreads -mthreads pthread --thread-safe -mt pthread-config"

# The ordering *is* (sometimes) important.  Some notes on the
# individual items follow:

# pthreads: AIX (must check this before -lpthread)
# none: in case threads are in libc; should be tried before -Kthread and
#       other compiler flags to prevent continual compiler warnings
# -Kthread: Sequent (threads in libc, but -Kthread needed for pthread.h)
# -kthread: FreeBSD kernel threads (preferred to -pthread since SMP-able)
# lthread: LinuxThreads port on FreeBSD (also preferred to -pthread)
# -pthread: Linux/gcc (kernel threads), BSD/gcc (userland threads)
# -pthreads: Solaris/gcc
# -mthreads: Mingw32/gcc, Lynx/gcc
# -mt: Sun Workshop C (may only link SunOS threads [-lthread], but it
#      doesn't hurt to check since this sometimes defines pthreads too;
#      also defines -D_REENTRANT)
# pthread: Linux, etcetera
# --thread-safe: KAI C++
# pthread-config: use pthread-config program (for GNU Pth library)

case "${host_cpu}-${host_os}" in
        *solaris*)

        # On Solaris (at least, for some versions), libc contains stubbed
        # (non-functional) versions of the pthreads routines, so link-based
        # tests will erroneously succeed.  (We need to link with -pthread or
        # -lpthread.)  (The stubs are missing pthread_cleanup_push, or rather
        # a function called by this macro, so we could check for that, but
        # who knows whether they'll stub that too in a future libc.)  So,
        # we'll just look for -pthreads and -lpthread first:

        acx_pthread_flags="-pthread -pthreads pthread -mt $acx_pthread_flags"
        ;;
esac

if test x"$acx_pthread_ok" = xno; then
for flag in $acx_pthread_flags; do

        case $flag in
                none)
                AC_MSG_CHECKING([whether pthreads work without any flags])
                ;;

                -*)
                AC_MSG_CHECKING([whether pthreads work with $flag])
                PTHREAD_CFLAGS="$flag"
                ;;

		pthread-config)
		AC_CHECK_PROG(acx_pthread_config, pthread-config, yes, no)
		if test x"$acx_pthread_config" = xno; then continue; fi
		PTHREAD_CFLAGS="`pthread-config --cflags`"
		PTHREAD_LIBS="`pthread-config --ldflags` `pthread-config --libs`"
		;;

                *)
                AC_MSG_CHECKING([for the pthreads library -l$flag])
                PTHREAD_LIBS="-l$flag"
                ;;
        esac

        save_LIBS="$LIBS"
        save_CFLAGS="$CFLAGS"
        LIBS="$PTHREAD_LIBS $LIBS"
        CFLAGS="$CFLAGS $PTHREAD_CFLAGS"

        # Check for various functions.  We must include pthread.h,
        # since some functions may be macros.  (On the Sequent, we
        # need a special flag -Kthread to make this header compile.)
        # We check for pthread_join because it is in -lpthread on IRIX
        # while pthread_create is in libc.  We check for pthread_attr_init
        # due to DEC craziness with -lpthreads.  We check for
        # pthread_cleanup_push because it is one of the few pthread
        # functions on Solaris that doesn't have a non-functional libc stub.
        # We try pthread_create on general principles.
        AC_TRY_LINK([#include <pthread.h>],
                    [pthread_t th; pthread_join(th, 0);
                     pthread_attr_init(0); pthread_cleanup_push(0, 0);
                     pthread_create(0,0,0,0); pthread_cleanup_pop(0); ],
                    [acx_pthread_ok=yes])

        LIBS="$save_LIBS"
        CFLAGS="$save_CFLAGS"

        AC_MSG_RESULT($acx_pthread_ok)
        if test "x$acx_pthread_ok" = xyes; then
                break;
        fi

        PTHREAD_LIBS=""
        PTHREAD_CFLAGS=""
done
fi

# Various other checks:
if test "x$acx_pthread_ok" = xyes; then
        save_LIBS="$LIBS"
        LIBS="$PTHREAD_LIBS $LIBS"
        save_CFLAGS="$CFLAGS"
        CFLAGS="$CFLAGS $PTHREAD_CFLAGS"

        # Detect AIX lossage: threads are created detached by default
        # and the JOINABLE attribute has a nonstandard name (UNDETACHED).
        AC_MSG_CHECKING([for joinable pthread attribute])
        AC_TRY_LINK([#include <pthread.h>],
                    [int attr=PTHREAD_CREATE_JOINABLE;],
                    ok=PTHREAD_CREATE_JOINABLE, ok=unknown)
        if test x"$ok" = xunknown; then
                AC_TRY_LINK([#include <pthread.h>],
                            [int attr=PTHREAD_CREATE_UNDETACHED;],
                            ok=PTHREAD_CREATE_UNDETACHED, ok=unknown)
        fi
        if test x"$ok" != xPTHREAD_CREATE_JOINABLE; then
                AC_DEFINE(PTHREAD_CREATE_JOINABLE, $ok,
                          [Define to the necessary symbol if this constant
                           uses a non-standard name on your system.])
        fi
        AC_MSG_RESULT(${ok})
        if test x"$ok" = xunknown; then
                AC_MSG_WARN([we do not know how to create joinable pthreads])
        fi

        AC_MSG_CHECKING([if more special flags are required for pthreads])
        flag=no
        case "${host_cpu}-${host_os}" in
                *-aix* | *-freebsd*)     flag="-D_THREAD_SAFE";;
                *solaris* | *-osf* | *-hpux*) flag="-D_REENTRANT";;
        esac
        AC_MSG_RESULT(${flag})
        if test "x$flag" != xno; then
                PTHREAD_CFLAGS="$flag $PTHREAD_CFLAGS"
        fi

        LIBS="$save_LIBS"
        CFLAGS="$save_CFLAGS"

        # More AIX lossage: must compile with cc_r
        AC_CHECK_PROG(PTHREAD_CC, cc_r, cc_r, ${CC})
else
        PTHREAD_CC="$CC"
fi

AC_SUBST(PTHREAD_LIBS)
AC_SUBST(PTHREAD_CFLAGS)
AC_SUBST(PTHREAD_CC)

# Finally, execute ACTION-IF-FOUND/ACTION-IF-NOT-FOUND:
if test x"$acx_pthread_ok" = xyes; then
        ifelse([$1],,AC_DEFINE(HAVE_PTHREAD,1,[Define if you have POSIX threads libraries and header files.]),[$1])
        :
else
        acx_pthread_ok=no
        $2
fi
AC_LANG_RESTORE
])dnl ACX_PTHREAD

#########################################################
# AQSIS_CHECK_BOOST

AC_DEFUN([AQSIS_CHECK_BOOST],[

AC_SUBST(AQSIS_BOOST_CPPFLAGS)

AC_ARG_WITH(boost_include,[  --with-boost-include=DIR path to BOOST includes [[/usr/include]]],
	[if test "$withval" != no; then
		AQSIS_BOOST_CPPFLAGS="-I$withval"
	else
		AQSIS_BOOST_CPPFLAGS="-I/usr/include"
	fi],[AQSIS_BOOST_CPPFLAGS="-I/usr/include"])

save_cppflags=$CPPFLAGS
CPPFLAGS="$AQSIS_BOOST_CPPFLAGS"/boost
AC_CHECK_HEADER(version.hpp)
if test "$ac_cv_header_version_hpp" = no; then
	AC_MSG_ERROR([*** ERROR: Can't find BOOST headers. You may need to install BOOST ***])
fi
CPPFLAGS=$save_cppflags

])

########################################################
# AQSIS_CHECK_FLTK

AC_DEFUN([AQSIS_CHECK_FLTK], [
AC_SUBST(AQSIS_FLTK_CFLAGS)
AC_SUBST(AQSIS_FLTK_LIBS)
AC_MSG_CHECKING([for the fltk UI library])
if fltk-config --version >/dev/null 2>/dev/null; then
	AQSIS_FLTK_VERSION=`fltk-config --version`
	AQSIS_FLTK_CFLAGS=`fltk-config --cflags`
	AQSIS_FLTK_LIBS=`fltk-config --ldflags`
	AC_MSG_RESULT([found version $AQSIS_FLTK_VERSION])
else
	AC_MSG_ERROR([couldn't find fltk library])
fi
])

######################################################################################
# AQSIS_CHECK_LIBJPEG

AC_DEFUN([AQSIS_CHECK_LIBJPEG], [

AC_SUBST(AQSIS_JPEG_CPPFLAGS)
AC_SUBST(AQSIS_JPEG_LDFLAGS)

AC_ARG_WITH(jpeg_include,[  --with-jpeg-include=DIR path to jpeg includes [[/usr/local/include]]],
	[if test "$withval" != no; then
		AQSIS_JPEG_CPPFLAGS="-I$withval"
	else
		AQSIS_JPEG_CPPFLAGS="-I/usr/include"
	fi],[AQSIS_JPEG_CPPFLAGS="-I/usr/include"])

AC_ARG_WITH(jpeg_lib,[  --with-jpeg-lib=DIR     path to jpeg libraries [[/usr/local/lib]]],
	[if test "$withval" != no; then
		AQSIS_JPEG_LDFLAGS="-L$withval -ljpeg"
	else
		AQSIS_JPEG_LDFLAGS="-L/usr/lib -ljpeg"
	fi],[AQSIS_JPEG_LDFLAGS="-L/usr/lib -ljpeg"])

save_cflags=$CFLAGS
save_libs=$LIBS

CFLAGS=$AQSIS_JPEG_CPPFLAGS
LIBS=$AQSIS_JPEG_LDFLAGS

AC_CHECK_HEADER(jpeglib.h)
AC_CHECK_LIB(jpeg, jpeg_read_header)

CFLAGS=$save_cflags
LIBS=$save_libs

if test ! "$ac_cv_header_jpeglib_h" || test ! "ac_cv_lib_jpeg_jpeg_read_header" ; then
        AC_MSG_ERROR([*** Please ensure you have libjpeg libraries and headers installed ***]);
fi

])

#####################################################################################
# AQSIS_CHECK_LIBTIFF

AC_DEFUN([AQSIS_CHECK_LIBTIFF], [

AC_SUBST(AQSIS_TIFF_CPPFLAGS)
AC_SUBST(AQSIS_TIFF_LDFLAGS)

AC_ARG_WITH(tiff_include,[  --with-tiff-include=DIR path to TIFF includes [[/usr/include]]],
	[if test "$withval" != no; then
		AQSIS_TIFF_CPPFLAGS="-I$withval"
	else
		AQSIS_TIFF_CPPFLAGS="-I/usr/include"
	fi],[AQSIS_TIFF_CPPFLAGS="-I/usr/include"])
	
AC_ARG_WITH(tiff_lib,[  --with-tiff-lib=DIR     path to TIFF libraries [[/usr/lib]]],
	[if test "$withval" != no; then
		AQSIS_TIFF_LDFLAGS="-L$withval -ltiff"
	else
		AQSIS_TIFF_LDFLAGS="-L/usr/local -ltiff"
	fi],[AQSIS_TIFF_LDFLAGS="-L/usr/local -ltiff"])

save_cflags=$CFLAGS
save_libs=$LIBS

CFLAGS=$AQSIS_TIFF_CPPFLAGS
LIBS=$AQSIS_TIFF_LDFLAGS

AC_CHECK_HEADER(tiff.h)
AC_CHECK_LIB(m, pow)
AC_CHECK_LIB(tiff, TIFFOpen)

CFLAGS=$save_cflags
LIBS=$save_libs

if test ! "$ac_cv_header_tiff_h" || test ! "ac_cv_lib_tiff_TIFFOpen" ; then
        AC_MSG_ERROR([*** Please ensure you have libtiff libraries and headers installed ***]);
fi

])

#########################################################
# AQSIS_CHECK_LIBZ

AC_DEFUN([AQSIS_CHECK_LIBZ], [

AC_SUBST(AQSIS_Z_CPPFLAGS)
AC_SUBST(AQSIS_Z_LDFLAGS)

AC_ARG_WITH(zlib_include,[  --with-libz-include=DIR path to ZLIB includes [[/usr/include]]],
	[if test "$withval" != no; then
		AQSIS_Z_CPPFLAGS="-I$withval"
	else
		AQSIS_Z_CPPFLAGS=-I/usr/include
	fi],[AQSIS_Z_CPPFLAGS=-I/usr/include])

AC_ARG_WITH(zlib_lib,[  --with-libz-lib=DIR     path to ZLIB libraries [[/usr/lib]]],
	[if test "$withval" != no; then
		AQSIS_Z_LDFLAGS="-L$withval -lz" 
	else
		AQSIS_Z_LDFLAGS="-L/usr/lib -lz" 
	fi],[AQSIS_Z_LDFLAGS="-L/usr/lib -lz"])

save_cflags=$CFLAGS
save_libs=$LIBS

CFLAGS=$AQSIS_Z_CPPFLAGS
LIBS=$AQSIS_Z_LDFLAGS

AC_CHECK_HEADER(zlib.h)
AC_CHECK_LIB(z, gzopen)

CFLAGS=$save_cflags
LIBS=$save_libs

if test ! "$ac_cv_header_zlib_h" || test ! "ac_cv_lib_z_gzopen" ; then
        AC_MSG_ERROR([*** Please ensure you have zlib libraries and headers  installed ***]);
fi

])

########################################################
# AQSIS_WITH_DEBUGGING

AC_DEFUN([AQSIS_WITH_DEBUGGING], [

AC_ARG_WITH(debug, [  --with-debugging        compile with debugging [[default=no]]], [debug=$withval], [debug=no])

if test x$debug = xyes; then
	AC_DEFINE(_DEBUG, [], "Enable additional debug code.")
	AQSIS_DEBUG_CFLAGS="-g -Wall" 
	AQSIS_OPT_CFLAGS="-O0"
	AC_MSG_NOTICE([Debugging enabled])
else
	AC_DEFINE(NDEBUG, [], "Disable additional debug code.")
	AC_MSG_NOTICE([Debugging disabled])
fi
  
])

########################################################
# AQSIS_WITH_OPENEXR

AC_DEFUN([AQSIS_WITH_OPENEXR], [
AC_SUBST(AQSIS_OPENEXR_CFLAGS)
AC_SUBST(AQSIS_OPENEXR_LIBS)
AC_SUBST(AQSIS_OPENEXR_SUBDIR)
AC_ARG_WITH(openexr, [  --with-openexr          build OpenEXR plug-ins [[no]]],[],[withval=no])
if test x$withval != xno; then
	AC_MSG_CHECKING([for OpenEXR])
	if pkg-config --exists OpenEXR >/dev/null 2>/dev/null; then
		AQSIS_OPENEXR_VERSION=`pkg-config --modversion OpenEXR`
		AQSIS_OPENEXR_CFLAGS=`pkg-config --cflags OpenEXR`
		AQSIS_OPENEXR_LIBS=`pkg-config --libs OpenEXR`
		AQSIS_OPENEXR_SUBDIR="exr2tif"
		AC_MSG_RESULT([found version $AQSIS_OPENEXR_VERSION])
	else
		AC_MSG_ERROR([couldn't find OpenEXR library ... if you don't require OpenEXR support, re-run configure with the --without-openexr option])
	fi
fi
])

########################################################
# AQSIS_WITH_PLUGINS

AC_DEFUN([AQSIS_WITH_PLUGINS], [
AC_SUBST(AQSIS_PLUGIN_LDFLAGS)
AC_SUBST(AQSIS_PLUGIN_SUBDIR)
AC_ARG_WITH(plugins, [  --with-plugins          compile with plugin support [[default=yes]]], [plugins=$withval], [plugins=yes])

if test x$plugins = xyes; then
	AC_DEFINE(PLUGINS, , [Support dynamically loaded plugins])
	AQSIS_PLUGIN_LDFLAGS="-ldl"
	AQSIS_PLUGIN_SUBDIR="plugins"
	AC_MSG_NOTICE([Plugins enabled])
else
	AC_MSG_NOTICE([Plugins disabled])
fi

])

########################################################
# AQSIS_WITH_PROFILING

AC_DEFUN([AQSIS_WITH_PROFILING], [

AC_ARG_WITH(profiling, [  --with-profiling        compile with profiling [[default=no]]], [profile=$withval], [profile=no])

dnl NOTE: Many systems do not have libc compiled with profiling.  We
dnl       should add a test for this library before including it in
dnl       the libs, otherwise configure fails with some strange
dnl       warning about the compiler not being able to produce
dnl       executables!!!
dnl Default to O2 optimisations for profiling.  It typically makes sense to
dnl profile optimised rather than non-optimised code, because the results
dnl can be different, and the end-user normally runs optimised code.

if test x$profile = xyes; then
dnl We need to disable shared libraries for profiling
dnl but we pass --export-dynamic so that we can still use dlopen
	AC_DISABLE_SHARED
	AQSIS_PROFILE_CFLAGS="-g -Wall -pg -ftest-coverage -fprofile-arcs -O2 -static -export-dynamic"
	AQSIS_PROFILE_LDFLAGS="-pg -static -export-dynamic"
	AC_MSG_NOTICE([Profiling enabled])
else
	AC_MSG_NOTICE([Profiling disabled])
fi
  
dnl AM_CONDITIONAL(PROFILE, test x$profile = xyes)

])

########################################################
# AQSIS_WITH_XSLTPROC

AC_DEFUN([AQSIS_WITH_XSLTPROC], [

AC_CHECK_PROG(have_xsltproc, xsltproc, yes, no)
if test x$have_xsltproc = "xno" ; then
	AC_MSG_WARN([*** Could not find xsltproc, you will need it if you edit any of the xml or xsl files ***]);
fi

])

