#!/usr/bin/perl -w

# apt-cacher-cleanup.pl
# Script to clean the cache for the Apt-cacher package caching system.
#
# Copyright (C) 2005, Eduard Bloch <blade@debian.org>
# Copyright (C) 2002-03, Jonathan Oxer <jon@debian.org>
# Portions  (C) 2002, Jacob Lundberg <jacob@chaos2.org>
# Distributed under the terms of the GNU Public Licence (GPL).


# add one argument like 1 to make it verbose

# do locking, not loosing files because someone redownloaded the index files
# right then
# use IO::Handle;

use strict;
use Cwd 'abs_path';

use Fcntl ':flock';
use IO::Handle;
use POSIX;
use Getopt::Long qw(:config no_ignore_case bundling pass_through);

my $configfile = '/etc/apt-cacher/apt-cacher.conf';
my $nice_mode=0;
my $verbose=0;
my $help;
my $force;

my %options = (
    "h|help" => \$help,
    "n|nice-mode=s"     => \$nice_mode,
    "v|verbose"           => \$verbose,
    "f|force"           => \$force,
    "c|config-file=s"        => \$configfile
);


&help unless ( GetOptions(%options));
&help if ($help);

$configfile=abs_path($configfile);

sub help {
    die <<EOM
    Usage: $0 [ -n ] [ -v ] [ -c configfile ]
    -n : nice mode, refresh index files first, then renice to 20 and continue
    -v : verbose mode
    -f : force executing, disable santity checks
EOM
    ;
}

sub printmsg {
   print @_ if $verbose;
}

#use strict;
#############################################################################
### configuration ###########################################################
# Include the library for the config file parser
require '/usr/share/apt-cacher/apt-cacher-lib.pl';

# Read in the config file and set the necessary variables

my $configref;
eval {
        $configref = read_config($configfile);
};
my %config = %$configref;

# not sure what to do if we can't read the config file...
die "Could not read config file: $@" if $@;

# check whether we're actually meant to clean the cache
if ( $config{clean_cache} ne 1 ) {
	exit 0;
}

#############################################################################

my $refresh=1;

my %valid;

### Preparation of the package lists ########################################

chdir "$config{cache_dir}/packages" && -w "." || die "Could not enter the cache dir";

if($> == 0 && !$config{user} && !$force) {
    die "Running $0 as root\nand no effective user has been specified. Aborting.\nPlease set the effective user in $configfile\n";
}

sub get {
    my ($path_info, $filename) = @_;

    my $fh;
    #print "| /usr/share/apt-cacher/apt-cacher.pl -i -c $configfile >/dev/null";
    open($fh, "| REMOTE_ADDR=CLEANUPREFRESH /usr/share/apt-cacher/apt-cacher -i -c $configfile >/dev/null");
    printmsg "GET $path_info\n";
    #printmsg("REMOTE_ADDR=CLEANUPREFRESH /usr/share/apt-cacher/apt-cacher -i -c $configfile >/dev/null\n");
    print $fh "GET $path_info\r\nConnection: Close\r\n\r\n";
    close($fh);
    if($? && ! $force) {
        die "Unable to update $path_info . Network problems?\nRun $0 with -v to get more details.\nCleanup aborted.\n";
    }
}



my @ifiles=(<*es.gz>, <*es.bz2>, <*es>);
for (@ifiles) {
   printmsg "D: $_\n";

   # preserve the index files
   $valid{$_}=1;

   # now refresh them, unless disabled by the setting above
   if($refresh) {
      # if the path is stored there, better use that
      if(-s "../private/$_.complete") {
         open(my $tmp, "../private/$_.complete");
         my $url=<$tmp>;
         &get($url);
         close $tmp;
      }
      else {
         my $tmp=$_;
         $tmp=~s/^/\//;
         $tmp=~s/_/\//g;
         &get($tmp);
      }
   }
}

setpriority 0, 0, 20 if $nice_mode;

# use the list of config files we already know
for my $file (@ifiles) { 
    printmsg "R: $file\n"; 
    extract_sums($file, \%valid) || die("Error processing $file in $config{cache_dir}/packages, cleanup stopped\n");
}

printmsg "Found ".scalar (keys %valid)." valid file entries\n";
#print join("\n",keys %valid);

for(<*.deb>, <*.udeb>, <*.bz2>, <*.gz>, <*.dsc>) {
    # should affect source packages but not index files which are added to the
    # valid list above
    if(! defined($valid{$_})) {
        unlink $_, "../headers/$_", "../private/$_.complete";
        printmsg "Removing source: $_ and company...\n";
    }
}

# similar thing for possibly remaining cruft
chdir "$config{cache_dir}/headers" && -w "." || die "Could not enter the cache dir";

# headers for previosly expired files
for(<*.deb>, <*.bz2>, <*.gz>, <*.dsc>) {
   if(! defined($valid{$_})) {
      unlink $_, "../private/$_.complete";
      printmsg "Removing expired headers: $_ and company...\n";
   }
}

chdir "$config{cache_dir}/private" && -w "." || die "Could not enter the cache dir";
for(<*.deb.complete>, <*.bz2.complete>, <*.gz.complete>, <*.dsc.complete>) {
   s/.complete$//;
   if(! defined($valid{$_})) {
      printmsg "Removing: $_.complete\n";
      unlink "$_.complete";
   }
}

# last step, kill some zombies

my $now = time();
for(<*.notify>) {
    my @info = stat($_);
    # even the largest package should be downloadable in two days or so
    if(int(($now - $info[9])/3600) > 48) {
        printmsg "Removing orphaned notify file: $_\n";
        unlink $_;
    }
}

#define_global_lockfile("$config{cache_dir}/exlock");
#&set_global_lock(": cleanup zombies");

chdir "$config{cache_dir}/packages";

for(<*>) {
    # must be empty and not complete and beeing downloaded right now
    if(!-s $_) {
        my $fromfile;
        open($fromfile, $_);
        if (flock($fromfile, LOCK_EX|LOCK_NB)) {
            # double-check, may have changed while locking
            if(!-s $_) {
                printmsg "Removing zombie files: $_ and company...\n";
                unlink $_, "../headers/$_", "../private/$_.complete";
                flock($fromfile, LOCK_UN);
            }
        }
    }
}

