// ---------------------------------------------------------------------------
// - Socket.hpp                                                              -
// - afnix:net module - socket class definition                              -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2011 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_SOCKET_HPP
#define  AFNIX_SOCKET_HPP

#ifndef  AFNIX_ADDRESS_HPP
#include "Address.hpp"
#endif

#ifndef  AFNIX_INPUTBUFFER_HPP
#include "InputBuffer.hpp"
#endif

#ifndef  AFNIX_INPUTTIMEOPUT_HPP
#include "InputTimeout.hpp"
#endif

#ifndef  AFNIX_OUTPUTSTREAM_HPP
#include "OutputStream.hpp"
#endif

namespace afnix {

  /// The Socket class is a base class for the socket server and socket
  /// client objects. A socket is composed of an address, a port number 
  /// and a stream descriptor. 
  /// @author amaury darsch

  class Socket : public InputBuffer, public InputTimeout, public OutputStream {
  public:
    /// the socket options
    enum t_option {
      SOCK_REUSEADDR = 0,  // enable address reuse
      SOCK_BROADCAST = 1,  // enable packet broadcast
      SOCK_DONTROUTE = 2,  // bypass routing table
      SOCK_KEEPALIVE = 3,  // set connection test
      SOCK_LINGER    = 4,  // set linger time
      SOCK_RCVSIZE   = 5,  // receive buffer size
      SOCK_SNDSIZE   = 6,  // send buffer size
      SOCK_HOPLIMIT  = 7,  // set hop limit
      SOCK_MCASTLOOP = 8,  // multicast use lopback
      SOCK_MCASTHOP  = 9, // multicast hop limit
      SOCK_MAXSEG    = 10, // tcp maximum segment size
      SOCK_NODELAY   = 11  // disable naggle algorithm
    };

  protected:
    /// the socket descriptor
    int d_sid;

  public:
    /// create a default socket
    Socket (void);

    /// create a socket by id
    /// @param sid  the socket id
    Socket (const int sid);

    /// destroy this socket
    ~Socket (void);

    /// @return the class name
    String repr (void) const;

    /// @return the stream descriptor
    int getsid (void) const;

    /// @return true if we have an ipv6 socket
    virtual bool isipv6 (void) const;

    /// @return true if we can broadcast messages
    virtual bool isbcast (void) const;

    /// @return true if the socket is open
    virtual bool isopen (void) const;

    /// close this socket
    virtual bool close (void);

    /// force the socket to close
    virtual bool shutdown (void);

    /// shutdown this socket. If the mode is false, further receive is not
    /// possible - if the mode is tue, further send is not possible
    /// @param mode the shutdown mode
    virtual bool shutdown (const bool mode);

    /// set a socket option
    /// @param opt  the socket option
    /// @param flag the flag to set
    virtual bool setopt (t_option opt, bool flag);

    /// set a socket option with a value
    /// @param opt  the socket option
    /// @param flag the flag to set
    /// @param val  the value to set
    virtual bool setopt (t_option opt, bool val, long val1);

    /// connect this socket by port and address
    /// @param port the port use
    /// @apram addr the address to use
    virtual bool connect (t_word port, const Address& addr);

    /// connect this socket by port to an aliases address
    /// @param port the port use
    /// @apram addr the address to use
    /// @apram alsf the alias address flag
    virtual bool connect (t_word port, const Address& addr, const bool alsf);

    /// bind this socket with a port
    /// @param port the port to bind
    virtual  bool bind (t_word port);

    /// bind this socket with a port and an address
    /// @param port the port to bind
    /// @param addr the addree to bind
    virtual bool bind (t_word port, const Address& addr);

    /// @return the socket address
    virtual Address* getsockaddr (void) const;

    /// @return the socket port
    virtual t_word getsockport (void) const;

    /// @return the socket authority
    virtual String getsockauth (void) const;

    /// @return the peer address
    virtual Address* getpeeraddr (void) const;

    /// @return the peer port
    virtual t_word getpeerport (void) const;

    /// @return the peer authority
    virtual String getpeerauth (void) const;

  private:
    // make the copy constructor private
    Socket (const Socket&);
    // make the assignment operator private
    Socket& operator = (const Socket&);

  public:
    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// evaluate an object data member
    /// @param robj  robj the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to evaluate
    static Object* meval (Runnable* robj, Nameset* nset, const long quark);

    /// apply this object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
                   Vector* argv);
  };
}

#endif
