/*
   Project: UL

   Copyright (C) 2005 Michael Johnston & Jordi Villa-Freixa

   Author: Michael Johnston

   Created: 2005-05-23 14:11:56 +0200 by michael johnston

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/

#ifndef _ULPROCESS_H_
#define _ULPROCESS_H_

#include <math.h>
#include <Foundation/Foundation.h>
#include <AdunKernel/AdunModelObject.h>
#include <AdunKernel/RemoteProcessAccess.h>
#include "ULFramework/ULServerInterface.h"
#include "ULFramework/ULMatrix.h"
#include "ULFramework/ULSimulation.h"
#include "ULFramework/ULIOManager.h"
#include "ULFramework/ULDatabaseInterface.h"

/**
Protocol contain methods that can be called on ULProcess instances
using distributed objects
*/

@protocol ULClientInterface
- (bycopy NSMutableDictionary*) transmitOptionsForProcess: (int) pid;
/**
Sets the results produced by the simulation controller to \e array which must be contain AdDataSet instances.
This method is sent by the server when the simulation exits.
*/
- (void) setControllerResults: (bycopy NSArray*) array;
- (void) processDidTerminate: (bycopy NSError*) error;
- (bycopy NSString*) controllerOutputDirectory;
- (bycopy NSString*) simulationOutputDirectory;
@end

/**
ULProcess instances represent a running simulation process. They contain information on the data required
to run the simulation i.e.  systems, options, host, output directories etc  and they also act as
the client interface to the userland for AdunServers.
\ingroup model
*/

@interface ULProcess : AdModelObject <ULClientInterface>
{
	id systems;		//!< An array of ULSystem instances containing the system to be simulated
	id options;		//!< A ULOptions instance giving the simulation parameters
	id host;		//!< The host on which the process is to be lauched
	id status;
	id elapsed;
	id simulationData;
	int processIdentifier;
	NSArray* dataSets;	//!< Data sets returned by the process when it finished
	NSString* simulationDataDirectory;
	NSString* controllerOutputDirectory;
}

/**
Returns a process initialised with the given system and options. The process id
is generated on creation
\param sys A ULSystem instance
\param opt A ULOptions instance
*/

+(id) processWithSystem: (id) sys options: (id) opt;
/**
Returns a process initialised with the given systems and options. The process id
is generated on creation
\param sys An array of ULSystem objects
\param opt A ULOptions instance
*/
+(id) processWithSystems: (id) sys options: (id) opt;
- (id) initWithSystems: (id) sys options: (id) opt;
- (int) processIdentifier;
- (void) setProcessIdentifier: (int) number;
- (id) length;
- (NSString*) processHost;
- (void) setProcessHost: (NSString*) value;
- (id) systems;
- (id) options;
- (NSString*) processStatus;
- (void) setProcessStatus: (NSString*) value;
- (id) started;
- (void) setLength: (id) value;
- (void) setStarted: (NSDate*) value;
- (void) processWillStart;
- (id) simulationData;
- (NSArray*) controllerResults;
@end

#endif // _ULPROCESS_H_

