/*
   Project: UL

   Copyright (C) 2006 Michael Johnston & Jordi Villa-Freixa

   Author: Michael Johnston

   Created: 2006-07-13 12:36:00 +0100 by michael johnston

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the 
   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/
#ifndef _ULFILESYSTEM_SIMULATIONSTORAGE_
#define _ULFILESYSTEM_SIMULATIONSTORAGE_
#include <Foundation/Foundation.h>

/**
Class for managing storage and retrieval of simulation data on the file system.
If this class is archived on decoding its owner should check if the data store the
object provides access to is still accesible/valid using isAccesible e.g. the stored data
may have been deleted or removed. 
*/

@interface ULFileSystemSimulationStorage: NSObject 
{
	BOOL isAccessible;
	BOOL isTemporary;
	NSFileManager* fileManager;
	NSString* storagePath;
	NSError* accessError;
}
/**
Designated initialiser.
Creates a ULFileSystemSimulationStorage instance that
provides access to the simulation data at path \e path
which is stored in the file system.
If the stored data cannot be accessed upon return anError points
to an NSError object describing the problem.
*/
- (id) initForReadingSimulationDataAtPath: (id) path error: (NSError**) anError;
/**
As initForReadingSimulationDataAtPath:error: with \e anError equal to NULL
*/
- (id) initForReadingSimulationDataAtPath: (id) path;
/**
Creates a new storage directory at path
*/
- (id) initStorageForSimulationAtPath: (NSString*) path;
/**
Returns the trajectory data for the simulation or
nil if there is none
*/
- (NSData*) trajectoryData;
/**
Returns the energy data for the simulation or
nil if there is none
*/
- (NSData*) energyData;
/**
Returns the system data for the simulation or
nil if there is none
*/
- (NSData*) systemData;
/**
Returns the restart data for the simulation or
nil if there is none
*/
- (NSData*) restartData;
/**
Returns YES if the data  is accesible.
NO otherwise. i.e. if the dataStorage
was removed for some reason.
*/
- (BOOL) isAccessible;
/**
Returns an NSError object describing the reason
why the data store cannot be accessed.
*/
- (NSError*) accessError;
/**
Returns the path to the location where the
data is stored.
*/
- (NSString*) storagePath;
/**
Deletes the stored data. Immediately sets isAccesible to NO
If the data is not accesible returns NO. If the data cannot be
deleted because it is write protected it will also return NO however in this case 
isAccesible will still return YES.
*/
- (BOOL) destroyStoredData;
/**
Sets weather the object should destroy the stored data when
it is released. Value is NO by default.
*/
- (void) setIsTemporary: (BOOL) value;
/**
Returns YES if the object will destroy the stored data when
it is released, NO otherwise.
*/
- (BOOL) isTemporary;

@end

#endif
