#!/usr/bin/ruby

$:<<'../lib'
require 'test/unit'
require 'activeldap'
require 'openssl'

# Load all demo sub-objects
Dir['../examples/objects/*.rb'].each {|t| require t}


class UserTest < Test::Unit::TestCase
  # Connect with password
  def test_1_binding_connect
    assert_nothing_raised() do
      pwb = Proc.new { 'password' }
      ActiveLDAP::Base.close
      ActiveLDAP::Base.connect(
        :user => 'root',
        :host => '127.0.0.1',
        :port => 389,
        :base => 'dc=localdomain',
        :password_block => pwb, 
        :allow_anonymous => false)
    end
  end

  # This adds all required attributes and writes
  def test_2_add
    user = User.new('testuser')

    assert_equal(false, user.exists?, 
      "testuser must not exist in LDAP prior to testing")
    
    assert_equal(['posixAccount'], user.objectClass, 
      "Class User's ldap_mapping should specify " + 
      "['posixAccount']. This was not returned.")

    user.objectClass << 'posixAccount' << 'shadowAccount' << 'person' << 'inetOrgPerson' << 'organizationalPerson'

    user.cn = 'Test User (Default Language)'
    assert_equal(['Test User (Default Language)'], user.cn, 
      'user.cn should have returned "Test User (Default Language)"')

    # test not_array
    assert_equal('Test User (Default Language)', user.cn(true),
      'user.cn should have returned "Test User (Default Language)"')

    user.cn = {'lang-en-us' => 'Test User (EN-US Language)'}
    assert_raise(ActiveLDAP::AttributeEmpty) { user.validate }
    # Test subtypes
    assert_equal([{'lang-en-us' => 'Test User (EN-US Language)'}], user.cn,
      'user.cn should match')

    user.cn = ['Test User (Default Language)', {'lang-en-us' => ['Test User (EN-US Language)', 'Foo']}]
    assert_raise(ActiveLDAP::AttributeEmpty) { user.validate }
    # Test multiple entries with subtypes
    assert_equal(['Test User (Default Language)', {'lang-en-us' => ['Test User (EN-US Language)', 'Foo']}],
       user.cn,
       'This should have returned an array of a normal cn and a lang-en-us cn.')

    assert_raise(ActiveLDAP::AttributeEmpty, 'This should have raised an error!') { user.validate }

    user.uidNumber = 9000
    # Test to_s on Fixnums
    assert_equal(['9000'], user.uidNumber, 'uidNumber did not get set correctly.')
    assert_raise(ActiveLDAP::AttributeEmpty) { user.validate }
    assert_equal('9000', user.uidNumber(true), 'uidNumber did not get changed to an array by #validate().')

    user.gidNumber = 9000
    # Test to_s on Fixnums
    assert_equal(['9000'], user.gidNumber(), 'gidNumber did not get set correctly.')
    assert_raise(ActiveLDAP::AttributeEmpty) { user.validate }
    assert_equal('9000', user.gidNumber(true), 'not_array argument failed')

    user.homeDirectory = '/home/foo'
    # just for sanity's sake
    assert_equal(['/home/foo'], user.homeDirectory, 'This should be /home/foo.')
    assert_raise(ActiveLDAP::AttributeEmpty) { user.validate }
    assert_equal('/home/foo', user.homeDirectory(true), 'This should be ["/home/foo"].')


    user.sn = ['User']

    user.userCertificate = File.read('../examples/example.der')
    user.jpegPhoto = File.read('../examples/example.jpg')
    user.validate

    assert_equal(true, ActiveLDAP::Base.schema.binary?('jpegPhoto'),
     'jpegPhoto is binary?')

    assert_equal(true, ActiveLDAP::Base.schema.binary?('userCertificate'),
     'userCertificate is binary?')

    assert_nothing_raised() { user.write }
  end


  # This tests the reload of a binary_required type
  def test_3_binary_required
    user = User.new('testuser')
    # validate add 
    assert_equal([{'binary' => [File.read('../examples/example.der')]}], 
      user.userCertificate, 
      'This should have been forced to be a binary subtype.')

    # now test modify 
    user.userCertificate = ''
    assert_nothing_raised() { user.write }

    user.userCertificate = File.read('../examples/example.der')
    assert_nothing_raised() { user.write }

    # validate modify
    user = User.new('testuser')
    assert_equal([{'binary' => [File.read('../examples/example.der')]}], 
      user.userCertificate, 
      'This should have been forced to be a binary subtype.')

    assert_equal(OpenSSL::X509::Certificate.new(File.read('../examples/example.der')).subject.to_s, 
      OpenSSL::X509::Certificate.new(user.userCertificate[0]['binary'][0]).subject.to_s,
      'Cert must parse correctly still')
  end

  # This tests the reload of a binary type (not forced!)
  def test_4_binary
    # reload and see what happens
    user = User.new('testuser')
    assert_equal([File.read('../examples/example.jpg')], user.jpegPhoto, 
      'This should have been equal to File.read(\'../examples/example.jpg\')')

    # now test modify
    user.jpegPhoto = ''
    assert_nothing_raised() { user.write }
    user.jpegPhoto = File.read('../examples/example.jpg')
    assert_nothing_raised() { user.write }

    # now validate modify
    user = User.new('testuser')
    assert_equal([File.read('../examples/example.jpg')], user.jpegPhoto, 
      'This should have been equal to File.read(\'../examples/example.jpg\')')

  end

  # This tests the removal of a objectclass
  def test_5_remove_oc
    user = User.new('testuser')
    assert_equal([], user.shadowMax, 'Should get the default nil value for shadowMax')

    # Remove shadowAccount
    user.objectClass.delete('shadowAccount')
    assert_nothing_raised() { user.validate }
    assert_nothing_raised() { user.write }

    assert_raise(NoMethodError, 'shadowMax should not be defined anymore' ) { user.shadowMax }
  end


  # Just modify a few random attributes
  def test_6_modify_with_subtypes
    user = User.new('testuser')

    user.cn = ['Test User', {'lang-en-us' => ['Test User', 'wad']}]
    assert_nothing_raised() { user.write }

    user = User.new('testuser')
    assert_equal(['Test User', {'lang-en-us' => ['Test User', 'wad']}], user.cn, 'Making sure a modify with mixed subtypes works')
  end

  # This tests some invalid input handling
  def test_7_invalid_input
  end

  # This tests deletion
  def test_8_delete
    user = User.new('testuser')
    user.delete
    assert_equal(user.exists?, false, 'user should no longer exist')
  end

end
