/* ------------------------------------------------------------------------
 * ImageImpl.cc
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-09-18 by Steve Houston.
 *
 * Copyright (c) 2000, 2001 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000 Steve Houston <steve@3dwm.org>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <stdio.h>
#include <stdlib.h>

// -- 3Dwm Includes
#include "Celsius/debug.hh"
#include "Celsius/Mutex.hh"
#include "Celsius/Guard.hh"
#include "Polhem/ImageImpl.hh"

using namespace Nobel;

// -- Code Segment

/**
 * ImageImpl - handles image loading.
 **/
ImageImpl::ImageImpl() 
    : _width(0), _height(0)
{
    // empty
}

ImageImpl::~ImageImpl()
{
    // empty
}

void ImageImpl::loadRaw(const BinaryData &raw_data, CORBA::Long width,
			CORBA::Long height, Nobel::PixelType pixelFormat)
{
    Guard<Mutex> guard(_mutex);
    
    // Update internal image data
    _binData = raw_data;
    _width = width;
    _height = height;
    _pixelType = pixelFormat;
}

// Free the memory used by the current image
void ImageImpl::free()
{
    // empty
}

/** 
 * TextureImpl - extends ImageImpl to handle loading of texture images.
 **/
TextureImpl::TextureImpl() 
    : _isReady(false)
{
    // empty
}

TextureImpl::~TextureImpl()
{
    // empty
}

CORBA::Boolean TextureImpl::init(Renderer_ptr r)
{
    Guard<Mutex> guard(_mutex);
    
    // TODO: ensure _width, _height etc. are initialized
    if (!r->genTexture(_width, _height, _pixelType, _binData, 
		       _texName))
	// There was a problem initialising the texture.
	// Return without enabling texturing.
	return false;
    
    // Texture is initialized, return with success
    _isReady = true;
    return true;
}

void TextureImpl::apply(Renderer_ptr r)
{
    Guard<Mutex> guard(_mutex);

    // Apply the texture to the renderer
    r->applyTexture(_texName);
}

void TextureImpl::free()
{
    Guard<Mutex> guard(_mutex);
    // TODO: somehow we need to get a message to the renderer to issue
    // a glDeleteTextures() on our texturename.
    ImageImpl::free();
}

CORBA::Boolean TextureImpl::isReady()
{
    return _isReady;
}

void TextureImpl::setName(Nobel::Tag name)
{
    Guard<Mutex> guard(_mutex);
    _texName = name;
}

Nobel::Tag TextureImpl::getName()
{
    return _texName;
}

/**
 * ProceduralTextureImpl
 */
ProceduralTextureImpl::ProceduralTextureImpl()
{
    // empty
}

ProceduralTextureImpl::~ProceduralTextureImpl()
{
    // empty
}

// initTexture - called before the texture is used. Returns 0 if
// success, TEX_SPLIT if the texture had to be split up to work on
// current hardware TEX_INIT_FAIL if unable to initialize texture.

CORBA::Boolean ProceduralTextureImpl::init(Renderer_ptr r)
{
    Guard<Mutex> guard(_mutex);

    // Make sure we have a texture generator
    if (CORBA::is_nil(_texGen))
	return false;

    // Get the initial texture
    BinaryData_var tex = _texGen->getTextureData();
    if (r->genTexture(_width, _height, _pixelType, tex.in(), _texName) == 0)
	return false;
    _isReady = true;
    
    return true;
}

void ProceduralTextureImpl::apply(Renderer_ptr r)
{ 
    Guard<Mutex> guard(_mutex);

    // Sanity check
    if (CORBA::is_nil(_texGen))
	return;

    BinaryData_var tex;
    Rectangle rect;
    
    // Retrieve the update from the texture generator and pass it
    // along to the renderer.
    if (_texGen->getUpdate(tex.out(), rect))
	r->updateTexture(rect, _pixelType, tex.in(), _texName);
    
    // Now apply the texture to the renderer
    r->applyTexture(_texName);
}

void ProceduralTextureImpl::setTextureGenerator(TextureGenerator_ptr texGen)
{
    Guard<Mutex> guard(_mutex);

    // Sanity check
    if (CORBA::is_nil(texGen))
	return;

    // Update values
    _texGen = TextureGenerator::_duplicate(texGen);
    _width = _texGen->getWidth();
    _height = _texGen->getHeight();
    _pixelType = _texGen->getPixelType();
}
