/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.plugin.gnutella.gui.table;

import xnap.gui.table.AbstractDynamicTableModel;
import xnap.gui.table.Column;
import xnap.gui.table.FilesizeCellRenderer;
import xnap.gui.table.StringCellRenderer;
import xnap.gui.table.TimeCellRenderer;
import xnap.net.event.*;
import xnap.plugin.gnutella.gui.table.CenteredCellRenderer;
import xnap.plugin.gnutella.net.*;
import xnap.plugin.gnutella.util.GnuPreferences;
import xnap.util.*;
import xnap.util.event.*;

import java.awt.event.*;
import java.beans.*;
import java.util.Vector;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;

public class ConnectionsTableModel extends AbstractDynamicTableModel
    implements ListListener, StatusChangeListener 
{

    //--- Constant(s) ---

    public static final int COL_HOST      = 0;
    public static final int COL_FILECOUNT = 1;
    public static final int COL_FILESIZE  = 2;
    public static final int COL_MESSAGES  = 3;
    public static final int COL_STATUS    = 4;
    public static final int COL_UPTIME    = 5;

    public static final int UPDATE_INTERVAL = 1000;

    //--- Data Field(s) ---

    protected Column columns[] = new Column[] {
	new Column("Host", String.class),
	new Column("Files", Integer.class),
	new Column("Total (GB)", Integer.class, new FilesizeCellRenderer()),
	new Column("Messages (I/O)", String.class, new CenteredCellRenderer()),
	new Column("Status", String.class, new StringCellRenderer()),
	new Column("Uptime", Integer.class, new TimeCellRenderer()),
    };
    
    private EventVector data;

    //--- Constructor(s) ---

    public ConnectionsTableModel(EventVector data)
    {
	super("Connections", GnuPreferences.getInstance());
	this.data = data;
	data.addListListener(this);
	setColumns(columns);

	Timer updateTimer = new Timer(UPDATE_INTERVAL, new Updater());
	updateTimer.start();
    }

    // --- Method(s) ---

    public void elementAdded(ListEvent e)
    {
	Servent s = (Servent)e.getElement();
	s.addStatusChangeListener(this);
	fireTableRowsInserted(e.getIndex(), e.getIndex());
    }

    public void elementRemoved(ListEvent e)
    {
	Servent s = (Servent)e.getElement();
	s.removeStatusChangeListener(ConnectionsTableModel.this);
	fireTableRowsDeleted(e.getIndex(), e.getIndex());
    }

    public boolean isCellEditable(int row, int column)
    {
	return false;
    }

    public Servent get(int i) 
    {
	return (Servent)data.get(mapToIndex(i));
    }

    public int getRowCount() 
    { 
	return data.size(); 
    }

    public Object get(int row, int col) 
    { 
	Servent s;
	try {
	    s = (Servent) data.get(row);
	}
	catch (ArrayIndexOutOfBoundsException ae) {
	    fireTableRowsDeleted(row, row);
	    return null;
	}
	
	switch (col) {
	case COL_HOST:
	    return s.getAddress() + ":" + s.getPort();
	case COL_FILECOUNT:
	    return new Integer(s.getNumberOfFiles());
	case COL_FILESIZE:
	    return new Integer(s.getNumberOfKBs() * 1024);
	case COL_MESSAGES:
	    if (s.getStatus() == Servent.STATUS_CONNECTED)
		return s.getMessagesReceived() + "/" + s.getMessagesSent();
	    else
		return "";
	case COL_STATUS:
	    return s.getStatusText() + (s.isIncoming() ? " (incoming)" : "");
	case COL_UPTIME:
	    return new Integer((int)s.getUpTime());
	default:
	    return null;
	}
    }

    public void statusChange(StatusChangeEvent e)
    {
	Debug.log("gnutella: status change " 
		  + ((Servent) e.getSource()).getStatusText());

	int i = data.indexOf(e.getSource());
	if (i != -1) {
	    fireTableRowsUpdated(i, i);
	}
    }

    public void updateTable()
    {
	for (int i = 0; i < data.size(); i++) {
	    Servent s = (Servent)data.get(i);
	    if (s.getStatus() == Servent.STATUS_CONNECTED) {
		fireTableRowsUpdated(i, i);
	    }
	}
    }
    
    private class Updater implements ActionListener 
    {
	public void actionPerformed(ActionEvent event)
	{
	    if (data.size() > 0) {
		fireTableRowsUpdated(0, data.size() - 1);
	    }
	}

    }

}
