/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.net;

import xnap.XNap;
import xnap.net.event.*;
import xnap.util.*;

import java.io.File;

public interface ITransferContainer {

    //--- Constant(s) ---

    /**
     * Abort transfer if stalled for this long.
     */
    int TRANSFER_TIMEOUT = 2 * 60 * 1000;

    int STATUS_NOT_STARTED     = 0;
    int STATUS_CONNECTING      = 1;
    int STATUS_LOCALLY_QUEUED  = 2;
    //int STATUS_REMOTELY_QUEUED = 3;
    int STATUS_DOWNLOADING     = 4;
    int STATUS_UPLOADING       = 5;
    int STATUS_SUCCESS         = 6;
    int STATUS_FAILED          = 7;
    //int STATUS_INCOMPLETE      = 8;
    int STATUS_ABORTED         = 9;
    int STATUS_SEARCHING       = 10;
    int STATUS_WAITING         = 11;
    int STATUS_DELETED         = 12;
    int STATUS_ERROR           = 13;

    String[] STATUS_MSGS = {
	"", 
	XNap.tr("connecting") + "...", 
	XNap.tr("locally queued"), 
	XNap.tr("remotelly queued"), 
	XNap.tr("downloading") + "...", 
	XNap.tr("uploading") + "...", 
	XNap.tr("finished"),
	XNap.tr("failed"), 
	XNap.tr("incomplete"), 
	XNap.tr("aborted"), 
	XNap.tr("searching") + "...", 
	XNap.tr("waiting") + "...",
	XNap.tr("deleted"),
	XNap.tr("error"),
    };

    //--- Method(s) ---

    /**
     * Aborts this transfer. The transfer might be started again by 
     * calling <code>start</code>.
     */
    void abort();

    /**
     * Fail transfer.
     */
    void fail(String msg);

    /**
     * Starts this transfer.
     */
    void start();

    /**
     * Returns true if the transfer is finished.
     *
     * @return true, if thread has ended, false if not yet started or still 
     *         running
     */
    boolean isDone();

    /**
     * Returns true if the transfer can be (re)started.
     */
    boolean isResumable();

    /**
     * Returns true if the transfer is finished.
     *
     * @return true, if thread is running
     */
    boolean isRunning();

    /**
     * Notifies the transfer, that is has benn queued.
     */
    void locallyQueued();
    
    /**
     * Returns how many bytes have been transferred since the transfer
     * was started.
     * 
     * @return number of transferred bytes
     */
    long getTotalBytesTransferred();

    /**
     * Returns the averate transfer rate.
     *
     * @return byte / s
     */
    long getAverageRate();

    /**
     * Returns the current transfer rate.
     *
     * @return byte / s
     */
    long getCurrentRate();

    /**
     * Returns the file the transfer is using.
     */
    File getFile();

    /**
     * Returns the filename that should be presented to the user.
     */
    String getFilename();

    /**
     * Returns the final filesize.
     */
    long getFilesize();

    /**
     * Returns the current status.
     */
    int getStatus();

    /**
     * Returns the current status message.
     */
    String getStatusText();

    /**
     * Returns the user.
     */
    IUser getUser(); 

    /**
     * Adds a status listener.
     */
    void addStatusChangeListener(StatusChangeListener l);

    /**
     * Remove a status listener.
     */
    void removeStatusChangeListener(StatusChangeListener l);

}
