/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui;

import xnap.*;
import xnap.util.Debug;

import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import java.io.*;
import java.util.*;
import javax.accessibility.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;
import javax.swing.table.*;

public class WizardDialog extends JDialog
{

    //--- Constant(s) ---

    //--- Data field(s) ---

    protected PreviousAction acPrevious = new PreviousAction();
    protected NextAction acNext = new NextAction();
    protected CancelAction acCancel = new CancelAction();
    protected FinishedAction acFinished = new FinishedAction();
    
    protected PContainer ctPanels;
    protected CardLayout clPanels;

    protected LinkedList panels = new LinkedList();
    protected int selectedIndex = 0;

    protected JButton jbNext;
    protected JButton jbFinished;

    //--- Constructor(s) ---

    public WizardDialog(boolean showCancel)
    {
	initialize(showCancel);
    }

    public WizardDialog()
    {
	this(true);
    }

    //--- Method(s) ---

    private void initialize(boolean showCancel) 
    {
	// center
	clPanels = new CardLayout();
	ctPanels = new PContainer();
	ctPanels.setBorder(new EmptyBorder(5, 5, 5, 5));
	ctPanels.setLayout(clPanels);

	// bottom
	JPanel jpButtons = new JPanel(new FlowLayout(FlowLayout.RIGHT));

	JButton jbPrevious = new JButton(acPrevious);
	jpButtons.add(jbPrevious);

	jbNext = new JButton(acNext);
	jbNext.setHorizontalTextPosition(SwingConstants.LEFT);
	jpButtons.add(jbNext);

	jbFinished = new JButton(acFinished);
	jpButtons.add(jbFinished);

	if (showCancel) {
	    JButton jbCancel = new JButton(acCancel);
	    jpButtons.add(jbCancel);
	}

	setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
	addWindowListener(new CloseListener());

	// set me up
	getContentPane().setLayout(new BorderLayout());
	getContentPane().add(ctPanels, BorderLayout.CENTER);
	getContentPane().add(jpButtons, BorderLayout.SOUTH);
    }

    public void addPanel(AbstractSaveablePanel panel)
    {
	String title = panel.getTitle();
	String desc = panel.getDescription();
	Icon icon = panel.getIcon();
	ctPanels.add(new TitledPanel(title, desc, icon, panel), title);

	panels.add(panel);
	updateActions();
    }

    /**
     * Overwrite this method if you like.
     */
    public boolean canCancel()
    {
	return true;
    }

    public void cancel()
    {
	dispose();
    }

    protected void updateActions()
    {
	acPrevious.setEnabled(selectedIndex > 0);
	acNext.setEnabled(selectedIndex < panels.size() - 1);
	if (acNext.isEnabled()) {
	    jbNext.requestFocus();
	}
	else {
	    jbFinished.requestFocus();
	}
    }

    public class PreviousAction extends AbstractAction {
	
	public PreviousAction()
	{
	    putValue(Action.NAME, XNap.tr("Previous"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("1leftarrow.png"));
            putValue(Action.SHORT_DESCRIPTION,XNap.tr("Go to previous page"));
            putValue(Action.MNEMONIC_KEY, new Integer('P'));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    clPanels.previous(ctPanels);
	    selectedIndex--;
	    updateActions();
        }

    }

    public class NextAction extends AbstractAction {
	
	public NextAction()
	{
	    putValue(Action.NAME, XNap.tr("Next"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("1rightarrow.png"));
            putValue(Action.SHORT_DESCRIPTION, XNap.tr("Go to next page"));
            putValue(Action.MNEMONIC_KEY, new Integer('N'));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    clPanels.next(ctPanels);
	    selectedIndex++;
	    updateActions();
        }

    }

    public class CancelAction extends AbstractAction {
	
	public CancelAction()
	{
	    putValue(Action.NAME, XNap.tr("Cancel"));
            putValue(Action.SHORT_DESCRIPTION,
		    XNap.tr("Close dialog discarding changes"));
            putValue(Action.MNEMONIC_KEY, new Integer('C'));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    if (canCancel()) {
		cancel();
	    }
        }

    }

    public class FinishedAction extends AbstractAction {
	
	public FinishedAction()
	{
	    putValue(Action.NAME, XNap.tr("Finished"));
            putValue(Action.SHORT_DESCRIPTION, 
		     XNap.tr("Close dialog saving changes"));
            putValue(Action.MNEMONIC_KEY, new Integer('F'));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    for (Iterator i = panels.iterator(); i.hasNext();) {
		((AbstractSaveablePanel)i.next()).save();
	    }

	    dispose();
        }

    }

    protected class PContainer extends JPanel
    {
	Dimension maxDim = new Dimension(0, 0);

	public void add(Component comp, Object constraints) 
	{
	    Dimension d = comp.getPreferredSize();
	    if (d.height > maxDim.height) {
		maxDim.height = d.height;
	    }
	    if (d.width > maxDim.width) {
		maxDim.width = d.width;
	    }

	    super.add(comp, constraints);
	}

	public Dimension getPreferredSize()
	{
	    // looks like the height is too small
	    return new Dimension(maxDim.width + 10, maxDim.height + 30);
	}
    }

    private class CloseListener extends WindowAdapter
    {
	public void windowClosing (java.awt.event.WindowEvent evt) {
	    if (canCancel()) {
		cancel();
	    }
	}
    }


}
