/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

package xnap.gui;

import xnap.net.HttpConnection;
import xnap.net.PluginInfoReader;
import xnap.plugin.IPlugin;
import xnap.plugin.PluginInfo;
import xnap.plugin.PluginManager;
import xnap.util.FileHelper;
import xnap.util.VersionParser;
import xnap.*;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.net.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

public class PluginDialog extends JDialog implements ListSelectionListener {

    //--- Constant(s) ---

    //--- Data field(s) ---

    private static PluginDialog me = null;

    private JLabel jlStatus;
    private DefaultListModel dlmPlugins;
    private JList jlPlugins;
    private JLabel jlName;
    private JLabel jlVersion;
    private JLabel jlInstalledVersion;
    private MultiLineLabel jlDescription;

    private InstallAction acInstall = new InstallAction();

    //--- Constructor(s) ---

    private PluginDialog() 
    {
	// status
	jlStatus = new JLabel(" ");
	jlStatus.setBorder(new EmptyBorder(5, 5, 5, 5));

	// plugin list
	dlmPlugins = new DefaultListModel();
	jlPlugins = new JList(dlmPlugins);
	jlPlugins.addListSelectionListener(this);

	// info panel
	JPanel jpInfo = new JPanel(new GridBagLayout());

	GridBagHelper.addLabel(jpInfo, XNap.tr("Name"));
	jlName = new JLabel();
	GridBagHelper.add(jpInfo, jlName);

	GridBagHelper.addLabel(jpInfo, XNap.tr("Version"));
	jlVersion = new JLabel();
	GridBagHelper.add(jpInfo, jlVersion);

	GridBagHelper.addLabel(jpInfo, XNap.tr("Installed"));
	jlInstalledVersion = new JLabel();
	GridBagHelper.add(jpInfo, jlInstalledVersion);

	JPanel jpDescription = new JPanel(new BorderLayout());
	jpDescription.setBorder(new TitledBorder(XNap.tr("Description", 1)));
	jlDescription = new MultiLineLabel();
	jpDescription.add(jlDescription);
	GridBagHelper.add(jpInfo, jpDescription);

	GridBagHelper.addVerticalSpacer(jpInfo);

	JPanel jpInstall = new JPanel(new FlowLayout(FlowLayout.LEFT));
	jpInstall.add(new JButton(acInstall));
	GridBagHelper.add(jpInfo, jpInstall);

	// buttons
        JPanel jpButtons = new JPanel(new FlowLayout(FlowLayout.RIGHT));

	jpButtons.add(new JButton(new CloseAction()));

	// content
        setTitle(XNap.tr("Install Plugins"));

	setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
	addWindowListener(new CloseListener());

	getContentPane().setLayout(new BorderLayout());
	getContentPane().add(jlStatus, BorderLayout.NORTH);
	getContentPane().add(new JScrollPane(jlPlugins), BorderLayout.WEST);
	getContentPane().add(new JScrollPane(jpInfo), BorderLayout.CENTER);
	getContentPane().add(jpButtons, BorderLayout.SOUTH);

        pack();

	Thread t = new Thread(new FetchWorker(), "FetchPluginList");
	t.start();
    }

    //--- Method(s) ---

    public static void showDialog(Component c)
    {
	if (me == null) {
	    me = new PluginDialog();
	    if (c != null) {
		me.setLocationRelativeTo(c);
	    }
	}
	me.show();
    }

    private void setStatus(String newValue)
    {
	jlStatus.setText(newValue);
	jlStatus.setToolTipText(newValue);
    }

    public void valueChanged(ListSelectionEvent e)
    {
	PluginInfo i = (PluginInfo)jlPlugins.getSelectedValue();
	IPlugin p = PluginManager.getInstance().getPluginByName(i.getName());

	jlName.setText(i.getName());
	jlVersion.setText(i.getVersion());
	jlInstalledVersion.setText((p != null) ? p.getVersion() : "");
	jlDescription.setText(i.getDescription());

	if (p != null) {
	    int c = VersionParser.compare(i.getVersion(), p.getVersion());
	    acInstall.setEnabled(c != 0); 
	}
	else {
	    acInstall.setEnabled(true);
	}
    }

    private class InstallAction extends AbstractAction {

        public InstallAction() 
	{
            putValue(Action.NAME, XNap.tr("Download"));
            putValue(Action.SHORT_DESCRIPTION, 
		     XNap.tr("Download selected plugin"));
	    //putValue(Action.SMALL_ICON, XNap.getIcon("eraser.png"));
            putValue(Action.MNEMONIC_KEY, new Integer(KeyEvent.VK_I));

	    this.setEnabled(false);
        }

        public void actionPerformed(ActionEvent event) 
	{
	    PluginInfo i = (PluginInfo)jlPlugins.getSelectedValue();

	    try {
		HttpConnection c = new HttpConnection();
		c.connect(i.getLocation());

		ProgressMonitorInputStream pIn = new ProgressMonitorInputStream
			(PluginDialog.this, "Downloading " + i.getFilename(), 
			 c.getInputStream());
		pIn.getProgressMonitor().setMillisToPopup(0);
		pIn.getProgressMonitor().setMillisToDecideToPopup(0);
		InputStream in = new BufferedInputStream(pIn);
		String f = FileHelper.getHomeDir("jars") + i.getFilename();

		HttpConnection.save(in, f);
	    }
	    catch (IOException e) {
		setStatus(e.getMessage());
	    }
        }

    }

    private class CloseAction extends AbstractAction {

        public CloseAction() 
	{
            putValue(Action.NAME, XNap.tr("Close"));
            putValue(Action.SHORT_DESCRIPTION, XNap.tr("Closes the dialog."));
            putValue(Action.MNEMONIC_KEY, new Integer(KeyEvent.VK_C));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    dispose();
	    me = null;
	}

    }

    private class CloseListener extends WindowAdapter
    {
	public void windowClosing (java.awt.event.WindowEvent evt) 
	{
	    dispose();
	    me = null;
	}
    }

    private class FetchWorker implements Runnable {

	public void run()
	{
	    setStatus(XNap.tr("Getting list..."));

	    try {
		String URL = "http://localhost/plugin.txt";
		PluginInfo[] infos = PluginInfoReader.read(URL);
		
		for (int i = 0; i < infos.length; i++) {
		    dlmPlugins.addElement(infos[i]);
		}
	    }
	    catch (IOException e) {
		setStatus(e.getMessage());
		return;
	    }

	    setStatus(dlmPlugins.size() + XNap.tr("Plugins available"));
	}
    }

}
