/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui;

import xnap.*;
import xnap.gui.*;
import xnap.gui.event.DoubleClickListener;
import xnap.gui.event.PopupListener;
import xnap.gui.table.*;
import xnap.io.*;
import xnap.net.*;
import xnap.net.event.ChannelEvent;
import xnap.util.*;
import xnap.util.event.ListEvent;
import xnap.util.event.ListListener;

import java.awt.*;
import java.awt.event.*;
import java.beans.*;
import java.io.*;
import java.lang.Math;
import java.util.*;
import java.util.ArrayList;
import java.util.ResourceBundle;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.table.*;

public class ChatGlobalSubPanel extends AbstractPanel implements ListListener
{

    //--- Constant(s) ---

    public static final int UPDATE_INTERVAL = 1000;

    // --- Data Field(s) ---

    protected ConsolePane cpChat;
    protected ChannelTableModel ctm;
    protected DefaultComboBoxModel dcbm;
    protected JComboBox jcbServers;
    protected JTable jta;
    
    private JSplitPane jspH;

    private CreateChannelAction acCreateChannel = new CreateChannelAction();
    private JoinChannelAction acJoinChannel = new JoinChannelAction();
    private UpdateChannelsAction acUpdateChannels = new UpdateChannelsAction(); 
    protected javax.swing.Timer updateTimer;

    //--- Constructor(s) ---

    public ChatGlobalSubPanel() 
    {
	initialize();

	updateTimer 
	    = new javax.swing.Timer(UPDATE_INTERVAL, new Updater());
	updateTimer.start();

	EventVector servers = ChatManager.getInstance().getChatServers();
   	servers.addListListener(this);
    }

    // --- Method(s) ---

    private void initialize()
    {
	// chat
	cpChat = new ConsolePane();

	JPanel jpChannels = new JPanel(new BorderLayout());
	
	// table
	ctm = new ChannelTableModel();
  	jta = ctm.createJTable();
  	jta.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
	jta.setShowGrid(false);

	jta.getInputMap().put(KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0),
			      acJoinChannel);
	jta.getActionMap().put(acJoinChannel, acJoinChannel);
	jta.addMouseListener(new DoubleClickListener(acJoinChannel, jta));
		
	JScrollPane jsp = new JScrollPane(jta);
	jpChannels.add(jsp, BorderLayout.CENTER);

	// table context menu
	JPopupMenu popup = new JPopupMenu();
	popup.add(acJoinChannel);
	popup.addSeparator();
	popup.add(acCreateChannel);
	
	MouseListener popupListener = new PopupListener(popup);
	jta.addMouseListener(popupListener);

	// server combo box
	JPanel jpServer = new JPanel(new BorderLayout());
	jpServer.setBorder(new EmptyBorder(5, 5, 5, 5));
	jpChannels.add(jpServer, BorderLayout.SOUTH);

	jpServer.add(new JLabel(XNap.tr("Server") + " "), BorderLayout.WEST);

	dcbm = new DefaultComboBoxModel();
    	jcbServers = new JComboBox(dcbm);
	jcbServers.addActionListener(acUpdateChannels);
	jcbServers.setRenderer(new ChatServerCellRenderer());
    	jpServer.add(jcbServers, BorderLayout.CENTER);

	// split pane
	jspH = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
	jspH.add(cpChat, JSplitPane.LEFT);
	jspH.add(jpChannels, JSplitPane.RIGHT);
	jspH.setDividerLocation(prefs.getChatGlobalVerticalDividerLocation());
	jspH.setResizeWeight(1);
	jspH.setOneTouchExpandable(true);

	setLayout(new BorderLayout(5, 5));
	add(jspH, "Center");
    }

    public int getDividerLocation()
    {
	return jspH.getDividerLocation();
    }

    public void elementAdded(ListEvent e)
    {
        final Object obj = e.getElement();
	SwingUtilities.invokeLater(new Runnable() 
	    {
		public void run() {
		    dcbm.addElement(obj);
		    try {
			((IChatServer)obj).updateChannels();
		    }
		    catch (IOException e) {
		    }
		}
	    });
    }
    
    public void elementRemoved(ListEvent e)
    {
        final Object obj = e.getElement();
	SwingUtilities.invokeLater(new Runnable() 
	    {
		public void run() {
		    dcbm.removeElement(obj);
		}
	    });
    }

    public JMenu getChannelTableMenu()
    {
	return ctm.createJMenu();
    }

    public IChannel getSelectedChannel()
    {
	int i = jta.getSelectedRow();
	if (i != -1) {
	    return ctm.get(i);
	}
	return null;
    }
    
    public IChatServer getSelectedServer() 
    {
	return (IChatServer)jcbServers.getSelectedItem();
    }

    public AbstractAction[] getActions()
    {
	return (new AbstractAction[] {
	    acJoinChannel, acCreateChannel, acUpdateChannels
	});
    }

    public void messageReceived(ChannelEvent e)
    {
	StringBuffer sb = new StringBuffer();
	sb.append((e.getUser() != null) ? e.getUser() + ": " : "");
	sb.append(e.getMessage());
	sb.append("\n");

	cpChat.appendLater(sb.toString());
    }

    public void savePrefs()
    {
	prefs.setChatGlobalVerticalDividerLocation(jspH.getDividerLocation());
    }

    private class JoinChannelAction extends AbstractAction {

        public JoinChannelAction() 
	{
            putValue(Action.NAME, XNap.tr("Join"));
            putValue(Action.SHORT_DESCRIPTION, XNap.tr("Join Selected Channel"));
	    putValue(Action.SMALL_ICON, 
		     XNapFrame.getIcon("connect_creating.png"));
	}

        public void actionPerformed(ActionEvent event) 
	{
	    IChannel c = getSelectedChannel();
	    if (c != null) {
		try {
		    c.join();
		}
		catch (IOException e) {
		    setStatus(e.getMessage());
		}
	    }
	    else {
		setStatus(XNap.tr("Please select a channel"));
	    }
        }
    }	
   
    private class CreateChannelAction extends AbstractAction {
	
	public CreateChannelAction() 
	{
            putValue(Action.NAME, XNap.tr("Create"));
            putValue(Action.SHORT_DESCRIPTION, XNap.tr("Create a new Channel"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("filenew.png"));
            putValue(Action.MNEMONIC_KEY, new Integer('C'));
        }

	public void actionPerformed(ActionEvent event) 
	{
	    IChatServer s = getSelectedServer();
	    if (s != null) {
		String name = JOptionPane.showInputDialog
		    (ChatGlobalSubPanel.this, XNap.tr("Name"), 
		     XNap.tr("Create Channel"), JOptionPane.QUESTION_MESSAGE);

		if (name != null && name.trim().length() > 0) {
		    try {
			s.create(name.trim());
			s.updateChannels();
			ctm.set(s.getChannels());
		    }
		    catch (IOException e) {
			setStatus(e.getMessage());
		    }
		}
	    }
	}
    }
    
    private class UpdateChannelsAction extends AbstractAction {
	
	public UpdateChannelsAction() 
	{
            putValue(Action.NAME, XNap.tr("Update"));
            putValue(Action.SHORT_DESCRIPTION, XNap.tr("Update Channels"));
	    putValue(Action.SMALL_ICON, XNapFrame.getIcon("reload.png"));
            putValue(Action.MNEMONIC_KEY, new Integer('R'));
        }

        public void actionPerformed(ActionEvent event) 
	{
	    IChatServer s = getSelectedServer();
	    if (s != null) {
		try {
		    if (event.getSource() == jcbServers) {
			s.updateChannels();
		    }
		    ctm.set(s.getChannels());
		}
		catch (IOException e) {
		    setStatus(e.getMessage());
		}
	    }
        }
    }


    private class Updater implements ActionListener 
    {
	public void actionPerformed(ActionEvent event)
	{
	    if (ChatGlobalSubPanel.this.hasFocus()) {
		IChatServer s = getSelectedServer();
		if (s != null && jta.getSelectedRowCount() == 0) {
		    try {
			ctm.set(s.getChannels());
		    }
		    catch (IOException e) {
		    }
		}
	    }
	}

    }

}
