/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui.tree;

import xnap.gui.Dialogs;
import xnap.gui.event.FileArray;
import xnap.gui.event.TransferableFile;
import xnap.util.FileHelper;

import org.apache.log4j.Logger;

import java.awt.*;
import java.awt.datatransfer.*;
import java.awt.dnd.*;
import java.awt.event.*;
import java.io.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.tree.*;

public class DroppableJTree extends JTree 
    implements DropTargetListener, Autoscroll
{

    //--- Constant(s) ---

    public static final int AUTOSCROLL_MARGIN = 14;

    //--- Data field(s) ---

    protected DropTarget dropTarget = new DropTarget(this, this);
    protected TreePath originalPath = null;
    protected Timer delayTimer;
    protected Component parent;
    protected boolean dndActive = false;

    private TreePath lastPath;

    private static Logger logger = Logger.getLogger(DroppableJTree.class);

    //--- Constructor(s) ---

    public DroppableJTree(TreeModel tm, Component c)
    {
	super(tm);
	parent = c;
	delayTimer = new Timer(500, new HoverListener());
	delayTimer.setRepeats(false);
    }
    
    //--- Method(s) ---

    public void autoscroll(Point p)
    {
	int row = getClosestRowForLocation(p.x, p.y);
	
	if (row < 0)
	    return;

	Rectangle bounds = getBounds();

	if (p.y + bounds.y <= AUTOSCROLL_MARGIN) {
	    row--;
	}
	else {
	    row++;
	}

	if (row >= 0 && row < getRowCount()) {
	    scrollRowToVisible(row);
	}
    }

    public Insets getAutoscrollInsets()
    {
	Rectangle outer = getBounds();
	Rectangle inner = getParent().getBounds();

	return new Insets(inner.y - outer.y + AUTOSCROLL_MARGIN,
			  inner.x - outer.x + AUTOSCROLL_MARGIN,
			  outer.height - inner.height 
			  - inner.y + outer.y + AUTOSCROLL_MARGIN,
			  outer.width - inner.width - inner.x
			  + outer.x + AUTOSCROLL_MARGIN);
    }

    /*
    public void paintComponent(Graphics g)
    {
	super.paintComponent(g);
	Rectangle outer = getBounds();
	Rectangle inner = getParent().getBounds();
	g.setColor(Color.red);
	g.drawRect(outer.x, outer.y, outer.width, outer.height);
	g.setColor(Color.blue);
	g.drawRect(inner.x, inner.y, inner.width, inner.height);
    }
    */

    public void dragEnter(DropTargetDragEvent event)
    {
	originalPath = getSelectionPath();
	dndActive = true;
    }

    public void dragExit(DropTargetEvent event)
    {
	logger.debug("dragExit!");

	// FIX: why does java 1.3 call dragExit twice before drop?
	if (lastPath == null) {
	    lastPath = getSelectionPath();
	    logger.debug("path: " + lastPath);
	}

	cleanUp();
    }

    public void dragOver(DropTargetDragEvent event)
    {
	Point p = event.getLocation();
	TreePath path = getClosestPathForLocation(p.x, p.y);
	if (path != getSelectionPath()) {
	    delayTimer.restart();
	    setSelectionPath(path);
	}
    }

    /**
     * After DND this method should be called to set the tree in its original
     * state.
     */
    protected void cleanUp()
    {
	logger.debug("cleanUp!");

	dndActive = false;
	if (originalPath != null) {
	    setSelectionPath(originalPath);
	    scrollPathToVisible(originalPath);
	}
	delayTimer.stop();
    }
        
    /**
     * Only issue TreeSelectionEvents if the tree is not in DND mode.
     */
    protected void fireValueChanged(TreeSelectionEvent e)
    {
	if (!dndActive)
	    super.fireValueChanged(e);
    }

    public void drop(DropTargetDropEvent event)
    {
	logger.debug("dropped!");

	delayTimer.stop();

	try {
	    Transferable t = event.getTransferable();
	    
	    if (!t.isDataFlavorSupported(TransferableFile.FILE_FLAVOR)) {
		event.rejectDrop();
		cleanUp();
		return;
	    }
	    
	    event.acceptDrop(DnDConstants.ACTION_MOVE);

	    FileArray f = 
		(FileArray)t.getTransferData(TransferableFile.FILE_FLAVOR);
	    
	    TreePath selected 
		= (getSelectionPath().equals(originalPath)) ? lastPath : getSelectionPath();
	    lastPath = null;
	    if (selected == null) {
		event.rejectDrop();
		cleanUp();
		return;
	    }
	    
	    Object o = selected.getLastPathComponent();
	    if (o instanceof File) {
		File dir = (File)o;
		File files[] = Dialogs.showMoveDialog(parent, f.getFiles(), 
						      dir);

		if (files == null) {
		    event.dropComplete(false);
		    cleanUp();
		    return;
		}

		for (int i = 0; i < files.length; i++) {
		    try {
			FileHelper.moveUnique(files[i], dir.getAbsolutePath());
		    }
		    catch (IOException ie) {
			logger.debug("drag move failed", ie);
		    }
		}
		event.dropComplete(true);
	    }
	}
	catch (IOException ie) {
	}
	catch (UnsupportedFlavorException ue) {
	}

	cleanUp();
    }
    
    public void dropActionChanged(DropTargetDragEvent event)
    {
    }

    protected class HoverListener implements ActionListener
    {
	public void actionPerformed(ActionEvent event)
	{
	    TreePath path = getSelectionPath();

	    if (path == null)
		return;
	    
	    if (isExpanded(path)) {
		collapsePath(path);
	    }
	    else {
		scrollPathToVisible(path);
		expandPath(path);
	    }
	}
    }
}

