/*
 *  XNap
 *
 *  A pure java file sharing client.
 *
 *  See AUTHORS for copyright information.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
package xnap.gui;

import xnap.util.*;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import javax.swing.*;
import javax.swing.border.*;
import javax.swing.event.*;

public class IconSplitPane extends JSplitPane
{

    //--- Constant(s) ---

    //--- Data field(s) ---

    private JList jlIcons;
    private DefaultListModel dlmIcons;
    private CardLayout clCenter;
    private Container jpCenter;
    protected ChangeListener changeListener = null;
    protected transient ChangeEvent changeEvent = null;
    protected Component selection;

    //--- Constructor(s) ---

    public IconSplitPane()
    {
	super(JSplitPane.HORIZONTAL_SPLIT);

	initialize();
    }

    //--- Method(s) ---

    private void initialize() 
    {
	setOneTouchExpandable(true);

	jlIcons = new JList();
	dlmIcons = new DefaultListModel();
	jlIcons.setModel(dlmIcons);
	jlIcons.addListSelectionListener(new IconListener());
	jlIcons.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
	jlIcons.setCellRenderer(new IconRenderer());

	JScrollPane jspIcons = new JScrollPane(jlIcons);
	jspIcons.setHorizontalScrollBarPolicy
	    (JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
	add(jspIcons, JSplitPane.LEFT);

	clCenter = new CardLayout();
	jpCenter = new Container();
	jpCenter.setLayout(clCenter);
	add(jpCenter, JSplitPane.RIGHT);
    }

    public void addTab(String description, Icon icon, Component c) 
    {
	insertTab(description, icon, c, dlmIcons.size());
    }

    public void insertTab(String description, Icon icon, Component c, 
			  int index) 
    {
	if (icon != null) {
	    if (icon instanceof ImageIcon) {
		((ImageIcon)icon).setDescription(description);
	    }
	    dlmIcons.insertElementAt(icon, index);
	}
	else {
	    dlmIcons.insertElementAt(description, index);
	}

	jpCenter.add(c, description, index);
	setDividerLocation(jlIcons.getPreferredSize().width + 30);

	if (dlmIcons.size() == 1) {
	    jlIcons.setSelectedIndex(0);
	    //setDividerLocation(jlIcons.getPreferredSize().width * 2);
	}
	else if (jlIcons.getSelectedIndex() >= index) {
	    jlIcons.setSelectedIndex(jlIcons.getSelectedIndex() + 1);
	}
    }

    public int indexOf(Component c)
    {
	int count = jpCenter.getComponentCount();
	for (int i = 0; i < count; i++) {
	    if (jpCenter.getComponent(i) == c) {
		return i;
	    }
	}

	return -1;
    }

    public Component[] getTabs()
    {
	return jpCenter.getComponents();
    }

    public Component getTabAt(int i)
    {
	return jpCenter.getComponents()[i];
    }

    public int getTabCount()
    {
	return jpCenter.getComponentCount();
    }

    public String getDescriptionAt(int i)
    {
	Object o = dlmIcons.get(i);
	if (o != null) {
	    if (o instanceof ImageIcon) {
		return ((ImageIcon)o).getDescription();
	    }
	    return o.toString();
	}

	return null;
    }

    public Icon getIconAt(int i)
    {
	Object o = dlmIcons.get(i);
	if (o instanceof Icon) {
	    return (Icon)o;
	}
	else {
	    return null;
	}
    }

    public String getTitleAt(int i)
    {
	return dlmIcons.get(i).toString();
    }

    public void setIconAt(int i, Icon icon)
    {
	if (icon != null) {
	    if (icon instanceof ImageIcon) {
		((ImageIcon)icon).setDescription(getDescriptionAt(i));
	    }

	    dlmIcons.set(i, icon);
	}
    }

    public int getSelectedIndex()
    {
	return jlIcons.getSelectedIndex();
    }

    public void setSelectedComponent(Component c)
    {
	int i = indexOf(c);
	if (i != -1) {
	    jlIcons.setSelectedIndex(i);
	}
    }

    public void remove(Component c)
    {
	if (jpCenter == null)
	    return;

	int i = indexOf(c);
	if (i != -1) {
	    removeTabAt(i);
	}
    }

    public void removeTabAt(int i)
    {
	if (i == jlIcons.getSelectedIndex()) {
	    jlIcons.setSelectedIndex(0);
	}
	jpCenter.remove(i);
	dlmIcons.remove(i);
    }

    public void addChangeListener(ChangeListener l) 
    {
        listenerList.add(ChangeListener.class, l);
    }

    /**
     * Removes a <code>ChangeListener</code> from this tabbedpane.
     *
     * @param l the ChangeListener to remove
     * @see #fireStateChanged
     * @see #addChangeListener
     */
    public void removeChangeListener(ChangeListener l) {
        listenerList.remove(ChangeListener.class, l);
    }

    /**
     * Send a <code>ChangeEvent</code>, whose source is this tabbedpane, to
     * each listener.  This method method is called each time
     * a <code>ChangeEvent</code> is received from the model.
     *
     * @see #addChangeListener
     * @see EventListenerList
     */
    protected void fireStateChanged() {
        // Guaranteed to return a non-null array
        Object[] listeners = listenerList.getListenerList();
        // Process the listeners last to first, notifying
        // those that are interested in this event
        for (int i = listeners.length-2; i>=0; i-=2) {
            if (listeners[i]==ChangeListener.class) {
                // Lazily create the event:
                if (changeEvent == null)
                    changeEvent = new ChangeEvent(this);
                ((ChangeListener)listeners[i+1]).stateChanged(changeEvent);
            }
        }
    }

    public Component getSelectedComponent()
    {
	return jpCenter.getComponent(jlIcons.getSelectedIndex());
    }

    //--- Inner Classe(s) ---

    private class IconListener implements ListSelectionListener
    {

	public void valueChanged(ListSelectionEvent e)
	{
	    int index = jlIcons.getSelectedIndex();
	    if (index != -1) {
		String description = null;
		Object o = dlmIcons.elementAt(index);
		if (o instanceof ImageIcon)
		    description = ((ImageIcon)o).getDescription();
		else if (o instanceof String)
		    description = (String)o;
		clCenter.show(jpCenter, description);
		fireStateChanged();
	    }
	}

    }

    private class IconRenderer extends JLabel implements ListCellRenderer 
    {

	public IconRenderer() {
	    this.setOpaque(true);
	    this.setHorizontalAlignment(CENTER);
	    this.setVerticalAlignment(CENTER);
	    this.setHorizontalTextPosition(CENTER);
	    this.setVerticalTextPosition(SwingConstants.BOTTOM);
	    this.setBorder(new EmptyBorder(5, 5, 5, 5));
	}

	public Component getListCellRendererComponent(JList list, Object value,
						      int index,
						      boolean isSelected,
						      boolean cellHasFocus)
	{
	    if (isSelected) {
		this.setBackground(list.getSelectionBackground());
		this.setForeground(list.getSelectionForeground());
	    } 
	    else {
		this.setBackground(list.getBackground());
		this.setForeground(list.getForeground());
	    }

	    this.setText(null);
	    this.setIcon(null);

	    if (value instanceof ImageIcon) {
		ImageIcon icon = (ImageIcon)value;
		this.setText(icon.getDescription());
		this.setIcon(icon);
	    }
	    else if (value instanceof String) {
		this.setText((String)value);
	    }

	    return this;
	}

    }   

}
