/*
 * Created on 24-nov-2005
 *
 * TODO To change the template for this generated file go to
 * Window - Preferences - Java - Code Style - Code Templates
 */
package org.herac.tuxguitar.player.impl;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import javax.sound.midi.Instrument;
import javax.sound.midi.InvalidMidiDataException;
import javax.sound.midi.MidiMessage;
import javax.sound.midi.MidiSystem;
import javax.sound.midi.MidiUnavailableException;
import javax.sound.midi.Receiver;
import javax.sound.midi.Sequencer;
import javax.sound.midi.Soundbank;
import javax.sound.midi.Synthesizer;
import javax.sound.midi.Transmitter;

import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.ToolBar;
import org.herac.tuxguitar.gui.TuxGuitar;
import org.herac.tuxguitar.gui.system.config.ConfigKeys;
import org.herac.tuxguitar.gui.system.config.ConfigEditor;
import org.herac.tuxguitar.gui.system.config.items.Option;
import org.herac.tuxguitar.gui.system.config.items.SoundOption;
import org.herac.tuxguitar.gui.util.SystemError;
import org.herac.tuxguitar.player.base.MidiControllers;
import org.herac.tuxguitar.player.base.MidiPlayer;
import org.herac.tuxguitar.player.base.MidiSequenceParser;
import org.herac.tuxguitar.song.managers.SongManager;
import org.herac.tuxguitar.song.models.Duration;
import org.herac.tuxguitar.song.models.InstrumentString;
import org.herac.tuxguitar.song.models.Note;
import org.herac.tuxguitar.song.models.SongTrack;

/**
 * @author julian
 * 
 * TODO To change the template for this generated type comment go to Window - Preferences - Java - Code Style - Code Templates
 */
public class MidiPlayerImpl implements MidiPlayer {

	private static final int MAX_CHANNELS = 16;
	
    private SongManager songManager;

    private Sequencer sequencer;
    
    private Synthesizer synthesizer;

    private Receiver receiver;
    
    private Soundbank soundbank;

    private MidiMetaEventListener controller;
    
    private boolean running;

    private boolean paused;
    
    private boolean changeTickPosition;

    private long tickPosition;
    
    private boolean metronomeEnabled;
    
    private int metronomeTrack;
    
    private int infoTrack;
    
    private boolean anySolo;
    
    private List systemErrors;
    
    public MidiPlayerImpl() {        
    	this.songManager = TuxGuitar.instance().getSongManager();
        this.controller = new MidiMetaEventListener();        
        this.systemErrors = new ArrayList();
		this.init();
		this.reset();        
    }

    /**
     * Inicia el Secuenciador y Sintetizador
     * @throws MidiUnavailableException 
     */

    public void init() {
        try{        	
        	getSynthesizer();
        	getSequencer();
        	
        	//check soundbank
    		boolean loadCustomSoundbank = TuxGuitar.instance().getConfig().getBooleanConfigValue(ConfigKeys.SOUNDBANK_CUSTOM);
    		if(loadCustomSoundbank){
    			if(!loadSoundbank(new File(TuxGuitar.instance().getConfig().getStringConfigValue(ConfigKeys.SOUNDBANK_CUSTOM_PATH)))){    		   		
    		   		this.systemErrors.add(SystemError.getError(TuxGuitar.getProperty("soundbank.error"), TuxGuitar.getProperty("soundbank.error.custom")));
    			}
    		}        	
        } catch (MidiUnavailableException e) {
        	e.printStackTrace();
        }         
    }    
    
    /**
     * Retorna el Sequenciador
     * @throws MidiUnavailableException 
     */
    private Sequencer getSequencer() throws MidiUnavailableException {
    	if (this.sequencer == null) {
    		this.sequencer = MidiSystem.getSequencer(false);
            this.sequencer.addMetaEventListener(this.controller);
        }
        if (!this.sequencer.isOpen()) {
        	this.sequencer.open();
        }            
        return this.sequencer;
    }

    /**
     * Retorna el Sintetizador
     * @throws MidiUnavailableException 
     */
    public Synthesizer getSynthesizer() throws MidiUnavailableException {
    	if (this.synthesizer == null) {
    		setSynthesizer(MidiSystem.getSynthesizer());
    	} 
        return this.synthesizer;
    }

    /**
     * Retorna el Soundbank por defecto
     * @throws MidiUnavailableException 
     */
    public Soundbank getSoundbank(){
    	if (this.soundbank == null) {      
    		try{    		  	
    			Synthesizer synthesizer = getSynthesizer();
    			this.soundbank = synthesizer.getDefaultSoundbank();    			
        	} catch (MidiUnavailableException e) {
        		e.printStackTrace();
        	}   
    	}
        return this.soundbank;
    }
    
    /**
     * Asigna un Synthesizer
     */
    public void setSynthesizer(Synthesizer synthesizer){
    	try {
    		this.soundbank = null;
    		if(this.synthesizer != null && this.synthesizer.isOpen()){
    			this.synthesizer.close();
    		}    		
    		this.synthesizer = synthesizer;
    		if(this.synthesizer != null){
    			this.synthesizer.open();  
				this.connect(this.synthesizer.getReceiver());
    		}
		} catch (MidiUnavailableException e) {
			e.printStackTrace();
		}                
    }
    
    /**
     * Conecta el Synthesizer al Sequencer
     */
    public void connect(Receiver receiver){
    	this.receiver = receiver;
    	this.connect();
    }
    
    /**
     * Conecta el Synthesizer al Sequencer
     */
    public void connect(){
    	try {									
    		Iterator it = getSequencer().getTransmitters().iterator();
			while(it.hasNext()){
				((Transmitter)it.next()).close();
			}
			Transmitter transmitter = getSequencer().getTransmitter();
            transmitter.setReceiver(this.receiver);
		} catch (MidiUnavailableException e) {
			e.printStackTrace();
		}   
    }    

    /**
     * Resetea los valores
     */
    public void reset(){
    	this.stop();
        this.tickPosition = Duration.QUARTER_TIME;
        this.setChangeTickPosition(false);
        this.controller.reset();       
    }

    /**
     * Cierra el Secuenciador y Sintetizador
     * @throws MidiUnavailableException 
     */
    public void close(){
    	this.stop();
    	this.soundbank = null;    	
    	if (this.synthesizer != null) {
    		this.synthesizer.close();
    		this.synthesizer = null;
    	}        
    	if (this.sequencer != null) {
    		this.sequencer.close();
    		this.sequencer = null;
    	}
    }
    
    /**
     * Para la reproduccion
     * @throws MidiUnavailableException 
     */
    public void stop(boolean paused) {        
    	this.setPaused(paused);    		 
    	try{
    		if(this.isRunning() && this.getSequencer().isOpen()){    			
    			this.allNotesOff();
    			this.systemReset();
    			this.getSequencer().stop();
    		}	
    	} catch (MidiUnavailableException e) {
    		e.printStackTrace();
    	}      	
    	this.setRunning(false);
    }    
    
    /**
     * Para la reproduccion
     * @throws MidiUnavailableException 
     */
    public void stop() {        
    	this.stop(false);   		               	
    }

    public void pause(){
    	this.stop(true); 
    }
    
    /**
     * Inicia la reproduccion
     * @throws MidiUnavailableException 
     */
    public synchronized void play(){     
    	try {      	
    		this.stop();     		    		    		    		
    		this.addSecuence();
    		this.updatePrograms();
    		this.updateControllers();
    		this.updateDefaultControllers();
    		this.setMetronomeEnabled(isMetronomeEnabled());    		
    		this.setChangeTickPosition(true);
    		this.setRunning(true);    		
    		this.getSequencer().start();
    		new Thread(new Runnable() {
    			public synchronized void run() {
    				try {                	
    					while (getSequencer().isRunning() && isRunning()) {
    						if (isChangeTickPosition()) {
    							changeTickPosition();
    						}	
    						tickPosition =  getSequencer().getTickPosition();
    						Thread.sleep(10);    						
    					}                    
    					//FINISH
    					if(isRunning()){
    						if(tickPosition >= (getSequencer().getTickLength() - 500)){
    							reset();
    						}else {
    							stop(isPaused());
    						}               
    					}
    				} catch (InterruptedException e) {
    					e.printStackTrace();
    				} catch (MidiUnavailableException e) {
    					reset();  
						e.printStackTrace();
					}
    			}
    		}).start();    	
    	}catch (MidiUnavailableException e) {
    		reset();   	
			e.printStackTrace();
		}
    }

    public void send(MidiMessage message){
    	if(this.receiver != null){
    		this.receiver.send(message,-1);
    	}
    }
    
    /**
     * Asigna el valor a running
     */
    public void setRunning(boolean running) {
        this.running = running;
    }

    /**
     * Retorna True si esta reproduciendo
     */
    public boolean isRunning() {
        return this.running;
    }

    public boolean isPaused() {
		return paused;
	}

	public void setPaused(boolean paused) {
		this.paused = paused;
	}

	/**
     * Retorna True si hay cambios en la posicion
     */
    private boolean isChangeTickPosition() {
        return changeTickPosition;
    }

    /**
     * Asigna los cambios de la posicion
     */
    private void setChangeTickPosition(boolean changeTickPosition) {
        this.changeTickPosition = changeTickPosition;
    }


    /**
     * Indica la posicion del secuenciador
     * @throws MidiUnavailableException 
     */
    public void setTickPosition(long position) {
    	setTickPosition(position,controller.getMove()); 
    }    
    
    /**
     * Indica la posicion del secuenciador
     * @throws MidiUnavailableException 
     */
    public void setTickPosition(long position,long move) {
    	this.tickPosition = position;
    	this.controller.setMove(move);
    	this.setChangeTickPosition(true);
    	if(!isRunning()){
    		this.changeTickPosition();
    	}
    }
    
    /**
     * Retorna el tick de la nota que esta reproduciendo
     */
    public long getTickPosition() {
    	return this.tickPosition - this.controller.getMove();
    }

    private void changeTickPosition(){
    	try{
    		if(isRunning()){    			    			
    			this.getSequencer().setTickPosition(tickPosition);    			
    			//reseteo los volumenes.    			        		
    			//this.setMetronomeEnabled(isMetronomeEnabled());
    			//this.updateControllers();    		        		
    		}    		
		} catch (MidiUnavailableException e) {
			e.printStackTrace();
		}     
        setChangeTickPosition(false);    	
    }
    
    /**
     * Agrega la Secuencia
     * @throws MidiUnavailableException 
     */
    public void addSecuence() {
    	try{
			MidiSequenceParser parser = new MidiSequenceParser(songManager,MidiSequenceParser.DEFAULT_PLAY_FLAGS);
			MidiSequenceImpl sequence = new MidiSequenceImpl(songManager);
			parser.parse(sequence);			
			this.getSequencer().setSequence(sequence.getSequence());			
			this.infoTrack = sequence.getInfoTrack();
			this.metronomeTrack = sequence.getMetronomeTrack();										
		} catch (MidiUnavailableException e) {
			e.printStackTrace();
		} catch (InvalidMidiDataException e) {
			e.printStackTrace();
		}    		
    }

    private void updateDefaultControllers(){    	
		for(int channel = 0; channel < MAX_CHANNELS;channel ++){			
			this.send(MidiMessageUtils.controlChange(channel,MidiControllers.REGISTERED_PARAMETER_FINE,0));
			this.send(MidiMessageUtils.controlChange(channel,MidiControllers.REGISTERED_PARAMETER_COARSE,0));        
			this.send(MidiMessageUtils.controlChange(channel,MidiControllers.DATA_ENTRY_COARSE,12));
			this.send(MidiMessageUtils.controlChange(channel,MidiControllers.REGISTERED_PARAMETER_COARSE,0));
			this.send(MidiMessageUtils.controlChange(channel,MidiControllers.REGISTERED_PARAMETER_FINE,1));
			this.send(MidiMessageUtils.controlChange(channel,MidiControllers.DATA_ENTRY_COARSE,64));
			this.send(MidiMessageUtils.controlChange(channel,MidiControllers.REGISTERED_PARAMETER_FINE,127));						
		}
    }
    
    public void updatePrograms() {    	
		Iterator it = this.songManager.getSong().getTracks().iterator();
		while(it.hasNext()){
			SongTrack track = (SongTrack)it.next();
			this.send(MidiMessageUtils.programChange(track.getChannel().getChannel(),track.getChannel().getInstrument()));
			if(track.getChannel().getChannel() != track.getChannel().getEffectChannel()){
				this.send(MidiMessageUtils.programChange(track.getChannel().getEffectChannel(),track.getChannel().getInstrument()));
			}
		}    		    
    }    
    
	public void updateControllers() {    	
		try {
			this.anySolo = false;			
			Iterator it = this.songManager.getSong().getTracks().iterator();
			while(it.hasNext()){
				SongTrack track = (SongTrack)it.next();
				this.updateController(track);       
				this.anySolo = ((!this.anySolo)?track.getChannel().isSolo():this.anySolo);
			}    		
			this.afterUpdate();
		} catch (MidiUnavailableException e) {
			e.printStackTrace();
		}         
    }
    
    private void updateController(SongTrack track) throws MidiUnavailableException  {    	    		    	    		
    	int volume = (int)(((double)this.songManager.getSong().getVolume() / 10.0) * track.getChannel().getVolume());      	
    	int balance = track.getChannel().getBalance();   	    		    		    	
    	updateController(track.getChannel().getChannel(),volume,balance);
    	if(track.getChannel().getChannel() != track.getChannel().getEffectChannel()){
    		updateController(track.getChannel().getEffectChannel(),volume,balance);
    	}    	
		getSequencer().setTrackMute(track.getNumber(),track.getChannel().isMute());
		getSequencer().setTrackSolo(track.getNumber(),track.getChannel().isSolo());       
    }
    
    private void updateController(int channel,int volume,int balance) throws MidiUnavailableException  {    	     	
    	this.send(MidiMessageUtils.controlChange(channel,MidiControllers.VOLUME,volume));
    	this.send(MidiMessageUtils.controlChange(channel,MidiControllers.BALANCE,balance));    	    	
    }
    
    private void afterUpdate() throws MidiUnavailableException{
    	getSequencer().setTrackSolo(this.infoTrack,this.anySolo);
		getSequencer().setTrackSolo(this.metronomeTrack,(isMetronomeEnabled() && this.anySolo));
    }
    
    public boolean isMetronomeEnabled() {
		return metronomeEnabled;	
	}

	public void setMetronomeEnabled(boolean metronomeEnabled) {
		try {
			this.metronomeEnabled = metronomeEnabled;    	
			this.getSequencer().setTrackMute(this.metronomeTrack,!isMetronomeEnabled());
			this.getSequencer().setTrackSolo(this.metronomeTrack,(isMetronomeEnabled() && this.anySolo));
		} catch (MidiUnavailableException e) {			
			e.printStackTrace();
		}			
	}
    
    public void allNotesOff(){    	
    	for(int channel = 0;channel < MAX_CHANNELS;channel ++){
    		this.send(MidiMessageUtils.controlChange(channel, MidiControllers.ALL_NOTES_OFF,0));
		}			    	
    }    
    
    private void systemReset(){    	
    	this.send(MidiMessageUtils.systemReset());    	
    }

    public void playBeat(final SongTrack track,final List notes) {
    	final int channel = track.getChannel().getChannel();
    	final int program = track.getChannel().getInstrument();
    	final int volume = (int)(((double)this.songManager.getSong().getVolume() / 10.0) * track.getChannel().getVolume());      	
    	final int balance = track.getChannel().getBalance();    	    		
		this.send(MidiMessageUtils.programChange(channel,program));
		this.send(MidiMessageUtils.controlChange(channel,MidiControllers.VOLUME,volume));
		this.send(MidiMessageUtils.controlChange(channel,MidiControllers.BALANCE,balance));    		    		    		
    	Iterator it = notes.iterator();
    	while(it.hasNext()){
    		final Note note = (Note)it.next();							    	    			
    		new Thread(new Runnable() {		
    			public void run() {
    				try {								
    		    		int key = track.getOffset() + (note.getValue() + ((InstrumentString)track.getStrings().get(note.getString() - 1)).getValue());
    		    		int velocity = note.getVelocity();    					    					
    					send(MidiMessageUtils.noteOn(channel, key, velocity));    						
    					Thread.sleep(500);
    					send(MidiMessageUtils.noteOff(channel, key, velocity));
    				} catch (InterruptedException e) {					
    					e.printStackTrace();
    				}
    			}		
    		}).start();
    	}					   	
    }
    
    public boolean loadSoundbank(File file){    
		try {			
			Soundbank soundbank = MidiSystem.getSoundbank(file);
			if (soundbank != null && getSynthesizer().isSoundbankSupported(soundbank)){										
				//unload the current soundbank
				if(getSoundbank() != null){
					getSynthesizer().unloadAllInstruments(getSoundbank());
				}
				//load the new soundbank
				getSynthesizer().loadAllInstruments(soundbank);
				
				this.soundbank = soundbank;
				return true;
			}    
		}catch (MidiUnavailableException e) {
			e.printStackTrace();
		} catch (InvalidMidiDataException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
		return false;
    }    
    
    public void write(OutputStream out){
    	try {    		
    		MidiSequenceParser parser = new MidiSequenceParser(songManager,MidiSequenceParser.DEFAULT_EXPORT_FLAGS);
			MidiSequenceImpl sequence = new MidiSequenceImpl(songManager);
			parser.parse(sequence);
			MidiSystem.write(sequence.getSequence(),1,out);
		} catch (IOException e) {		
			e.printStackTrace();
		}
    }
    
    public String getInstrumentName(int instrument){
    	String name = null;
    	Soundbank soundBank = getSoundbank();
    	if(soundBank != null){
    		Instrument[] instruments = soundBank.getInstruments();    		
    		if(instrument >= 0 && instrument < instruments.length){
    			return instruments[instrument].getName();
    		}
    	}    	
    	return Integer.toString(instrument);
    }

	public List getSystemErrors() {				
		return this.systemErrors;
	}
    
    public Option getConfigOption(ConfigEditor editor,ToolBar toolBar,Composite parent){
    	return new SoundOption(editor,toolBar,parent);
    }
    
}