/*
 * robdefines.h
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** @file
  * Declares DefineInputStream. 
  */


#ifndef __RT_DEFINESTREAM__
#define __RT_DEFINESTREAM__

#include "robvars.h"

#include <rtstring.h>
#include <rtcollect.h>
#include <rtstreams.h>
#include <rtmap.h>

namespace rt {

/// An input stream which automatically replaces Defines while reading
class DefineInputStream : public lrt::InputStream  {
public:
	static const lrt::String whitespace;
	DefineInputStream(lrt::InputStream *in, const lrt::String &separators, const char commentStart = 0);
	virtual void close();
	virtual ~DefineInputStream();

	void setIgnoreComments(bool);
	void setAutoTrim(bool);
	void setEnableDefines(bool);
	
	virtual void mark();
	virtual void reset();
	virtual bool markSupported();
	virtual bool fail();

	virtual int read();
	RobLoadReturnType getWord(lrt::String& word, const lrt::String& specialSeps = "");
	lrt::String getLine(bool skipEmpty = true);

	int getLineNumber();
	// Skips until the end of the line. 
	// Returns false if end of file, true normally. 
	bool skipRestOfLine();
	// Expects the end of the line. 
	// Separators or comments (if enabled) before the end of line are allowed. 
	// If any other non-separator character is encountered before the end of line, returns false. 
	// Otherwise returns true. 
	bool expectEndOfLine();

	bool eos();

	/** checks the identifier (eg label name, define param) for validity.
      * The function assumes that the first magic char (eg '?') was removed from the identifier. */
	bool isIdentifierValid(const lrt::String& str);

private:
	class Define {
	public:
		Define(const lrt::String& name = "");
		~Define() {}

		lrt::InputStream* getStream();

		lrt::String name, value;
		lrt::Vector<lrt::String> params;
		int lineOffset;
	};

	class StreamInfo {
	public:
		StreamInfo();
		StreamInfo(Define& define);
		~StreamInfo();

		int lineNum;
		int lineOffset;
		int buf;
		bool eol;
		lrt::StringMap<lrt::String> params;
	};

	class StringInStream : public lrt::InputStream {
	public:
		StringInStream(const lrt::String&);
		virtual ~StringInStream() {}
		virtual int read();
		bool eos();
	private:
		const lrt::String& str;
		int pos;
	};

	void unread(int c);
	RobLoadReturnType readParams(lrt::StringMap<lrt::String>& params, lrt::Vector<lrt::String>& paramNames);
	RobLoadReturnType createDefine();
	
	lrt::Vector<StreamInfo> infos;
	lrt::Vector<lrt::InputStream*> streams;
	lrt::StringMap<Define> defineMap;

	enum {
		identCharInvalid = 0x00,
		identCharValidFirst = 0x01,
		identCharValidNext = 0x02,
		identCharValidAlways = 0x03
	};

	int validAsIdentifierChar[256];
	void initValidIdentifierChars();

	int markLineNum;
	lrt::String separators;
	char commentStart;
	bool ignoreComments, autoTrim, defines;
	int curStream;
};

} // namespace

#endif 
