/*
 * rtstreams.h
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

/** \file
  * Contains all the basic input and output streams.
  * This includes the abstract classes <tt>InputStream</tt> and <tt>OutputStream</tt>,
  * but also the concrete file stream classes <tt>FileXputStream</tt> and a
  * parsing stream.
  */

#ifndef __LRT_STREAMS__
#define __LRT_STREAMS__

#ifdef __SYMBIAN32__
  class RFs;
  class RFile;
  template<int S> class TBuf8;
#else
//#include <stdio.h>
#endif

#include "rtstring.h"

namespace lrt {

class File;
template <class T>class Array;



/** InputStream is the base class of all input streams (such as files opened for reading) in libRT.
  * Since this abstract input stream does not implement any source of data, it is impossible to
  * construct one.
  * For useful concrete input streams, look at the following classes:
  * @see FileInputStream
  * @see ParseInputStream
  */
class InputStream {

public:
	/** Returns whether marking and resetting a position in the stream is supported.
	  * The markSupported method of class InputStream always returns false.
	  * @see mark()
	  */
	virtual bool markSupported();

	/** Saves the current position in the stream for later restoring it.
	  * Note that if marking is not supported, <tt>mark()</tt> has no effect.
	  * @see markSupported()
	  * @see reset()
	  */
	virtual void mark();

	/** If possible, restores the stream position saved by <tt>mark()</tt>.
	  * If marking is not supported, <tt>reset()</tt> has no effect.
	  * @see markSupported()
	  * @see mark()
	  */
	virtual void reset();

	/** Reads a byte from the stream. If the stream cannot be read (or is EOF), this method returns -1.
	  * This method must be implemented by subclassers!
	  * @return The byte (in the lowest 8 bits of the returned integer), or -1 if the stream has ended.
	  */
	virtual int read() = 0;

	/** Reads up to len bytes from the stream, storing them in b, beginning at position off.
	  * Note that b must be large enough.
	  * @return The number of bytes actually read, or -1 if the stream's end was reached.
	  */
	virtual int read(Array<char>& b, int off, int len);

	/** Tries to fill the array b with bytes from the stream.
	  * <b>Note:</b> You need <i>never</i> subclass this method.
	  * @return The number of bytes actually read, or -1 if the stream's end was reached.
	  */
	int read(Array<char>& b); // you need not override this one

	/** Checks if this stream's end has been reached.
	  * This method must be implemented by subclassers!
	  * @return <tt>true</tt> if the stream has ended, <tt>false</tt> if there is still some data to read.
	  */
	virtual bool eos() = 0;

	/** Returns true if there was some system failure for this stream (such as, it couldn't be opened).
	  */
	virtual bool fail(); // you need not override this one

	/** Closes the stream, and releases any system resources. Closed streams cannot be re-opened.
	  * The close method of class <tt>InputStream</tt> does nothing.
	  */
	virtual void close();

	/** Destroys the stream.
	  * The destructor of class <tt>InputStream</tt> just calls the close method.
	  */
	virtual ~InputStream();

protected:
	/** Constructs an input stream.
	  * The constructor of class <tt>InputStream</tt> just does nothing and is only present to
	  * avoid construction of plain <tt>InputStream</tt>s.
	  */
	InputStream();

	/** Set <tt>_fail</tt>, if a system-dependent operation for this stream (such as opening) fails. */
	bool _fail;
};

/** FileInputStream implements a input stream which receives data from a file.
  * Marking and resetting positions is always supported by such streams.
  */
class FileInputStream : public InputStream {

public:
	/** Creates a FileInputStream from the file denoted by the given file descriptor <tt>file</tt>.
	  * @param file The file which should be opened.
	  */
	FileInputStream(const File& file, bool textMode);

	/** Creates a FileInputStream from the file given by its file name <tt>fileName</tt>.
	  * Note that you should always separate paths with forward slashes '<tt>/</tt>'; libRT will
	  * convert them to the system dependent path separators.
	  */
	FileInputStream(const String& fileName, bool textMode);

	/** @return True since marking is supported by FileInputStreams. */
	virtual bool markSupported();

	/** Marks the current position in the stream for later returning to it. */
	virtual void mark();

	/** Resets the current position to the previously marked one. If no position
	  * was marked before, the position is reset to the beginning of the file.
	  */
	virtual void reset();

	/** Reads one byte from the file.
	  * @return The byte read, or -1 if the end of file was reached.
	  */
	virtual int read();

	/** Checks if this file's end has been reached.
	  * @return <tt>true</tt> if the file has ended, <tt>false</tt> if there is still some data to read.
	  */
	virtual bool eos();

	/** Closes the file. */
	virtual void close();

	virtual ~FileInputStream();

private:
#ifdef __SYMBIAN32__
	RFs* session;
	RFile* file;
	TBuf8<80>* buf;
	int pos;
	void init(const String&, bool);
#else
	void* handle;
#endif
	int markPos;
	bool open;
};

/** A FilterInputStream is a stream which just performs any operations on another stream.
  * This class has no practical application, but it's a good starting point for clever streams.
  */
class FilterInputStream : public InputStream {

public:
	/** Constructs a new FilterInputStream, which gets its data from the underlying stream <tt>in</tt>.
	  */
	FilterInputStream(InputStream* in);

	/** Returns true only if the underlying input stream supports marking. */
	virtual bool markSupported();

	/** Marks the current position in the stream for later restoring it.
	  * This will only work if the underlying input stream supports marking.
	  */
	virtual void mark();

	/** Restores the previously marked position in the stream.
	  * This will only work if the underlying input stream supports marking.
	  */
	virtual void reset();

	/** Reads a byte from the underlying input stream. */
	virtual int read();

	/** Fills the array b with bytes from the underlying input stream. */
	virtual int read(Array<char>& b, int off, int len);

	/** Returns true if there was some system failure for this stream (such as, it couldn't be opened).
	  */
	virtual bool fail();  // you needn't override this one

	/** Returns the underlying input stream from this filter stream and detaches it
	  * so that it won't be deleted, and can be used further after this filter stream has been destroyed.
	  */
	virtual InputStream* detach();

	/** Checks if the underlying stream's end has been reached.
	  * @return <tt>true</tt> if the stream has ended, <tt>false</tt> if there is still some data to read.
	  */
	virtual bool eos();

	/** Closes the underlying input stream. */
	virtual void close();

	/** Destroys this FilterInputStream and the underlying input stream. */
	virtual ~FilterInputStream();

protected:
	/** The underlying input stream, which serves as data source for this stream. */
	InputStream* in;
	/** Is the underlying input stream detached? */
	bool _detach;
};

/** A ParseInputStream can parse any input stream for tokens ("words") and read whole lines from them.
  * It also keeps information about the current line number.
  */
class ParseInputStream : public FilterInputStream {

public:
	/** The separators which make up whitespace, i.e.\ '<tt> </tt>', '<tt>\t</tt>' and
	  * '<tt>\n</tt>'. This is a useful constant for the <tt>separators</tt> parameter.
	  */
	static const String whitespace;
	/** Creates a new ParseInputStream.
	  * @param in The input stream which provides the data for this parser.
	  * @param separators Characters which separate tokens ("words") from each other.
	  * @param commentStart The character which denotes the start of a end-of-line comment.
	  */
	ParseInputStream(InputStream* in, const String& separators, const char commentStart = 0);

	/** Set if comments can be ignored. If true, comments will be skipped automatically
	  * by a getWord() or getLine(). Also, lines which only consist of a comment will be
	  * skipped completely. The line number counter will still work correctly.
	  * This option is turned off by default.
	  */
	void setIgnoreComments(bool);

	/** Set if returned words and lines are auto-trimmed. If true, words and lines returned
	  * by a getWord() or getLine() will be stripped of any whitespace (including line ends).
	  * Also, words or lines which only consist of whitespace will be ignored completely.
	  * The line number counter will still work correctly.
	  * This option is turned off by default.
	  */
	void setAutoTrim(bool);

	/** Marks the current position in the stream for later restoring it.
	  * This will only work if the underlying input stream supports marking.
	  */
	virtual void mark();

	/** Restores the previously marked position in the stream.
	  * This will only work if the underlying input stream supports marking.
	  */
	virtual void reset();

	/** Reads a byte from the underlying input stream, or from the internal buffer.
	  */
	virtual int read();

	/** Reads a word from the input stream.
	  * @param specialSeps (optional) If given, does not use the input stream's default
	  *                    separators (as given on construction), but the given ones instead.
	  */
	String getWord(const String& specialSeps = "");

	/** Reads a whole line from the input stream.
	  * @param skipEmpty (optional) If true, getLine() will skip empty lines and instead
	  *                  return the next non-empty line.
	  */
	String getLine(bool skipEmpty = true);

	/** Gets the current line number. */
	int getLineNumber();

	/** Discards the rest of the current line. This includes the line end character '\\n'.*/
	bool skipRestOfLine();

	/** Returns true if the end of the underlying input stream is reached. */
	bool eos();

private:
	void unread(int c);
	int lineNum, markLineNum;
	String separators;
	char commentStart;
	int buf; // next char
	bool ignoreComments, autoTrim;
	bool eol; // we're at the end of the line
};

/** OutputStream is the base class for all output streams (such as files opened for writing) in libRT.
  * Since this abstract OutputStream does not implement any data destination, it is
  * impossible to construct one. For concrete OutputStreams, look at:
  * @see FileOutputStream
  */
class OutputStream {

public:
	/** Writes a byte to the output stream.
	  * Subclassers <i>must</i> implement this method.
	  * @return Whether writing succeeded.
	  */
	virtual bool write(int b) = 0;

	/** Writes part of an array to the output stream.
	  * @return Whether writing succeeded.
	  */
	virtual bool write(const Array<char>& b, int off, int len);

	/** Writes the whole array b to the output stream.
	  * Since String is derived from Array<char> in libRT, you may also use this
	  * method to write Strings to the output stream.
	  * @return Whether writing the array succeeded.
	  */
	virtual bool write(const Array<char>& b); // you need not override this one

	/** Convenience method. */
	bool write(const char* c);

	/** Advises the system output stream to flush its output buffer.
	  * The <tt>flush()</tt> method of OutputStream does nothing.
	  */
	virtual void flush();

	/** Closes the system output stream.
	  * Note that closed streams cannot be re-opened.
	  */
	virtual void close();

	/** Returns true if there was some system failure for this stream (such as, it couldn't be opened).
	  */
	virtual bool fail(); // you need not override this one

	/** Destroys the output stream.
	  * The destructor of class OutputStream just calls the <tt>close()</tt> method.
	  */
	virtual ~OutputStream();

protected:
	/** Constructs an output stream.
	  * The constructor of class <tt>OutputStream</tt> just does nothing and is only present to
	  * avoid construction of plain <tt>OutputStream</tt>s.
	  */
	OutputStream();

	/** Set <tt>_fail</tt>, if a system-dependent operation for this stream (such as opening) fails. */
	bool _fail;
};

/** Class FileOutputStreams implements an output stream which writes data to a file.
  * It supports opening files in either overwrite or append mode.
  */
class FileOutputStream : public OutputStream {

public:
	/** Creates a FileOutputStream which writes data to the file denoted by its name <tt>fileName</tt>.
	  * Note that you should always separate paths with forward slashes '<tt>/</tt>'; libRT will
	  * convert them to the system dependent path separators.
	  * @param fileName The file's name to write to.
	  * @param append If specified, data is appended to the file, if it exists.
	  */
	FileOutputStream(const String& fileName, bool append = false);

	/** Creates a FileOutputStream which writes data to the file denoted by its file descriptor <tt>file</tt>.
	  * @param file The file to write to.
	  * @param append If specified, data is appended to the file, if it exists.
	  */
    FileOutputStream(const File& file, bool append = false);

	/** Writes a byte to the file. */
	virtual bool write(int b);

    /** Writes part of an array to the file. */
	virtual bool write(const Array<char>& b, int off, int len);

	virtual bool write(const Array<char>& b); // you need not override this one

	/** Flushes the system's output buffer (if it exists) for this file. */
	virtual void flush();

	/** Closes the system output stream. */
	virtual void close();

	virtual ~FileOutputStream();

private:
#ifdef __SYMBIAN32__
	RFs* session;
	RFile* file;
	void init(const String&, bool);
#else
	void* handle;
#endif
	bool open;
};

/** A FilterOutputStream is a stream which just performs any operations on another stream.
  * This class has no practical application, but it's a good starting point for clever streams.
  */
class FilterOutputStream : public OutputStream {

public:
	/** Constructs a new FilterOutputStream, which writes its data from the underlying stream <tt>out</tt>.
	  */
	FilterOutputStream(OutputStream* out);

	/** Writes a byte to the underlying output stream.
	  * @return Whether writing succeeded.
	  */
	virtual bool write(int b);

	/** Writes part of an array to the underlying output stream.
	  * @return Whether writing succeeded.
	  */
	virtual bool write(const Array<char>& b, int off, int len);

	virtual bool write(const Array<char>& b); // you need not override this one

	/** Advises the underlying output stream to flush its output buffer.
	  */
	virtual void flush();

	/** Closes the underlying output stream.
	  * Note that closed streams cannot be re-opened.
	  */
	virtual void close();

	/** Destroys this FilterOutputStream and the underlying output stream. */
	virtual ~FilterOutputStream();

protected:
	/** The underlying output stream, where data is written to. */
	OutputStream* out;
};



} // namespace

#endif

