#if !defined(__LRT_NETWORK__) && !defined(__SYMBIAN32__) // && !(defined(__WIN32) && !defined(__MSVC__))
#define __LRT_NETWORK__
#define __NETWORK__ // alternate macro
/** \file
  * Contains networking classes. These are wrapper classes for the Simple Sockets
  * Library (a C library), which has been developed at the NASA Robotics Center. 
  * The main networking class is Network, which contains static methods to do
  * all important things - connecting to a remote server, setting up a server, 
  * parsing URLs and even performing simple HTTP requests. 
  * @see lrt::Network
  */

#include "rtstreams.h"
#include "rtmap.h"

namespace lrt {

class String;
template<class T> class StringMap;
class Time;

class Socket;
class ServerSocket;
class NetworkInputStream;
class NetworkOutputStream;

/** The network manager for libRT. Has methods for creating sockets. */
class Network {
public: // types
	/// HTTP Request method. Currently supported: GET, POST and HEAD. 
	enum HttpMethod { GET = 0, POST = 1, HEAD = 2 }; // change methodTable in network.cpp if this is changed!
	/// HTTP transmission page structure.
	struct Page { 
		Page() : headers(false), code(0), text() {} ///< Creates an empty Page. 
		bool isError() { return (code / 100 != 2); } ///< was the request successful? 
		StringMap<String> headers; ///< received headers (after request), headers to send (before request)
		String text; ///< page text (after request), post text (before request)	
		int code; ///< answer code (200, 404, etc.)
		String message; ///< answer message (Ok, Not Found, etc.)
	};
public: // static
	/// creates a client socket
	/// @param server Server to connect to
	/// @param port Port to connect to on the remote server
	/// @return Zero if error, a Socket object otherwise
	static Socket* create(const String& server, int port);
	/// creates a server socket
	/// @param port Port to listen to. 
	/// @return Zero if error, a ServerSocket object otherwise
	static ServerSocket* createServer(int port);

	/// Get the maximum waiting duration before the socket read operation is stopped. 
	static const Time& getTimeout();
	/// Set the maximum waiting duration before a socket read operation is cancelled
	/// Initially, this is set to 60 seconds to avoid application blocking. 
	/// You can set the timeout to zero (Time()) to disable timeout entirely. 
	static void setTimeout(const Time&);

	/// Splits an URL into protocol, server, port and file. 
	/// Components that are not given in the original url are set to their default values. 
	static void splitURL(const String& url, String& proto, String& server, int& port, String& file);
	/** Performs a HTTP request. 
	  * If you are performing a POST request, the data in page.text will be transmitted to the server. 
	  * You should set the page.headers[Content-Type] appropriately before performing the request. 
	  * Note especially that for sending forms to PHP pages, the default encoding text/plain does not work. 
	  * You have to create a query string in the same format as for GET requests (see urlEncode()) and use 
	  * the Content-Type "application/x-www-form-urlencoded".
	  */
	static bool httpRequest(const String& url, HttpMethod method, Page& page);
	/// Extended form 
	static bool httpRequest(const String& server, int port, HttpMethod method, const String& file, Page& page);

	/// Encodes a key-value table to an URL encoded form string. Additionally, non-acceptable characters
	/// are encoded to %xx hex number fomat. 
	static String urlEncode(const StringMap<String>& data);
	/// Encodes all non-acceptable characters in the given string to %xx hex numbers.  
	static String urlEncode(const String& str);

private:
	static Time timeout;
};

/// A network socket which can be used to transmit data over a TCP/IP based network. 
/// If one of the streams still exist when the Socket is being deleted, it will be deleted as well. 
/// So you must keep the Socket object alive as long as you use the stream. 
/// @see Network::create() for a factory method. 
class Socket {
public:
	virtual ~Socket(); 

	/// Gets the input stream of this socket, which you can use to read data from the remote computer.
	NetworkInputStream* getInput();
	/// Gets the output stream of this socket, which can be used to send data to the remote computer. 
	NetworkOutputStream* getOutput();
	/// Closes this socket and the associated streams.  
	void close();
	/// Check if the Socket is still open. 
	bool isClosed();
	/// Get the comms partner name
	String getPeerName();

protected:
	Socket(void* sock);
	friend class ServerSocket; // but why is this necessary?
	friend class Network;
	friend class NetworkInputStream;
	friend class NetworkOutputStream;

	void* sock;
	NetworkInputStream* in;
	NetworkOutputStream* out;
};

/// A ServerSocket is a Socket which advises a service to the network. As soon as it has been created
/// and the accept() method has been called, a remote computer can connect to the specified port 
/// on this computer. The resulting Socket is returned. 
class ServerSocket : public lrt::Socket {
public:
	~ServerSocket();
	/// Returns the Socket which is used to communicate with the client that has connected to this server. 
	Socket* accept();

protected:
	ServerSocket(void* sock);
	friend class Network;
};

/// Stream which reads data through a network from a remote computer. 
class NetworkInputStream : public InputStream {
public:
	~NetworkInputStream();

	virtual int read();
	virtual int read(Array<char> &b, int off, int len);
	virtual bool eos();
	/// close() does nothing here, you should close the Socket instead
	virtual void close();


protected:
	NetworkInputStream(Socket* sock);
	friend class Socket; 

	Socket* sock;
};

/// Stream which sends data to a remote computer through a network. 
class NetworkOutputStream: public OutputStream {
public:
	virtual ~NetworkOutputStream();

	virtual bool write(int b);
	virtual bool write(const Array<char> &b, int off, int len);
	virtual bool write(const Array<char> &b);
	/// close() does nothing here, you should close the Socket instead
	virtual void close();

protected:
	NetworkOutputStream(Socket* sock);
	friend class Socket; 

	Socket* sock;
};

} // namespace

#endif

