///////////// rtlist.h /////////////////////
/** \file
  * Contains the class template <tt>List</tt>, a double-linked list collection.
  * @see List
  */

#ifndef __LRT_LIST__
#define __LRT_LIST__

#include "rtiterator.h"
#include "rtdefs.h"

namespace lrt{

/** A double-linked list collection. 
  * As opposed to <tt>Vector</tt>s, elements can be inserted and removed very quickly
  * into lists, and sequential access to elements (forwards and backwards) is quick, too.
  * (The class Iterator is used for this.)
  * However, random access to elements is quite slow.
  */	
template<class T> class List
{
public:
	class Node
	{
	public:
		inline Node* getNext() const	{ return next; }
		inline Node* getPrev() const	{ return prev; }
		inline T& accessElement()		{ return element; }
		inline const T& accessElement() const { return element; }
	private:
		Node( const T &element ) : 	element(element), next(0), prev(0) {}
		~Node() {}

		inline void setNext( Node* nn)	{ next = nn; }
		inline void setPrev( Node* nn)	{ prev = nn; }

		friend class List<T>;
		T element;
		Node *next;
		Node *prev;
	};

	/** Concrete implementation of an iterator on lists. 
	  * This class is for use by List itself only. Use Iterator instead. */
	class IteratorImpl : public lrt::IIteratorImpl<T>
	{
	public:
		virtual ~IteratorImpl() {}
		
		// reimplementing just the virtual functions...
		virtual void goToNext() { if(element != 0) element = element->getNext(); }
		virtual void goToPrev() { if(element != 0) element = element->getPrev(); }
		virtual bool equals(const IIteratorImpl<T> *it) const 
		{ 
			const IteratorImpl* i = LRT_DYNAMIC_CAST(const IteratorImpl*)(it); 
			if(!i) return false; 
			return (element == i->element); 
		}

		virtual bool hasNext() const	{ if(!element) return false; return (element->getNext() != 0); }
		virtual bool hasPrev() const	{ if(!element) return false; return (element->getPrev() != 0); }
		virtual bool hasElement() const	{ return (element != 0); }

		virtual T& get()				{ return element->accessElement(); }
		virtual const T& get() const	{ return element->accessElement(); }

		virtual bool canModify()		{ return canMod; }
		virtual void remove()			{ if(!element) return; Node* el = element; element = list->remove(el); }

	private:
		friend class List<T>;
		virtual IIteratorImpl<T>* clone() const	{ return new IteratorImpl(list, element, canMod); }
		// el=0 means no element
		IteratorImpl(List<T>* const l, Node* el, bool cm) : list(l), element(el), canMod(cm) {} 
		List<T>* const list;
		Node *element;
		bool canMod;
	};

	/** Creates an empty list. */
	List();
	/** Creates a new list which contains a copy of all entries of the given list. */
	List(const List<T> &);
	/** Clears the list, then copies all the entries from the given list over to this one. */
	List<T> &operator=(const List<T> &);
	/** Destroys the list and all of its elements. */
	~List();

	/** Append an element to the list. 
	  * @param elem The element to append (it will be copied). 
	  * @return A reference to the appended element. */
	T& append( const T& elem );
	/** Prepend an element to the list. 
	  * @param elem The element to prepend (it will be copied). 
	  * @return A reference to the prepended element. */
	T& prepend( const T& elem );
	/** Insert an element before the given position. 
	  * @param pos The position after the new element. If the iterator points to nothing, 
	  *	           the new element is simply appended to the list. 
	  * @param elem The element to insert (it will be copied). 
	  * @return A reference to the inserted element. */
	T& insertBefore( Iterator<T> &pos, const T& elem ); // inserts before
	/** Insert an element after the given position. 
	  * @param pos The position before the new element. If the iterator points to nothing, 
	  *	           the new element is simply appended to the list. 
	  * @param elem The element to insert (it will be copied). 
	  * @return A reference to the inserted element. */
	T& insertAfter ( Iterator<T> &pos, const T& elem ); // inserts after
	/** Remove an element from the list. 
	  * Moves the iterator to the next element of the list. */
	inline void remove( Iterator<T> &pos );
	/** Remove an element from the list. 
	  * Returns a pointer to the next element of the list. */
	Node* remove(Node* element);

	/** Remove all elements from the list, and destroy them. */
	void clear();

	/** Returns the current element count of the list. */
	inline int length() const;
	/** Returns the current element count of the list. */
	inline int getSize() const;

	inline Node* first();
	inline const Node* first() const;
	inline Node* last();
	inline const Node* last() const;

	/** Returns an unmodifyable iterator to the first element of the list. 
	  * If the list is empty, the iterator points to nothing; you can use 
	  * Iterator::hasElement() to check this. */
	inline Iterator<T> begin() const;
	/** Returns an unmodifyable iterator to the last element of the list. 
	  * If the list is empty, the iterator points to nothing; you can use 
	  * Iterator::hasElement() to check this. */
	inline Iterator<T> end() const;
	/** Returns an modifyable iterator to the first element of the list. 
	  * If the list is empty, the iterator points to nothing; you can use 
	  * Iterator::hasElement() to check this. */
	inline Iterator<T> begin();
	/** Returns an modifyable iterator to the last element of the list. 
	  * If the list is empty, the iterator points to nothing; you can use 
	  * Iterator::hasElement() to check this. */
	inline Iterator<T> end();
private:
	friend class IteratorImpl; // my iterator
	int len;
	Node *_begin, *_end;	
};


// MSVC++ Bug Workaround
//#ifndef __LRT_TPTR_DEFINED
//#define __LRT_TPTR_DEFINED
//LRT_DEFINE_PTR(T)
//#endif

/** Removes the given node from the list, 
  * deletes it and advances the pointer to the next node.
  * @param list The list to remove the node from.
  * @param n A list node. 
  */
template<class T> void deleteNode(List<T>& list, typename List<T>::Node*& n);

} // namespace

#include "rtlist.templ.cpp"

#endif

