/*
 * Copyright (c) 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.vise;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

import java.sql.*;

import com.redhat.rhdb.misc.RHDBUtils;

/**
 * Dialog for testing DatabaseModels. This dialog is a modal dialog
 * whose owner is another Dialog. The tests that are run are:
 * <ul>
 * <li>Driver can be loaded
 * <li>Connection can be opened.
 * </ul>
 *
 * @author Liam Stewart
 * @version 1.2.0
 *
 * @see com.redhat.rhdb.vise.DatabaseModel
 */
public class TestDatabaseConnectionDialog extends JDialog implements ActionListener {
	private ActionFactory afactory;
	private GenericAction okAction;
	private int result;
	private Dialog owner;
	private DatabaseModel dbmodel;
	private JTextArea text;
	private JButton okButton;

	/** User has closed window with intention of accepting changes. */
	public static final int OK_OPTION = 0;

	/** User has closed window with intention of discarding changes. */
	public static final int CANCEL_OPTION = 1;

	/** User has closed window with intention of discarding changes. */
	public static final int CLOSED_OPTION = 2;

	/**
	 * Creates a new <code>TestDatabaseConnectionDialog</code> instance.
	 *
	 * @param owner a <code>Dialog</code> value
	 */
	public TestDatabaseConnectionDialog(Dialog owner)
	{
		super(owner, true);

		this.owner = owner;
			
		// Listeners
		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e)
			{
				close();
			}
		});

		afactory = new ActionFactory("/com/redhat/rhdb/images");
		okAction = afactory.getAction(ActionFactory.ACTION_OK);
		okAction.setMnemonic(null);
		okAction.addActionListener(this);

		initComponents();
	}

	/**
	 * Action listener.
	 *
	 * @param e an <code>ActionEvent</code> value
	 */
	public void actionPerformed(ActionEvent e)
	{
		String command = e.getActionCommand();

		if (command.equals(okAction.getActionCommand()))
			close();
		else
			System.err.println("TestDatabaseConnectionDialog: unknown action command" + command);
	}
	
	/**
	 * Get the database model being tested.
	 *
	 * @return a <code>DatabaseModel</code> value
	 */
	public DatabaseModel getDatabaseModel()
	{
		return dbmodel;
	}
		
	/**
	 * Set the database model to test.
	 *
	 * @param d a <code>DatabaseModel</code> value
	 */
	public void setDatabaseModel(DatabaseModel d)
	{
		this.dbmodel = d;

		setTitle("Testing Database Connection: " + d.getName());
	}

	/**
	 * Show the test dialog and run tests.
	 */
	public void show()
	{
		pack();
		setLocationRelativeTo(owner);
		okButton.requestFocus();
		doTest();

		super.show();
	}

	/**
	 * Show the test dialog and run tests for given database model.
	 *
	 * @param d a <code>DatabaseModel</code> value
	 */
	public void show(DatabaseModel d)
	{
		setDatabaseModel(d);
		show();
	}

	//
	// private methods
	//
	
	private void close()
	{
		result = CLOSED_OPTION;
		hide();
	}

	private void initComponents()
	{
		JPanel content;
		JPanel buttonPanel;
		Component testpanel;

		content = new JPanel();
		content.setLayout(new BorderLayout());
		content.setBorder(BorderFactory.createEmptyBorder(12, 12, 11, 12));
		setContentPane(content);

		okButton = okAction.getButton();
		getRootPane().setDefaultButton(okButton);

		buttonPanel = new JPanel();
		buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.X_AXIS));
		buttonPanel.setBorder(BorderFactory.createEmptyBorder(17, 0, 0, 0));
		buttonPanel.add(Box.createHorizontalGlue());
		buttonPanel.add(okButton);
		buttonPanel.add(Box.createHorizontalGlue());

		getContentPane().add(buttonPanel, BorderLayout.SOUTH);

		testpanel = createTestPanel();

		getContentPane().add(testpanel, BorderLayout.CENTER);
	}

	private JPanel createTestPanel()
	{
		JPanel panel = new JPanel();
		JScrollPane scroller;

		text = new JTextArea(15, 40);
		text.setEditable(false);
		text.setLineWrap(true);
		text.setWrapStyleWord(true);

		scroller = new JScrollPane(text);
		panel.add(scroller);

		return panel;
	}

	private void doTest()
	{
		String reason = "";
		boolean passed = true;

		// clear the text panel
		text.setText("");

		text.append("Test 1: Driver\n");
		try {
			Class.forName(dbmodel.getDriverClass());
			passed = true;
		} catch (ClassNotFoundException ex) {
			passed = false;
			reason = RHDBUtils.removeIncluding(ex.getMessage(),
											   "Stack Trace:");
		} finally {
			if (passed)
				text.append("PASSED\n");
			else
			{
				text.append("FAILED\n");
				text.append("Unable to load JDBC driver.\n");
			}
		}
		text.append("\n");

		text.append("Test 2: Connection\n");
		try {
			Connection con = DriverManager.getConnection(dbmodel.getURL(),
														 dbmodel.getUser(),
														 dbmodel.getPassword());
			con.close();
			passed = true;
		} catch (SQLException ex) {
			passed = false;
			reason = RHDBUtils.removeIncluding(ex.getMessage(),
											   "Stack Trace:");
		} finally {
			if (passed)
				text.append("PASSED\n");
			else
			{
				text.append("FAILED\n");
				text.append("Unable to connect to database.\n");
				text.append("Reason: " + reason + "\n");
			}
		}

		text.repaint();
	}

}// TestDatabaseConnectionDialog
