/*
 * Copyright (c) 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Fernando Nasser
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.vise;

import java.awt.event.*;
import javax.swing.*;

/**
 * Modal dialog for editing planner options.
 *
 * @author <a href="mailto:fnasser@redhat.com">Fernando Nasser</a>
 *
 * @version 1.2.0
 */
public class PlannerOptionsDialog extends javax.swing.JDialog {
	private ActionFactory afactory;
	private GenericAction okAction, cancelAction;
    private JPanel poperations;
    private JCheckBox seqscan;
    private JCheckBox indexscan;
    private JCheckBox tidscan;
    private JCheckBox sort;
    private JCheckBox hashagg;
    private JCheckBox nestloop;
    private JCheckBox mergejoin;
    private JCheckBox hashjoin;
    private JPanel pcost;
    private JLabel lrpc;
    private JTextField rpc;
    private JLabel ltc;
    private JTextField tc;
    private JLabel litc;
    private JTextField itc;
    private JLabel loc;
    private JTextField oc;
    private JPanel pgeqo;
    private JCheckBox geqo;
    private JLabel lthreshold;
    private JTextField threshold;
    private JLabel lpoolsize;
    private JTextField poolsize;
    private JLabel leffort;
    private JTextField effort;
    private JLabel lgenerations;
    private JTextField generations;
    private JLabel lselectionbias;
    private JTextField selectionbias;
    private JPanel buttons;
	private JButton cancelButton, okButton;

	/** User has closed window with intention of accepting changes. */
	public static final int OK_OPTION = 0;

	/** User has closed window with intention of discarding changes. */
	public static final int CANCEL_OPTION = 1;

	/** User has closed window with intention of discarding changes. */
	public static final int CLOSED_OPTION = 2;

    private int result;
    
    private PlannerOptions popts, defopts;   
    
    /** Creates new form PlannerOptionsDialog */
    public PlannerOptionsDialog(PlannerOptions defs, java.awt.Frame parent) {
        super(parent, true);
		this.defopts = defs;

		// Actions
		afactory = new ActionFactory();

		okAction     = afactory.getAction(ActionFactory.ACTION_OK);
		cancelAction = afactory.getAction(ActionFactory.ACTION_CANCEL);

		okAction.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                okActionPerformed(evt);
            }
        });
		cancelAction.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                cancelActionPerformed(evt);
            }
        });

		cancelAction.setMnemonic(null);
		okAction.setMnemonic(null);

        initComponents();

		cancelButton.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW).put(KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0),
																		cancelAction.getName());
		cancelButton.getActionMap().put(cancelAction.getName(), cancelAction);
    }
    
    /** This method is called from within the constructor to
     * initialize the form.
     */
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        poperations = new JPanel();
        seqscan = new JCheckBox();
        indexscan = new JCheckBox();
        tidscan = new JCheckBox();
        sort = new JCheckBox();
        hashagg = new JCheckBox();
        nestloop = new JCheckBox();
        mergejoin = new JCheckBox();
        hashjoin = new JCheckBox();
        pcost = new JPanel();
        lrpc = new JLabel();
        ltc = new JLabel();
        litc = new JLabel();
        loc = new JLabel();
        rpc = new JTextField();
        tc = new JTextField();
        itc = new JTextField();
        oc = new JTextField();
        pgeqo = new JPanel();
        geqo = new JCheckBox();
        lthreshold = new JLabel();
        lpoolsize = new JLabel();
        leffort = new JLabel();
        lgenerations = new JLabel();
        lselectionbias = new JLabel();
        threshold = new JTextField();
        poolsize = new JTextField();
        effort = new JTextField();
        generations = new JTextField();
        selectionbias = new JTextField();
        buttons = new JPanel();

        getContentPane().setLayout(new javax.swing.BoxLayout(getContentPane(), javax.swing.BoxLayout.Y_AXIS));

        setTitle(ExplainResources.getString(ExplainResources.PLANNEROPTIONS_TITLE));
        setLocationRelativeTo(getOwner());
        addWindowListener(new java.awt.event.WindowAdapter() {
            public void windowClosing(java.awt.event.WindowEvent evt) {
                closeDialog(evt);
            }
        });

		cancelButton = cancelAction.getButton();

		okButton = okAction.getButton();
		okButton.setPreferredSize(cancelButton.getPreferredSize());
		getRootPane().setDefaultButton(okButton);

        poperations.setLayout(new java.awt.GridLayout(4, 2));

        poperations.setBorder(new javax.swing.border.TitledBorder(
						ExplainResources.getString(ExplainResources.POPTS_TITLE_OPERATIONS)));
        seqscan.setSelected(true);
        seqscan.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_SEQSCAN));
        poperations.add(seqscan);

        indexscan.setSelected(true);
        indexscan.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_INDEXSCAN));
        poperations.add(indexscan);

        tidscan.setSelected(true);
        tidscan.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_TIDSCAN));
        poperations.add(tidscan);

        sort.setSelected(true);
        sort.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_SORT));
        poperations.add(sort);

        hashagg.setSelected(true);
        hashagg.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_HASHAGG));
        poperations.add(hashagg);

        nestloop.setSelected(true);
        nestloop.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_NESTLOOP));
        poperations.add(nestloop);

        mergejoin.setSelected(true);
        mergejoin.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_MERGEJOIN));
        poperations.add(mergejoin);

        hashjoin.setSelected(true);
        hashjoin.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_HASHJOIN));
        poperations.add(hashjoin);

        getContentPane().add(poperations);

        pcost.setLayout(new java.awt.GridBagLayout());

        pcost.setBorder(new javax.swing.border.TitledBorder(
				ExplainResources.getString(ExplainResources.POPTS_TITLE_ESTIMATES)));
        lrpc.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_RPC));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        pcost.add(lrpc, gridBagConstraints);

        ltc.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_TC));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        pcost.add(ltc, gridBagConstraints);

        litc.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_ITC));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        pcost.add(litc, gridBagConstraints);

        loc.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_OC));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        pcost.add(loc, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        pcost.add(rpc, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        pcost.add(tc, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        pcost.add(itc, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        pcost.add(oc, gridBagConstraints);

        getContentPane().add(pcost);

        pgeqo.setLayout(new java.awt.GridBagLayout());

        pgeqo.setBorder(new javax.swing.border.TitledBorder(
				ExplainResources.getString(ExplainResources.POPTS_TITLE_GEQO)));
        geqo.setSelected(true);
        geqo.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_GEQO));
        geqo.addChangeListener(new javax.swing.event.ChangeListener() {
            public void stateChanged(javax.swing.event.ChangeEvent evt) {
                geqoStateChanged(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 11, 0);
        gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
        pgeqo.add(geqo, gridBagConstraints);

        lthreshold.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_THRESHOLD));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        pgeqo.add(lthreshold, gridBagConstraints);

        lpoolsize.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_POOLSIZE));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        pgeqo.add(lpoolsize, gridBagConstraints);

        leffort.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_EFFORT));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        pgeqo.add(leffort, gridBagConstraints);

        lgenerations.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_GENERATIONS));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        pgeqo.add(lgenerations, gridBagConstraints);

        lselectionbias.setText(ExplainResources.getString(ExplainResources.POPTS_LABEL_SELBIAS));
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        pgeqo.add(lselectionbias, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        pgeqo.add(threshold, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        pgeqo.add(poolsize, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        pgeqo.add(effort, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 4;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        pgeqo.add(generations, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 5;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.weightx = 1.0;
        pgeqo.add(selectionbias, gridBagConstraints);

        getContentPane().add(pgeqo);

        buttons.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.RIGHT));

        buttons.add(okButton);
        buttons.add(cancelButton);

        getContentPane().add(buttons);

        pack();
    }

    private void geqoStateChanged(javax.swing.event.ChangeEvent evt) {
        // If disabled, disable all entry boxes
        // If enabled, enable the entry boxes again
		if (geqo.isSelected())
		{
			lthreshold.setEnabled(true);
			threshold.setEnabled(true);
			lpoolsize.setEnabled(true);
			poolsize.setEnabled(true);
			leffort.setEnabled(true);
			effort.setEnabled(true);
			lgenerations.setEnabled(true);
			generations.setEnabled(true);
			lselectionbias.setEnabled(true);
			selectionbias.setEnabled(true);
		}
		else
		{
			lthreshold.setEnabled(false);
			threshold.setEnabled(false);
			lpoolsize.setEnabled(false);
			poolsize.setEnabled(false);
			leffort.setEnabled(false);
			effort.setEnabled(false);
			lgenerations.setEnabled(false);
			generations.setEnabled(false);
			lselectionbias.setEnabled(false);
			selectionbias.setEnabled(false);
		}
    }

    private void cancelActionPerformed(java.awt.event.ActionEvent evt) {
        setVisible(false);
        //dispose();
		result = CANCEL_OPTION;
    }

	/**	Set the options to the new values, if changed
	 * NOTE: We do not change the original object
	 * so the caller must get the new one with getPlannerOptions()
	 *
	 * 
	 */
    private void okActionPerformed(java.awt.event.ActionEvent evt) {

		// Set the options to the new values, if changed
    	// NOTE: We do not change the original object
		// so the caller must get the new one with getPlannerOptions()
		if (!setOptions())
			return;
		
		result = OK_OPTION;
        setVisible(false);
    }
    
    /** Closes the dialog */
    private void closeDialog(java.awt.event.WindowEvent evt) {
		result = CLOSED_OPTION;
        setVisible(false);
        //dispose();
    }
    
    /**
     * Set the planner options object being edited to the new values
     * according to the user input.  If the value is the default,
     * just leave as null.
     *
	 * @return a <code>boolean</code> value if values OK
     */
    private boolean setOptions() {
    	boolean valuesOK = true;
	
    	// Start with all options set to null
		// Note that we are not changing the original object
    	PlannerOptions newpopts = new PlannerOptions(defopts);
		
		// Set a value (will generate a SET command) if we want something
		// different from the default or if we have to reset it back to
		// the default because we changed it on a previous run
		// Also, skip the ones not available on this backend version
		// They are cleared as we can't get a default from the backend
		// for those it does not know about.
		if (seqscan.isSelected() != defopts.enable_seqscan.booleanValue())
        	newpopts.enable_seqscan = new Boolean(seqscan.isSelected());

		if (indexscan.isSelected() != defopts.enable_indexscan.booleanValue())
        	newpopts.enable_indexscan = new Boolean(indexscan.isSelected());

		if (tidscan.isSelected() != defopts.enable_tidscan.booleanValue())
        	newpopts.enable_tidscan = new Boolean(tidscan.isSelected());

		if (sort.isSelected() != defopts.enable_sort.booleanValue())
        	newpopts.enable_sort = new Boolean(sort.isSelected());

		if ((defopts.enable_hashagg != null) &&
			(hashagg.isSelected() != defopts.enable_hashagg.booleanValue()))
        	newpopts.enable_hashagg = new Boolean(sort.isSelected());

		if (nestloop.isSelected() != defopts.enable_nestloop.booleanValue())
        	newpopts.enable_nestloop = new Boolean(nestloop.isSelected());

		if (mergejoin.isSelected() != defopts.enable_mergejoin.booleanValue())
        	newpopts.enable_mergejoin = new Boolean(mergejoin.isSelected());

		if (hashjoin.isSelected() != defopts.enable_hashjoin.booleanValue())
        	newpopts.enable_hashjoin = new Boolean(hashjoin.isSelected());

		if (geqo.isSelected() != defopts.geqo.booleanValue())
        	newpopts.geqo = new Boolean(geqo.isSelected());
	
		// Give an appropriate message for each field if parsing fails
		String varName = "";
		
		try {
			varName = ExplainResources.getString(ExplainResources.POPTS_LABEL_RPC);
			if (!rpc.getText().trim().equals(defopts.random_page_cost.toString()))
    	    	newpopts.random_page_cost = new Float(rpc.getText().trim());

			varName = ExplainResources.getString(ExplainResources.POPTS_LABEL_TC);
			if (!tc.getText().trim().equals(defopts.cpu_tuple_cost.toString()))
        		newpopts.cpu_tuple_cost = new Float(tc.getText().trim());

			varName = ExplainResources.getString(ExplainResources.POPTS_LABEL_ITC);
			if (!itc.getText().trim().equals(defopts.cpu_index_tuple_cost.toString()))
        		newpopts.cpu_index_tuple_cost = new Float(itc.getText().trim());

			varName = ExplainResources.getString(ExplainResources.POPTS_LABEL_OC);
			if (!oc.getText().trim().equals(defopts.cpu_operator_cost.toString()))
        		newpopts.cpu_operator_cost = new Float(oc.getText().trim());

			varName = ExplainResources.getString(ExplainResources.POPTS_LABEL_THRESHOLD);
			if (!threshold.getText().trim().equals(defopts.geqo_threshold.toString()))
        		newpopts.geqo_threshold = new Integer(threshold.getText().trim());

			varName = ExplainResources.getString(ExplainResources.POPTS_LABEL_POOLSIZE);
			if (!poolsize.getText().trim().equals(defopts.geqo_pool_size.toString()))
        		newpopts.geqo_pool_size = new Integer(poolsize.getText().trim());

			varName = ExplainResources.getString(ExplainResources.POPTS_LABEL_EFFORT);
			if (!effort.getText().trim().equals(defopts.geqo_effort.toString()))
        		newpopts.geqo_effort = new Integer(effort.getText().trim());

			varName = ExplainResources.getString(ExplainResources.POPTS_LABEL_GENERATIONS);
			if (!generations.getText().trim().equals(defopts.geqo_generations.toString()))
        		newpopts.geqo_generations = new Integer(generations.getText().trim());
			
			varName = ExplainResources.getString(ExplainResources.POPTS_LABEL_SELBIAS);
			if (!selectionbias.getText().trim().equals(defopts.geqo_selection_bias.toString()))
        		newpopts.geqo_selection_bias = new Float(selectionbias.getText().trim());
		} catch (NumberFormatException nfe) {
			JOptionPane.showMessageDialog(this,
										  ExplainResources.getString(ExplainResources.ERROR_BAD_VALUE,
																	 varName,
																	 nfe.getMessage()),
										  ExplainResources.getString(ExplainResources.ERROR_BAD_VALUE_TITLE),
										  JOptionPane.ERROR_MESSAGE);
			valuesOK = false;
			return valuesOK;
		}
		
		popts = newpopts;
		return valuesOK;
    }
    
    /**
     * Return what the user choose, OK or CANCEL (or if window was CLOSED)
     */
    public int getResult() {
        return result;
    }

	/**
	 * Set current planner options to be edited.
	 *
	 * @param opts a <code>PlannerOptions</code> object
	 */
	 public void setPlannerOptions(PlannerOptions defs, PlannerOptions opts) {
		this.defopts = defs;
	 	this.popts = opts;
		
		// Now set the field contents to either what is set or to
		// the default if it has not been specified
		// Clear and disable the ones not available on this backend 
		// version (default field is null).

		if (popts.enable_seqscan == null)
        	seqscan.setSelected(defopts.enable_seqscan.booleanValue());
		else
        	seqscan.setSelected(popts.enable_seqscan.booleanValue());

		if (popts.enable_indexscan == null)
        	indexscan.setSelected(defopts.enable_indexscan.booleanValue());
		else
        	indexscan.setSelected(popts.enable_indexscan.booleanValue());

		if (popts.enable_tidscan == null)
        	tidscan.setSelected(defopts.enable_tidscan.booleanValue());
		else
        	tidscan.setSelected(popts.enable_tidscan.booleanValue());

		if (popts.enable_sort == null)
        	sort.setSelected(defopts.enable_sort.booleanValue());
		else
        	sort.setSelected(popts.enable_sort.booleanValue());

		if (defopts.enable_hashagg == null)
        	hashagg.setEnabled(false);
		else
		{
        	hashagg.setEnabled(true);
			if (popts.enable_hashagg == null)
    	    	hashagg.setSelected(defopts.enable_hashagg.booleanValue());
			else
        		hashagg.setSelected(popts.enable_hashagg.booleanValue());
		}

		if (popts.enable_nestloop == null)
        	nestloop.setSelected(defopts.enable_nestloop.booleanValue());
		else
        	nestloop.setSelected(popts.enable_nestloop.booleanValue());

		if (popts.enable_mergejoin == null)
        	mergejoin.setSelected(defopts.enable_mergejoin.booleanValue());
		else
        	mergejoin.setSelected(popts.enable_mergejoin.booleanValue());

		if (popts.enable_hashjoin == null)
        	hashjoin.setSelected(defopts.enable_hashjoin.booleanValue());
		else
        	hashjoin.setSelected(popts.enable_hashjoin.booleanValue());

		if (popts.geqo== null)
        	geqo.setSelected(defopts.geqo.booleanValue());
		else
        	geqo.setSelected(popts.geqo.booleanValue());

		if (popts.random_page_cost == null)
        	rpc.setText(defopts.random_page_cost.toString());
		else
        	rpc.setText(popts.random_page_cost.toString());

		if (popts.cpu_tuple_cost == null)
        	tc.setText(defopts.cpu_tuple_cost.toString());
		else
        	tc.setText(popts.cpu_tuple_cost.toString());

		if (popts.cpu_index_tuple_cost == null)
        	itc.setText(defopts.cpu_index_tuple_cost.toString());
		else
        	itc.setText(popts.cpu_index_tuple_cost.toString());

		if (popts.cpu_operator_cost == null)
        	oc.setText(defopts.cpu_operator_cost.toString());
		else
        	oc.setText(popts.cpu_operator_cost.toString());

		if (popts.geqo_threshold == null)
        	threshold.setText(defopts.geqo_threshold.toString());
		else
        	threshold.setText(popts.geqo_threshold.toString());

		if (popts.geqo_pool_size == null)
        	poolsize.setText(defopts.geqo_pool_size.toString());
		else
        	poolsize.setText(popts.geqo_pool_size.toString());

		if (popts.geqo_effort == null)
        	effort.setText(defopts.geqo_effort.toString());
		else
        	effort.setText(popts.geqo_effort.toString());

		if (popts.geqo_generations == null)
        	generations.setText(defopts.geqo_generations.toString());
		else
        	generations.setText(popts.geqo_generations.toString());

		if (popts.geqo_selection_bias == null)
        	selectionbias.setText(defopts.geqo_selection_bias.toString());
		else
        	selectionbias.setText(popts.geqo_selection_bias.toString());
	}

	/**
	 * Gets the edited planner options.
	 *
	 * @return a <code>PlannerOptions</code> value
	 */
	public PlannerOptions getPlannerOptions() {
		return popts;
	}
}
