/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.treedisplay;

import java.awt.Point;

/**
 * Simple implementation of TreeLayoutNode. A node is just a
 * rectangle. All nodes have the same width and height.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 1.0
 */
public class DefaultTreeLayoutNode extends AbstractTreeLayoutNode {

	/** Height of each node. */
	public static final int NODE_HEIGHT = 40;

	/** Width of each node. */
	public static final int NODE_WIDTH = 60;

	/**
	 * Creates a new <code>DefaultTreeLayoutNode</code> instance.
	 */
	public DefaultTreeLayoutNode()
	{
		setWidth(NODE_WIDTH);
		setHeight(NODE_HEIGHT);
	}
	
	/**
	 * Creates a new <code>DefaultTreeLayoutNode</code> instance with
	 * specific data.
	 *
	 * @param o an <code>Object</code> value
	 */
	public DefaultTreeLayoutNode(Object o)
	{
		this();
		setData(o);
	}

	/**
	 * Does the node contain the point (x, y)?
	 *
	 * @param x an <code>int</code> value
	 * @param y an <code>int</code> value
	 * @return a <code>boolean</code> value
	 */
	public boolean contains(int x, int y)
	{
		Point ul = new Point(getX(), getY());
		Point lr = new Point(getX() + getWidth(), getY() + getHeight());
		Point c = new Point(x, y);

		if (c.x >= ul.x && c.x <= lr.x && c.y >= ul.y && c.y <= lr.y)
			return true;
		
		return false;
	}

	/**
	 * Get the bounding width of the node.
	 *
	 * @return bounding width
	 */
	public int getBoundingWidth()
	{
		int cc = getChildCount();
		int w = 0, i, ch, max;

		if (cc == 0)
			return getWidth();

		switch (getTreeLayoutModel().getOrientation())
		{
			case TreeDisplay.TOP:
			case TreeDisplay.BOTTOM:
				w = 0;
				for (i = 0; i < cc; i++)
					w += getChild(i).getBoundingWidth();

				w += ((cc - 1) * DefaultTreeLayoutModel.WIDTH_SPACE_TB);

				if (w < getWidth())
					w = getWidth();

				break;

			case TreeDisplay.LEFT:
			case TreeDisplay.RIGHT:
				max = 0;
				for (i = 0; i < cc; i++)
				{
					ch = getChild(i).getBoundingWidth();
					if (ch > max)
						max = ch;
				}

				w = getWidth() + DefaultTreeLayoutModel.WIDTH_SPACE_LR + max;

				break;
		}

		return w;
	}

	/**
	 * Get the bounding height of the node.
	 *
	 * @return bounding height
	 */
	public int getBoundingHeight()
	{
		int cc = getChildCount();
		int max, ch, i, w = 0;

		if (cc == 0)
			return getHeight();

		switch (getTreeLayoutModel().getOrientation())
		{
			case TreeDisplay.TOP:
			case TreeDisplay.BOTTOM:
				max = 0;
				for (i = 0; i < cc; i++)
				{
					ch = getChild(i).getBoundingHeight();
					if (ch > max)
						max = ch;
				}

				w = getHeight() + DefaultTreeLayoutModel.HEIGHT_SPACE_TB + max;
				
				break;

			case TreeDisplay.LEFT:
			case TreeDisplay.RIGHT:
				for (i = 0; i < cc; i++)
					w += getChild(i).getBoundingHeight();

				w += ((cc - 1) * DefaultTreeLayoutModel.HEIGHT_SPACE_LR);

				if (w < getHeight())
					w = getHeight();

				break;
		}

		return w;
	}

	/**
	 * Get the X coordinate of the top left corner of bounding box.
	 *
	 * @return an <code>int</code> value.
	 */
	public int getBoundingX()
	{
		int cc = getChildCount();
		int x = 0, i, min, ch;

		switch (getTreeLayoutModel().getOrientation())
		{
			case TreeDisplay.TOP:
			case TreeDisplay.BOTTOM:
				if (cc == 0)
					x = getX();
				else
					x = getChild(0).getBoundingX();
				break;
			case TreeDisplay.LEFT:
				x = getX();
				break;
			case TreeDisplay.RIGHT:
				if (cc == 0)
					x = getX();
				else
				{
					min = -1;
					for (i = 0; i < cc; i++)
					{
						ch = getChild(i).getBoundingX();
						if (min == -1) min = ch;
						if (ch < min) min = ch;
					}
					x = min;
				}
				break;
		}

		return x;
	}

	/**
	 * Get the Y coordinate of the top left corner of bounding box.
	 *
	 * @return an <code>int</code> value.
	 */
	public int getBoundingY()
	{
		int cc = getChildCount();
		int y = 0, min, ch, i;

		switch (getTreeLayoutModel().getOrientation())
		{
			case TreeDisplay.TOP:
				y = getY();
				break;
			case TreeDisplay.BOTTOM:
				if (cc == 0)
					y = getY();
				else
				{
					min = -1;
					for (i = 0; i < cc; i++)
					{
						ch = getChild(i).getBoundingY();
						if (min == -1) min = ch;
						if (ch < min) min = ch;
					}
					y = min;
				}
				break;
			case TreeDisplay.LEFT:
			case TreeDisplay.RIGHT:
				if (cc == 0)
					y = getY();
				else
					y = getChild(0).getBoundingY();
				break;
		}

		return y;
	}

}// DefaultTreeLayoutNode
