/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.treedisplay;

import java.util.ArrayList;
import javax.swing.event.EventListenerList;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;
import java.awt.Point;

/**
 * Partial implementation of the TreeLayoutModel class. Implementations
 * most methods are provided. Concrete subclasses of this only need to
 * provide an implementation of the {@link #layout} method.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 */

public abstract class AbstractTreeLayoutModel implements TreeLayoutModel {
	private EventListenerList listenerList;
	private Class layoutnodeclass;
	private int orientation = TreeDisplay.TOP;

	/**
	 * The root of the tree.
	 */
	protected TreeLayoutNode root;

	/**
	 * The TreeModel to layout.
	 */
	protected TreeModel model;

	//
	// abstract methods
	//

	// inherits doc comment
	public abstract void layout();
	public abstract TreeLayoutNode createNode(Object o);
	public abstract boolean supportsOrientation(int o);

	//
	// implemented methods
	//

	// inherits doc comment
	public int getOrientation()
	{
		return orientation;
	}

	// inherits doc comment
	public void setOrientation(int o) throws TreeDisplayException
	{
		if (supportsOrientation(o))
		{
			orientation = o;
			layout();
		}
		else
			throw new TreeDisplayException("Orientation " + o + " not supported.");
	}

	// inherits doc comment
	public Class getTreeLayoutNodeClass()
	{
		return layoutnodeclass;
	}

	// inherits doc comment
	public void setTreeLayoutNodeClass(Class c)
	{
		try {
			if (c.newInstance() instanceof TreeLayoutNode)
				layoutnodeclass = c;
		} catch (Exception e) {
			return;
		}

		if (model != null)
			layout();
	}

	// inherits doc comment
	public TreeModel getModel()
	{
		return model;
	}

	// inherits doc comment
	public void setModel(TreeModel m)
	{
		model = m;

		layout();
	}

	// inherits doc comment
	public int getWidth()
	{
		if (root != null)
			return root.getBoundingWidth();
		return 0;
	}

	// inherits doc comment
	public int getHeight()
	{
		if (root != null)
			return root.getBoundingHeight();
		return 0;
	}

	// inherits doc comment
	public TreeLayoutNode getRoot()
	{
		return root;
	}

	// inherits doc comment
	public TreeLayoutNode getNode(int x, int y)
	{
		return getNode(x, y, root);
	}

	// inherits doc comment
	public TreeLayoutNode getNode(Point p)
	{
		return getNode(p.x, p.y, root);
	}

	// inherits doc comment
	public TreeLayoutNode getNode(Object data)
	{
		return getNode(data, root);
	}

	// inherits doc comment
	public TreePath getTreePath(TreeLayoutNode n)
	{
		ArrayList ar = new ArrayList(20);

		getPath(n, ar);

		return new TreePath(ar.toArray());
	}

	// inherits doc comment
	public void addTreeModelListener(TreeModelListener l)
	{
		if (listenerList == null)
		{
			listenerList = new EventListenerList();
		}

		listenerList.add(TreeModelListener.class, l);
	}

	// inherits doc comment
	public void removeTreeModelListener(TreeModelListener l)
	{
		if (listenerList == null)
			return;

		listenerList.remove(TreeModelListener.class, l);
	}
	
	//
	// implementation of TreeModelListener methods
	//

	// inherits doc comment
	public void treeNodesChanged(TreeModelEvent e)
	{
		layout();
		notifyListeners(e, "treeNodesChanged");
	}

	// inherits doc comment
	public void treeNodesInserted(TreeModelEvent e)
	{
		layout();
		notifyListeners(e, "treeNodesInserted");
	}

	// inherits doc comment
	public void treeNodesRemoved(TreeModelEvent e)
	{
		layout();
		notifyListeners(e, "treeNodesRemoved");
	}

	// inherits doc comment
	public void treeStructureChanged(TreeModelEvent e)
	{
		layout();
		notifyListeners(e, "treeStructureChanged");
	}
	
	//
	// private methods
	//

	// return the layout node that contains the point (x, y) starting
	// at node current.
	private TreeLayoutNode getNode(int x, int y, TreeLayoutNode current)
	{
		if (current == null)
			return null;
		
		if (current.contains(x, y))
			return current;
		
		for (int i = 0; i < current.getChildCount(); i++) {
			TreeLayoutNode n = getNode(x, y, current.getChild(i));
			if (n != null)
				return n;
		}

		return null;
	}

	private TreeLayoutNode getNode(Object data, TreeLayoutNode current)
	{
		if (current == null)
			return null;
		if (current.getData() == data)
			return current;
		for (int i = 0; i < current.getChildCount(); i++)
		{
			TreeLayoutNode n = getNode(data, current.getChild(i));
			if (n != null)
				return n;
		}

		return null;
	}

	// used in constructing a TreePath
	private void getPath(TreeLayoutNode node, ArrayList ar)
	{
		if (node == null)
			return;

		TreeLayoutNode parent = node.getParent();
		if (parent == null)
		{
			ar.add(node.getData());
		}
		else
		{
			getPath(parent, ar);
			ar.add(node.getData());
		}
	}

	// notify listeners that something in the layout model has changed
	private void notifyListeners(TreeModelEvent e, String action)
	{
		if (listenerList != null)
		{
			Object[] listeners = listenerList.getListenerList();
			
			for (int i = listeners.length - 2; i >= 0; i -= 2)
			{
				if (action.equals("treeNodesChanged"))
					((TreeModelListener)listeners[i+1]).treeNodesChanged(e);
				else if (action.equals("treeNodesInserted"))
					((TreeModelListener)listeners[i+1]).treeNodesInserted(e);
				else if (action.equals("treeNodesRemoved"))
					((TreeModelListener)listeners[i+1]).treeNodesRemoved(e);
				else if (action.equals("treeStructureChanged"))
					((TreeModelListener)listeners[i+1]).treeStructureChanged(e);
			}
		}
	}
}// AbstractTreeLayoutModel
