/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.explain;

import java.util.*;
import java.io.*;

/**
 * The <code>ExplainTree</code> class represents a plan tree
 * generated by PostgreSQL. Tree nodes are instances of {@link
 * com.redhat.rhdb.explain.ExplainTreeNode ExplainTreeNode}.
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 */
public class ExplainTree implements Serializable {
	private ExplainTreeNode root;
	
	/**
	 * Creates a new <code>ExplainTree</code> instance.
	 *
	 */
	public ExplainTree()
	{
		root = null;
	}

	/**
	 * Creates a new <code>ExplainTree</code> instance with a specific root.
	 *
	 * @param r an <code>ExplainTreeNode</code> value
	 */
	public ExplainTree(ExplainTreeNode r)
	{
		root = r;
	}

	/**
	 * Gets the root of the tree.
	 *
	 * @return an <code>ExplainTreeNode</code> value
	 */
	public ExplainTreeNode getRoot()
	{
		return root;
	}

	/**
	 * Sets the root of the tree.
	 *
	 * @param r an <code>ExplainTreeNode</code> value
	 */
	public void setRoot(ExplainTreeNode r)
	{
		root = r;
	}

	/**
	 * Checks if a given node is in the tree.
	 *
	 * @param n an <code>ExplainTreeNode</code> value
	 * @return a <code>boolean</code> value
	 */
	public boolean inTree(ExplainTreeNode n)
	{
		return locate(n, root);
	}

	/**
	 * Is the tree empty?
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean isEmpty()
	{
		if (root == null)
		{
			return true;
		}
		return false;
	}
	
	/**
	 * Return a String representation of the tree.
	 *
	 * @return a <code>String</code> value
	 */
	public String toString()
	{
		StringWriter strwriter = new StringWriter();
		PrintWriter printer = new PrintWriter(strwriter);
		
		printTree(root, "", printer);

		return strwriter.toString().trim();
	}

	//
	// private methods
	//

	/*
	 * Find a specific node starting at current. DFS
	 */
	private boolean locate(ExplainTreeNode node, ExplainTreeNode current)
	{
		boolean found = false;
		
		if (current == node)
		{
			found = true;
		}
		else
		{
			Enumeration en = current.getChildren();
			while (en.hasMoreElements() && !found)
			{
				found = locate(node, (ExplainTreeNode) en.nextElement());
				if (found) break;
			}
		}
		
		return found;
	}
	
	/*
	 * Print the tree
	 */
	private void printTree(ExplainTreeNode node, String indent, PrintWriter printer)
	{
		printer.print(indent + node.getName());
		printer.print(" [ ");

		Set s = node.getDetailTypes();
		Iterator it = s.iterator();
		while (it.hasNext())
		{
			String str = (String) it.next();
			printer.print(str + ": ");
			printer.print(node.getDetail(str) + " ");
		}
		printer.println("]");

		Enumeration en = node.getChildren();
		while (en.hasMoreElements())
		{
			printTree((ExplainTreeNode) en.nextElement(), indent + "  ", printer);
		}
	}
}// ExplainTree
