// File:	TopoDS_Builder.cxx
// Created:	Tue Apr  9 18:25:03 1991
// Author:	Remi LEQUETTE
//		<rle@phobox>
 

#include <TopoDS_Builder.ixx>

#include <TopoDS_TShape.hxx>
#include <TopoDS_TWire.hxx>
#include <TopoDS_TCompound.hxx>
#include <TopoDS_ListIteratorOfListOfShape.hxx>

#include <TopoDS_FrozenShape.hxx>
#include <TopoDS_UnCompatibleShapes.hxx>

//=======================================================================
//function : TopoDS_Builder
//purpose  : constructor
//=======================================================================

TopoDS_Builder::TopoDS_Builder ()
{
}

//=======================================================================
//function : MakeShape
//purpose  : Make a Shape from a TShape
//=======================================================================

void TopoDS_Builder::MakeShape (TopoDS_Shape& S, 
				const Handle(TopoDS_TShape)& T) const
{
  S.TShape(T);
  S.Location(TopLoc_Location());
  S.Orientation(TopAbs_FORWARD);
}

//=======================================================================
//function : MakeWire
//purpose  : Make an empty wire
//=======================================================================

void TopoDS_Builder::MakeWire (TopoDS_Wire& W) const
{
  Handle(TopoDS_TWire) TW = new TopoDS_TWire();
  MakeShape(W,TW);
}

//=======================================================================
//function : MakeCompound
//purpose  : Make an empty Compound
//=======================================================================

void TopoDS_Builder::MakeCompound (TopoDS_Compound& C) const
{
  Handle(TopoDS_TCompound) TC = new TopoDS_TCompound();
  MakeShape(C,TC);
}


//=======================================================================
//function : Add
//purpose  : insert a Shape in an Other
//=======================================================================

void TopoDS_Builder::Add (TopoDS_Shape& aShape, 
			  const TopoDS_Shape& aComponent) const
{

  // From now the Component cannot be edited
  aComponent.TShape()->Free(Standard_False);

  // Note that freezing aComponent before testing if aShape is free
  // prevents from self-insertion
  // but aShape will be frozen when the Exception is raised
  TopoDS_FrozenShape_Raise_if(!aShape.Free(),"TopoDS_Buider::Add");

  // check the compatibility
  TopAbs_ShapeEnum ComponentType = aComponent.ShapeType();

  switch (aShape.ShapeType()) {

    case TopAbs_COMPOUND :
      break;

    case TopAbs_COMPSOLID :
      if (ComponentType != TopAbs_SOLID) {
	TopoDS_UnCompatibleShapes::Raise
	  ("TopoDS_Builder::Add:insertion of non SOLID in COMPSOLID");
	}
      break;

    case TopAbs_SOLID :
      if ((ComponentType != TopAbs_VERTEX) &&
	  (ComponentType != TopAbs_EDGE)   &&
	  (ComponentType != TopAbs_SHELL)) {
	TopoDS_UnCompatibleShapes::Raise
	  ("TopoDS_Builder::Add:insertion of non VERTEX, EDGE, SHELL in SOLID");
	}
      break;

    case TopAbs_SHELL :
      if (ComponentType != TopAbs_FACE) {
	TopoDS_UnCompatibleShapes::Raise
	  ("TopoDS_Builder::Add:insertion of non FACE in SHELL");
	}
      break;

    case TopAbs_FACE :
      if ((ComponentType != TopAbs_VERTEX) &&
	  (ComponentType != TopAbs_WIRE)) {
	TopoDS_UnCompatibleShapes::Raise
	  ("TopoDS_Builder::Add:insertion of non VERTEX, WIRE in FACE");
	}
      break;

    case TopAbs_WIRE :
      if (ComponentType != TopAbs_EDGE) {
	TopoDS_UnCompatibleShapes::Raise
	  ("TopoDS_Builder::Add:insertion of non EDGE in WIRE");
	}
      break;

    case TopAbs_EDGE :
      if (ComponentType != TopAbs_VERTEX) {
	TopoDS_UnCompatibleShapes::Raise
	  ("TopoDS_Builder::Add:insertion of non VERTEX in EDGE");
	}
      break;

    case TopAbs_VERTEX :
      TopoDS_UnCompatibleShapes::Raise
	("TopoDS_Builder::Add:insertion of VERTEX in VERTEX");
      break;
      
    case TopAbs_SHAPE :
      default :
	TopoDS_UnCompatibleShapes::Raise
	  ("TopoDS_Builder::Add: shape with type SHAPE ????");
      break;
    }

  TopoDS_ListOfShape& L = aShape.TShape()->ChangeShapes();
  L.Append(aComponent);
  TopoDS_Shape& S = L.Last();
  
  // compute the relative Orientation
  if (aShape.Orientation() == TopAbs_REVERSED) {
    S.Reverse();
  }

  // and the Relative Location
  S.Move(aShape.Location().Inverted());

  // Set the TShape as modified.
  aShape.TShape()->Modified(Standard_True);
}

//=======================================================================
//function : Remove
//purpose  : Remove a Shape from an other one
//=======================================================================

void TopoDS_Builder::Remove (TopoDS_Shape& aShape, 
			     const TopoDS_Shape& aComponent) const
{
  // check  if aShape  is  not Frozen
  TopoDS_FrozenShape_Raise_if (!aShape.Free(),"TopoDS_Builder::Remove");
  
  // compute the relative Orientation and Location of aComponent
  TopoDS_Shape S = aComponent;
  if (aShape.Orientation() == TopAbs_REVERSED)
    S.Reverse();
  S.Location(S.Location().Predivided(aShape.Location()));

  TopoDS_ListOfShape& L = aShape.TShape()->ChangeShapes();
  TopoDS_ListIteratorOfListOfShape It(L);
  while (It.More()) {
    if (It.Value() == S) {
      L.Remove(It);
      aShape.TShape()->Modified(Standard_True);
      break;
    }
    It.Next();
  }
}
