/* ============================================================
 * File  : streamstatus.cpp
 * Author: Eric Giesselbach <ericgies@kabelfoon.nl>
 * Date  : 2003-12-22
 * Description : av stream handling
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// development snapshot, many todo's

#include <iostream>

#include <qapplication.h>
#include <qnetwork.h>
#include <qdatetime.h>
#include <qpainter.h>
#include <qdir.h>
#include <qtimer.h>
#include <qprocess.h>
#include <qstringlist.h>
#include <qdict.h>
#include <qvbox.h>
#include <qhbox.h>
#include <qlayout.h>
#include <qregexp.h>
#include <qurl.h>
#include <math.h>


#include "streamstatus.h"

void myX11MapWindow (uint id);
void myX11UnmapWindow (uint id);
uint myX11GrabKeyboard(uint id);
uint myX11UngrabKeyboard();

//void listenXEvents(uint windowid, XEvent *event);

using namespace std;

// ----------------------------------------------------------------------


VideoArea::VideoArea(QWidget *parent, const char *name, WFlags f)
               : QWidget(parent, name, f)
{
   myParent = parent;
   //setFocusPolicy(QWidget::StrongFocus);
}

void VideoArea::focusInEvent ( QFocusEvent *ev )
{
   QWidget::focusInEvent(ev);
}

void VideoArea::focusOutEvent ( QFocusEvent *ev )
{
   QWidget::focusOutEvent(ev);
}

VideoContainer::VideoContainer(QWidget *parent, const char *name, WFlags f)
               : QWidget(parent, name, f)
{
    hide();
    
    par = parent;
    setCaption(name);
    setBackgroundMode(NoBackground);

    vidArea = new VideoArea(this, "videoArea", 0);
    setBackgroundColor( Qt::black );
    vidArea->show();

    lastZoom = false;
    keyboardGrabbed = false;

    hide();
}

QString VideoContainer::getVidWindowId()
{
    return QString::number(vidArea->winId(), 10);
}

int VideoContainer::getMyWindowId()
{
    return vidArea->winId();
}

void VideoContainer::goFullscreen(double aspectRatio, bool zoom)
{
    setGeometry( screenRect );
    vidArea->setGeometry( calcVideoRectFromAspect(aspectRatio, screenRect, zoom) );
    lastZoom = zoom;
    
    if (!keyboardGrabbed)
      keyboardGrabbed = ( myX11GrabKeyboard(getMyWindowId()) == 0 );
}

void VideoContainer::showEvent( QShowEvent * )
{
   keyboardGrabbed = false;
}

void VideoContainer::goPreview(double aspectRatio, bool zoom)
{
    setGeometry( previewRect );
    vidArea->setGeometry( calcVideoRectFromAspect(aspectRatio, previewRect, zoom) );

    if (!keyboardGrabbed)
      keyboardGrabbed = ( myX11GrabKeyboard(getMyWindowId()) == 0 );
}

void VideoContainer::setPreviewRect(QRect rec)
{
   if (rec.height() < 1)
     rec = QRect(0, 0, 200, 150);
   
   previewRect = rec;
}

void VideoContainer::setScreenRect(QRect rec)
{
    if (rec.height() < 1)
      rec = QRect(0, 0, 400, 300);
      
    screenRect = rec;
}

bool VideoContainer::previouslyZoomed()
{
   return lastZoom;
}

QRect VideoContainer::calcVideoRectFromAspect(double vaspect, QRect targetRect, bool zoom)
{
   int vheight = 0;
   int vwidth = 0;
   int vdelta = 0;
   
   double saspect;
   QRect vRect;

   if (vaspect == 0)
     vRect = targetRect;
   else
   {
      saspect = (double)targetRect.width()/targetRect.height();
      
   	if ( (vaspect > saspect) != zoom )
   	{
			vRect.setWidth( targetRect.width() );
			vRect.setLeft(0);
			vheight = int( (double)targetRect.width() / vaspect );
			vdelta  = int( (targetRect.height() - vheight)/2 );
			vRect.setTop(vdelta);
			vRect.setHeight(vheight);
   	}
   		else
   	{
			vRect.setHeight( targetRect.height() );
			vRect.setTop(0);
			vwidth = int( (double)targetRect.height() * vaspect );
			vdelta = int( (targetRect.width() - vwidth)/2 );
			vRect.setLeft(vdelta);
			vRect.setWidth(vwidth);
   	}
   }

   return vRect;
}

// ----------------------------------------------------------------------

StreamStatus::StreamStatus(QObject *parent)
{
    myParent = parent;
    proc = NULL;

    m_videoContainer = 0;
    m_fullscreen = Preview;

    streamUrl   = "";
    streamName  = "";
    streamDescr = "";
    streamHandler = "";

    lastPlayedUrl   = "";
    lastPlayedName  = "";
    lastPlayedDescr = "";
    lastPlayedHandler = "";
    lastPlayedConsole = "";

    pending = false;

    playerEncap = new PlayerEncap();

    setStatus(idle);

    QTimer *timer = new QTimer( this );
    connect( timer, SIGNAL(timeout()), this, SLOT(timerEvent()) );
    timer->start( 1000, false );

}

StreamStatus::~StreamStatus()
{
    if (proc)
    {
      if ( proc->isRunning() ) proc->tryTerminate();
      //QTimer::singleShot( 1000, proc, SLOT( kill() ) );
    }

    delete( playerEncap );
}


void StreamStatus::timerEvent()
{
    if ( setIdleOnZero > 0 && --setIdleOnZero == 0)
    {
      setStatus(idle);
      emit pollSignal();
    }

    if ( getStatus() != idle )
      emit pollSignal();
}


void StreamStatus::setVideoContainer(VideoContainer *vid)
{
    m_videoContainer = vid;
}

QString StreamStatus::getVideoWindowId()
{
    if ( videoSet() )
      return m_videoContainer->getVidWindowId();
    else
      return "";
}

QString& StreamStatus::getStreamUrl()
{
    return streamUrl;
}

QString& StreamStatus::getStreamName()
{
    return streamName;
}

QString& StreamStatus::getStreamDescr()
{
    return streamDescr;
}

QString& StreamStatus::getStreamHandler()
{
    return streamHandler;
}


QString& StreamStatus::getLastPlayedUrl()
{
    return lastPlayedUrl;
}

QString& StreamStatus::getLastPlayedName()
{
    return lastPlayedName;
}

QString& StreamStatus::getLastPlayedDescr()
{
    return lastPlayedDescr;
}

QString& StreamStatus::getLastPlayedHandler()
{
    return lastPlayedHandler;
}

// should reconsider this construction, e.g. push configured
// custom properties ( = optional, not linked to specific functions)
// added remark: what did I try to say here?

// these parameters can occur more than once in player output
bool StreamStatus::checkCustomStreamInfoExists(const QString& info)
{
  return playerEncap->checkStreamPropertyConfigured( info );
}

QString StreamStatus::getCustomStreamInfo(const QString& info)
{
   return playerEncap->getStreamProperty( info, true );
}

QString StreamStatus::getCustomStreamArea(const QString& info)
{
   return playerEncap->getStreamPropertyArea( info );
}

QString StreamStatus::getCustomStreamTitle(const QString& info)
{
   return playerEncap->getStreamPropertyTitle( info );
}

// these parameters are considered to occur once per player run
// if parameter occurs more often, the last value is stored
QString StreamStatus::getStreamInfo(StreamInfo info)
{
    QRegExp reg;
    QString text;

    text = "";

    switch (info)
    {
      case StreamVideoFormat:
        text = playerEncap->getStreamProperty("StreamVideoFormat", false);
        break;
      case StreamVideoWidth:
        text = playerEncap->getStreamProperty("StreamVideoWidth", false);
        break;
      case StreamVideoHeight:
        text = playerEncap->getStreamProperty("StreamVideoHeight", false);
        break;
      case StreamLength:
        text = playerEncap->getStreamProperty("StreamLength", false);
        break;
      case StreamVideoFps:
        text = playerEncap->getStreamProperty("StreamVideoFps", false);
        break;
      case StreamTime:
        text = playerEncap->getStreamProperty("StreamTime", false);
        break;
      case StreamCache:
        if ( getStatus() == buffering )
          text = playerEncap->getStreamProperty("StreamBufferCache", false);
        else
          text = playerEncap->getStreamProperty("StreamPlayCache", false);
        break;
      case StreamStability:
        text = getStreamStability();
        break;
      case StreamFilename:
        text = playerEncap->getStreamProperty("StreamFilename", false);
        break;
      case StreamAudioCodec:
        text = playerEncap->getStreamProperty("StreamAudioCodec", false);
        break;
      case StreamAudioFormat:
        text = playerEncap->getStreamProperty("StreamAudioFormat", false);
        break;
      case StreamVideoCodec:
        text = playerEncap->getStreamProperty("StreamVideoCodec", false);
        break;
      case StreamBitrate:
        text = playerEncap->getStreamProperty("StreamBitrate", false);
        break;
      case StreamVideoBitrate:
        text = playerEncap->getStreamProperty("StreamVideoBitrate", false);
        break;
      case StreamRate:
        text = playerEncap->getStreamProperty("StreamRate", false);
        break;
      case StreamChannels:
        text = playerEncap->getStreamProperty("StreamChannels", false);
        break;
      case StreamVolume:
        text = playerEncap->getStreamProperty("StreamVolume", false);
        break;
    }

    return text;
}

double StreamStatus::getVideoAspect()
{
   int vheight = 0;
   int vwidth = 0;
      
   QString str;

   str = getStreamInfo(StreamVideoWidth);
   if (str != "") vwidth = str.toInt();
   
   str = getStreamInfo(StreamVideoHeight);
   if (str != "") vheight = str.toInt();
   
   if (vheight == 0)
     return (double)3/2;
   else
     return (double)vwidth/vheight;
}

void StreamStatus::setStatus(Status stat)
{
    setIdleOnZero = 0;
    Status oldstatus = status;
    status = stat;

    switch (stat)
    {
      case idle:
      {
        reset();
        statusString = "idle";
        
        // limit player dump size
        if ( lastPlayedConsole.length() > 20000 ) 
          lastPlayedConsole = lastPlayedConsole.right(20000);
        
        if ( status != oldstatus) emit statusChange();
        break;
      }
      case starting:
      {
        reset();
        statusString = "starting";
        lastPlayedConsole += "\n***" + QString(TARGET": starting " + pendingName + ", " + pendingUrl + "***\n\n");
        if ( status != oldstatus) emit statusChange();
        break;
      }
      case videoinit:
      {
   // todo: proper video layout buildup in gui
        if ( videoSet() )
        {
          video = true;
          statusString = "videoinit";
          if ( status != oldstatus) emit statusChange();
          m_videoContainer->show();
        }
	  else
            if ( status != oldstatus) emit statusChange();
        break;
      }
      case buffering:
      {
        statusString = "buffering";
        if ( status != oldstatus) emit statusChange();
        break;
      }
      case playing:
      {
        if (oldstatus != paused)
        {
				if (video)
				{
					videoAspect = getVideoAspect();
				   m_videoContainer->goPreview(videoAspect, false);
				}		
				lastPlayedUrl   = streamUrl;
				lastPlayedName  = streamName;
				lastPlayedDescr = streamDescr;
				lastPlayedHandler = streamHandler;
				lastPlayedConsole += "\n***" + QString(TARGET": stream playing***\n\n");
				if (video && videoSet() )
					myX11MapWindow(m_videoContainer->getMyWindowId());
        }
        statusString = "playing";
        if ( status != oldstatus) emit statusChange();
        break;
      }
      case paused:
      {
        statusString = "paused";
        if ( status != oldstatus) emit statusChange();
        break;
      }
      case stopping:
      {
        statusString = "stopping";
        if ( status != oldstatus) emit statusChange();
        break;
      }
      case stopped:
      {
        if ( videoSet() )
          m_videoContainer->hide();
        video = false;
        statusString = "stopped";
        setIdleOnZero = 1;
        lastPlayedConsole += "\n***" + QString(TARGET": stream stopped***\n\n");
        if ( status != oldstatus) emit statusChange();
        break;
      }
      case failed:
      {
        if ( videoSet() ) // to be sure
          m_videoContainer->hide();
        video = false;
        statusString = "failed";
        setIdleOnZero = 5;
        if ( status != oldstatus) emit statusChange();
        lastPlayedConsole += "\n***" + QString(TARGET": stream failed***\n\n");
        break;
      }
      case nostream: //player exited on status starting
      {
        if ( videoSet() )
          m_videoContainer->hide();
        video = false;
        statusString = "no stream";
        setIdleOnZero = 5;
        lastPlayedConsole += "\n***" + QString(TARGET": no stream***\n\n");
        if ( status != oldstatus) emit statusChange();
        break;
      }
      default:
      {
        if ( status != oldstatus) emit statusChange();
        break;
      }
    }

}

void StreamStatus::setVideoRect(QRect& rect)
{
    if ( videoSet() )
    {
    	m_videoContainer->setPreviewRect(rect);
    }
}

void StreamStatus::setVideoMaxRect(QRect& rect)
{
	if ( videoSet() )
	{
		m_videoContainer->setScreenRect(rect);
	}
}

QString& StreamStatus::getStatusString()
{
    return statusString;
}

StreamStatus::Status& StreamStatus::getStatus()
{
    return status;
}

// parse player output by playerencap
void StreamStatus::parsePlayerOutput(const QString msg)
{
    QString messageLabel;
    QString messageData;

    //cout << "player out: " << msg << endl;
    //  [KVolume: 16 %
    
 // cout << "player prop: " << label << ": " << value << endl;
    if ( getStatus() != playing )
      lastPlayedConsole += msg + "\n";
    
    int i = msg.find(QRegExp( playerEncap->getPlayerSys("separator") ), 0);
    if (i != -1)
    {
      messageLabel = msg.left(i);
      messageData  = msg.mid(i+1, 999);
            
      // returns player.xml filters.item.statuschange property
      Status newstat = (Status)playerEncap->pushParameter(messageLabel, messageData);
      
      if (newstat != none && newstat != status)
        setStatus(newstat);

      if ( status == playing ) // assuming cache information in every update during play
      {
        QString cache = playerEncap->getCurrentCacheUsage();
        reportStreamCache( cache );
      }
    }
}

QString& StreamStatus::getLastPlayedConsole()
{
  return lastPlayedConsole;
}

void StreamStatus::appendLastPlayedConsole(QString line)
{
   lastPlayedConsole += "\n***" + QString(TARGET"" + line + "***\n\n");
}


void StreamStatus::reportStreamCache(QString& cacheValue)
{
    bool check;
    int val;
    val = cacheValue.toInt(&check);
    if (check)
    {
      cacheval[cacheind++] = val;
      if (cacheind>50) cacheind = 0;
    }
}

const QString StreamStatus::getStreamStability()
{
    //double tot, sum;
    float tot, sum, res;
    tot = 0;
    sum = 0;
    for (int i=0; i<50; i++)
    {
      tot += cacheval[i];
      sum += cacheval[i] * cacheval[i];
    }
    tot = tot / 50;
    res = (sum / 50) - (tot * tot);
    res = sqrt( res ); // res = stddev
    res = tot - res;   // avg-stddev<=0: risking empty buffer
    if (res < 0) res = 0;
    if (tot != 0)
      res = (res / tot) * 100;  // stability
    else
      res = 0;                  // empty buffer or local stream
    return QString::number( res, 'f', 0 );
}

bool StreamStatus::isVideo()
{
    return video;
}

bool StreamStatus::videoSet()
{
   if (m_videoContainer )
     return true;
   else
     return false;
}

void StreamStatus::reset()
{
    playerEncap->reset();

    streamUrl   = "";
    streamDescr = "";
    streamHandler = "";
    streamName  = "";
    
    for (int i=0; i<50; i++)
      cacheval[i] = 0;
    cacheind = 0;

    video = false;
    if ( m_fullscreen != Preview )
      issueCommand(DefaultScreen);
}



void StreamStatus::initStream(QString url, QString name, QString descr, QString handler)
{
    pendingUrl   = url;
    pendingName  = name;
    pendingDescr = descr;
    pendingHandler = handler;
    
    pending = true;

    if (proc)
      stopStream();
    else
      startStream();
}

bool StreamStatus::isPending()
{
    return pending;
}

// useLastZoom = false: toggles preview -> fullscreen -> fullscreen zoom
// useLastZoom = true: switch to fullscreen, zoom if previous video was zoomed
void StreamStatus::toggleFullScreen(bool useLastZoom)
{
      switch (m_fullscreen)
      {
         case Preview:
           if ( useLastZoom && videoSet() )
           {
               m_videoContainer->goFullscreen(videoAspect, m_videoContainer->previouslyZoomed());
               if ( m_videoContainer->previouslyZoomed() )
                 m_fullscreen = Zoom;
               else
                 m_fullscreen = Full;
           }
             else issueCommand(FullScreen);
         break;
         case Full:
           issueCommand(FullScreen);
         break;
         case Zoom:
           issueCommand(DefaultScreen);
         break;
      }
}

bool StreamStatus::isFullScreen()
{
   return (m_fullscreen != Preview) && video;
}

void StreamStatus::issueCommand(StreamCommand command)
{
  switch (command)
  {
    case VolumeUp:
      playerCommand( playerEncap->getPlayerCmd("volumeup") );
      break;
    case VolumeDn:
      playerCommand( playerEncap->getPlayerCmd("volumedn") );
      break;
    case AVInc:
      playerCommand( playerEncap->getPlayerCmd("avinc") );
      break;
    case AVDec:
      playerCommand( playerEncap->getPlayerCmd("avdec") );
      break;
    case Forward:
      playerCommand( playerEncap->getPlayerCmd("forward") );
      break;
    case Rewind:
      playerCommand( playerEncap->getPlayerCmd("rewind") );
      break;
    case Mute:
      playerCommand( playerEncap->getPlayerCmd("mute") );
      break;
    case Pause:
      playerCommand( playerEncap->getPlayerCmd("pause") );
      break;
    case FullScreen:
      if ( videoSet() )
      {
        //playerCommand( playerEncap->getPlayerCmd("fullscreen") );
        if (m_fullscreen == Full)
        {
          m_videoContainer->goFullscreen(videoAspect, true);
          m_fullscreen = Zoom;
        }
        else
        {
          m_videoContainer->goFullscreen(videoAspect, false);
          m_fullscreen = Full;
        }
      }
      break;
    case DefaultScreen:
      if ( videoSet() )
      {
        m_videoContainer->goPreview(videoAspect, false);
      }
      m_fullscreen = Preview;
      break;
  }
}


void StreamStatus::startStream()
{

    if ( proc ) return;

    setStatus(starting);
    streamUrl   = pendingUrl;
    streamName  = pendingName;
    streamDescr = pendingDescr;
    streamHandler = pendingHandler;

    proc = new QProcess( this );
    proc->setCommunication( QProcess::Stdin |
                            QProcess::Stdout|
                            QProcess::Stderr|
                            QProcess::DupStderr );

//    proc->addArgument( "sh" );
//    proc->addArgument( "-c" );
//    proc->addArgument( "tail -c 100000000000000 -f /data/linuxhome/files/lessig | mplayer -" );

    proc->addArgument( playerEncap->getPlayerSys("player") );

    if ( videoSet() )
    {
      proc->addArgument( playerEncap->getPlayerSys("window") );
      proc->addArgument( getVideoWindowId() );
      //proc->addArgument( playerEncap->getPlayerSys("scale") );
      //proc->addArgument( QString::number(m_videorect.width()) );
    }

    PlayerEncap::StringMap::Iterator i;
    for ( i = playerEncap->playerParam.begin();
          i != playerEncap->playerParam.end(); ++i )
    {
       proc->addArgument( i.key()  );
       if ( i.data() != "" )
         proc->addArgument( i.data() );
    }
    
    // dump audio to tempfile
    QString username = QString(getenv("USER"));
    proc->addArgument( "-af" );
    proc->addArgument( "export=/tmp/mplayer-af_export_" + username );
    
    QString file = QUrl(streamUrl).fileName(); 
    // file == "" matches find below
    if ( file != "" && QString(""PLAYLIST).find(file.right(4), 0, false) != -1 )
    {
       proc->addArgument( "-playlist" );
    }
    
    proc->addArgument( streamUrl );

    pending = false;

    connect( proc, SIGNAL(readyReadStderr()),
            this, SLOT(readFromStderr()) );
    connect( proc, SIGNAL(readyReadStdout()),
            this, SLOT(readFromStdout()) );
    connect( proc, SIGNAL(processExited()),
            this, SLOT(streamExited()) );

    if ( !proc->start() )
    {
      fprintf( stderr, "error starting player\n" );
      setStatus(failed);
    }

}

void StreamStatus::stopStream()
{
   if ( proc && proc->isRunning() )
   {
      setStatus(stopping);
      proc->tryTerminate();
      QTimer::singleShot( 7000, proc, SLOT(kill()) );
   }
}

void StreamStatus::playerCommand(QString command)
{
   if ( proc )
   {
     //cout << "player command: " << command << endl;
     proc->writeToStdin(command + "\n");
   }
}

void StreamStatus::readFromStdout()
{
   QString val = "";
   QString temp = " ";
   while ( temp != "" )
   {
     temp = QString(proc->readStdout());
     val += temp;
   }
   QStringList lines = QStringList::split( QRegExp("[\\0033\\r\\n]"), val );
   for ( QStringList::iterator line = lines.begin();
         line != lines.end(); ++line )
   {
      temp = *line;
      if ( temp.find("[") == 0 )
        temp = temp.remove(0,2);
      parsePlayerOutput(temp);
   }
}

void StreamStatus::readFromStderr()
{
  // is rerouted to stdout
}

void StreamStatus::streamExited()
{
   delete proc;
   proc = NULL;

   if ( getStatus() == starting | getStatus() == buffering )
     setStatus(nostream);  // nothing to play,pause or videoinit = no stream
   else
     setStatus(stopped);

   if ( isPending() ) startStream();
}


