/*
 * Copyright (C) MX4J.
 * All rights reserved.
 *
 * This software is distributed under the terms of the MX4J License version 1.0.
 * See the terms of the MX4J License in the documentation provided with this software.
 */

package mx4j.tools.remote.soap;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceAlreadyExistsException;
import javax.management.InstanceNotFoundException;
import javax.management.IntrospectionException;
import javax.management.InvalidAttributeValueException;
import javax.management.ListenerNotFoundException;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanRegistrationException;
import javax.management.NotCompliantMBeanException;
import javax.management.ObjectInstance;
import javax.management.ObjectName;
import javax.management.ReflectionException;
import javax.management.remote.JMXServiceURL;
import javax.management.remote.NotificationResult;
import javax.security.auth.Subject;

import mx4j.tools.remote.Connection;
import org.apache.axis.MessageContext;

/**
 *
 * @author <a href="mailto:biorn_steedom@users.sourceforge.net">Simone Bordet</a>
 * @version $Revision: 1.4 $
 */
public class SOAPConnectionPivot implements Connection, SOAPConnection
{
   private final Map connections = new HashMap();
   private SOAPConnectionManager connectionManager;

   public String connect(Object credentials) throws IOException, SecurityException
   {
      JMXServiceURL address = retrieveJMXServiceURL();

      // Lookup the ConnectionManager
      synchronized (this)
      {
         if (connectionManager == null)
         {
            connectionManager = SOAPConnectorServer.find(address);
            if (connectionManager == null) throw new IOException("Could not find ConnectionManager. Make sure a SOAPConnectorServer is in classloader scope and bound at this address " + address);
         }
      }
      Connection connection = connectionManager.connect(credentials);
      String id = connection.getConnectionId();

      // Need to keep connections here, otherwise they're garbage collected
      synchronized (connections)
      {
         connections.put(id, connection);
      }

      return id;
   }

   private JMXServiceURL retrieveJMXServiceURL() throws MalformedURLException
   {
      MessageContext context = MessageContext.getCurrentContext();
      String url = (String)context.getProperty(MessageContext.TRANS_URL);
      JMXServiceURL temp = new JMXServiceURL("service:jmx:" + url);
      return new JMXServiceURL("soap", "localhost", temp.getPort(), temp.getURLPath());
   }

   private String retrieveConnectionID()
   {
      MessageContext context = MessageContext.getCurrentContext();
      return (String)context.getProperty(SOAPConstants.CONNECTION_ID_HEADER_NAME);
   }

   private SOAPConnection findConnection() throws IOException
   {
      String id = retrieveConnectionID();
      SOAPConnection connection = null;
      synchronized (connections)
      {
         connection = (SOAPConnection)connections.get(id);
      }
      if (connection == null) throw new IOException("Could not find connection with ID " + id);
      return connection;
   }

   public void close() throws IOException
   {
      String id = retrieveConnectionID();
      Connection connection = null;
      synchronized (connections)
      {
         connection = (Connection)connections.remove(id);
      }
      if (connection == null) throw new IOException("Could not find connection with ID " + id);
      connection.close();
   }

   public String getConnectionId() throws IOException
   {
      Connection connection = (Connection)findConnection();
      return connection.getConnectionId();
   }

   public ObjectInstance createMBean(String className, ObjectName name, Object[] params, String[] signature, Subject delegate)
           throws ReflectionException,
           InstanceAlreadyExistsException,
           MBeanRegistrationException,
           MBeanException,
           NotCompliantMBeanException,
           IOException
   {
      SOAPConnection connection = findConnection();
      return connection.createMBean(className, name, params, signature, delegate);
   }

   public ObjectInstance createMBean(String className, ObjectName name, ObjectName loaderName, Object[] params, String[] signature, Subject delegate)
           throws ReflectionException,
           InstanceAlreadyExistsException,
           MBeanRegistrationException,
           MBeanException,
           NotCompliantMBeanException,
           InstanceNotFoundException,
           IOException
   {
      SOAPConnection connection = findConnection();
      return connection.createMBean(className, name, loaderName, params, signature, delegate);
   }

   public void unregisterMBean(ObjectName name, Subject delegate) throws InstanceNotFoundException, MBeanRegistrationException, IOException
   {
      SOAPConnection connection = findConnection();
      connection.unregisterMBean(name, delegate);
   }

   public ObjectInstance getObjectInstance(ObjectName name, Subject delegate) throws InstanceNotFoundException, IOException
   {
      SOAPConnection connection = findConnection();
      return connection.getObjectInstance(name, delegate);
   }

   public Set queryMBeans(ObjectName name, Object query, Subject delegate) throws IOException
   {
      SOAPConnection connection = findConnection();
      return connection.queryMBeans(name, query, delegate);
   }

   public Set queryNames(ObjectName name, Object query, Subject delegate) throws IOException
   {
      SOAPConnection connection = findConnection();
      return connection.queryNames(name, query, delegate);
   }

   public boolean isRegistered(ObjectName name, Subject delegate) throws IOException
   {
      SOAPConnection connection = findConnection();
      return connection.isRegistered(name, delegate);
   }

   public Integer getMBeanCount(Subject delegate) throws IOException
   {
      SOAPConnection connection = findConnection();
      return connection.getMBeanCount(delegate);
   }

   public Object getAttribute(ObjectName name, String attribute, Subject delegate)
           throws MBeanException,
           AttributeNotFoundException,
           InstanceNotFoundException,
           ReflectionException,
           IOException
   {
      SOAPConnection connection = findConnection();
      return connection.getAttribute(name, attribute, delegate);
   }

   public AttributeList getAttributes(ObjectName name, String[] attributes, Subject delegate)
           throws InstanceNotFoundException, ReflectionException, IOException
   {
      SOAPConnection connection = findConnection();
      return connection.getAttributes(name, attributes, delegate);
   }

   public void setAttribute(ObjectName name, Attribute attribute, Subject delegate)
           throws InstanceNotFoundException,
           AttributeNotFoundException,
           InvalidAttributeValueException,
           MBeanException,
           ReflectionException,
           IOException
   {
      SOAPConnection connection = findConnection();
      connection.setAttribute(name, attribute, delegate);
   }

   public AttributeList setAttributes(ObjectName name, AttributeList attributes, Subject delegate)
           throws InstanceNotFoundException,
           ReflectionException,
           IOException
   {
      SOAPConnection connection = findConnection();
      return connection.setAttributes(name, attributes, delegate);
   }

   public Object invoke(ObjectName name, String operationName, Object[] params, String[] signature, Subject delegate)
           throws InstanceNotFoundException,
           MBeanException,
           ReflectionException,
           IOException
   {
      SOAPConnection connection = findConnection();
      return connection.invoke(name, operationName, params, signature, delegate);
   }

   public String getDefaultDomain(Subject delegate) throws IOException
   {
      SOAPConnection connection = findConnection();
      return connection.getDefaultDomain(delegate);
   }

   public String[] getDomains(Subject delegate) throws IOException
   {
      SOAPConnection connection = findConnection();
      return connection.getDomains(delegate);
   }

   public MBeanInfo getMBeanInfo(ObjectName name, Subject delegate) throws InstanceNotFoundException, IntrospectionException, ReflectionException, IOException
   {
      SOAPConnection connection = findConnection();
      return connection.getMBeanInfo(name, delegate);
   }

   public boolean isInstanceOf(ObjectName name, String className, Subject delegate) throws InstanceNotFoundException, IOException
   {
      SOAPConnection connection = findConnection();
      return connection.isInstanceOf(name, className, delegate);
   }

   public void addNotificationListener(ObjectName name, ObjectName listener, Object filter, Object handback, Subject delegate)
           throws InstanceNotFoundException, IOException
   {
      SOAPConnection connection = findConnection();
      connection.addNotificationListener(name, listener, filter, handback, delegate);
   }

   public void removeNotificationListener(ObjectName name, ObjectName listener, Subject delegate)
           throws InstanceNotFoundException, ListenerNotFoundException, IOException
   {
      SOAPConnection connection = findConnection();
      connection.removeNotificationListener(name, listener, delegate);
   }

   public void removeNotificationListener(ObjectName name, ObjectName listener, Object filter, Object handback, Subject delegate)
           throws InstanceNotFoundException, ListenerNotFoundException, IOException
   {
      SOAPConnection connection = findConnection();
      connection.removeNotificationListener(name, listener, filter, handback, delegate);
   }

   public Integer[] addNotificationListeners(ObjectName[] names, Subject[] delegates) throws InstanceNotFoundException, IOException
   {
      SOAPConnection connection = findConnection();
      return connection.addNotificationListeners(names, delegates);
   }

   public void removeNotificationListeners(ObjectName observed, Integer[] ids, Subject delegate) throws InstanceNotFoundException, ListenerNotFoundException, IOException
   {
      SOAPConnection connection = findConnection();
      connection.removeNotificationListeners(observed, ids, delegate);
   }

   public NotificationResult fetchNotifications(long clientSequenceNumber, int maxNotifications, long timeout) throws IOException
   {
      SOAPConnection connection = findConnection();
      return connection.fetchNotifications(clientSequenceNumber, maxNotifications, timeout);
   }
}
