/*
 * Copyright (c) 2005-2007 Laf-Widget Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Laf-Widget Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.lafwidget.layout;

import java.awt.*;
import java.beans.PropertyChangeEvent;

import javax.swing.border.Border;
import javax.swing.plaf.UIResource;

/**
 * {@link Border} implementation that respects the transition composite effects
 * set by the {@link TransitionLayout}. All {@link Border}s installed on the
 * component hierarchy are wrapped in instances of this class once the
 * {@link TransitionLayout} is installed on that component (with
 * {@link TransitionLayoutManager#track(Container, boolean)} or
 * {@link TransitionLayoutManager#track(Container, boolean, boolean)}.
 * 
 * <p>
 * Note that installing the {@link TransitionLayout} on a component may break
 * applications that operate directly on the installed borders, resulting in
 * {@link ClassCastException} or incorrect logic being called. The relevant code
 * needs to be changed using the {@link #getDelegate()}. In addition, the
 * applications will get {@link PropertyChangeEvent}s when these borders are
 * set / unset.
 * </p>
 * 
 * @author Kirill Grouchnikov
 */
public class TransitionBorder implements Border {
	/**
	 * The delegate (original) border. Handles all the {@link Border} methods,
	 * including {@link #paintBorder(Component, Graphics, int, int, int, int)}
	 * that operates on a {@link Graphics2D} object with
	 * {@link TransitionLayout}-relevant {@link Composite}.
	 */
	protected Border delegate;

	/**
	 * Creates a new wrapper border.
	 * 
	 * @param original
	 *            The original border.
	 */
	public TransitionBorder(Border original) {
		this.delegate = original;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.border.Border#getBorderInsets(java.awt.Component)
	 */
	public Insets getBorderInsets(Component c) {
		return this.delegate.getBorderInsets(c);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.border.Border#isBorderOpaque()
	 */
	public boolean isBorderOpaque() {
		return this.delegate.isBorderOpaque();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.border.Border#paintBorder(java.awt.Component,
	 *      java.awt.Graphics, int, int, int, int)
	 */
	public void paintBorder(Component c, Graphics g, int x, int y, int width,
			int height) {
		Graphics2D graphics = (Graphics2D) g.create();
		graphics.setComposite(TransitionLayout.getAlphaComposite(c));

		this.delegate.paintBorder(c, graphics, x, y, width, height);
		graphics.dispose();
	}

	/**
	 * Returns the original (delegate) border.
	 * 
	 * @return The original (delegate) border.
	 */
	public Border getDelegate() {
		return delegate;
	}

	/**
	 * Wrapper border that implements the {@link UIResource} interface.
	 * 
	 * @author Kirill Grouchnikov
	 */
	public static class BorderUIResource extends TransitionBorder implements
			UIResource {
		/**
		 * Creates a new wrapper border.
		 * 
		 * @param original
		 *            The original border.
		 */
		public BorderUIResource(Border original) {
			super(original);
		}
	}
}
