/*
 * Copyright (c) 2005-2007 Laf-Widget Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Laf-Widget Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.lafwidget.animation;

import java.awt.Component;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.util.*;

import javax.swing.ButtonModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.jvnet.lafwidget.LafWidgetRepository;
import org.jvnet.lafwidget.LafWidgetUtilities;

/**
 * Implements listener on model changes of some component. Once model changes
 * (select, arm, focus, ...), the listener notifies the {@link FadeTracker}.
 * 
 * @author Kirill Grouchnikov
 */
public class FadeStateListener {
	/**
	 * Optional application callback.
	 */
	protected FadeTrackerCallback callback;

	/**
	 * The associated component.
	 */
	protected Component comp;

	/**
	 * The associated model.
	 */
	protected ButtonModel buttonModel;

	/**
	 * Key - {@link FadeKind}, value - {@link Boolean}
	 */
	protected Map<FadeKind, Boolean> prevStateMap;

	/**
	 * Listener on the model changes.
	 */
	protected ChangeListener modelListener;

	/**
	 * Listener on the focus gain and loss.
	 */
	protected FocusListener focusListener;

	/**
	 * Instance ID of focus loop animation.
	 */
	protected long focusLoopFadeInstanceId;

	/**
	 * A set of fade kinds to ignore.
	 */
	protected Set<FadeKind> toIgnore;

	/**
	 * Creates a new listener on model changes that can cause fade animation
	 * transitions.
	 * 
	 * @param comp
	 *            Component.
	 * @param buttonModel
	 *            Model for the component.
	 * @param callback
	 *            Optional application callback.
	 */
	public FadeStateListener(Component comp, ButtonModel buttonModel,
			FadeTrackerCallback callback) {
		this(comp, buttonModel, callback, new HashSet<FadeKind>());
	}

	/**
	 * Creates a new listener on model changes that can cause fade animation
	 * transitions.
	 * 
	 * @param comp
	 *            Component.
	 * @param buttonModel
	 *            Model for the component.
	 * @param callback
	 *            Optional application callback.
	 * @param toIgnore
	 *            Set of fade kinds to ignore.
	 */
	public FadeStateListener(Component comp, ButtonModel buttonModel,
			FadeTrackerCallback callback, Set<FadeKind> toIgnore) {
		this.comp = comp;
		this.buttonModel = buttonModel;
		this.callback = callback;
		this.toIgnore = toIgnore;

		this.prevStateMap = new HashMap<FadeKind, Boolean>();
		if (buttonModel != null) {
			this.prevStateMap.put(FadeKind.SELECTION, buttonModel.isSelected());
			this.prevStateMap.put(FadeKind.ARM, buttonModel.isArmed());
			this.prevStateMap.put(FadeKind.ROLLOVER, buttonModel.isRollover());
			this.prevStateMap.put(FadeKind.PRESS, buttonModel.isPressed());
			this.prevStateMap.put(FadeKind.ENABLE, buttonModel.isEnabled());
		}
		this.prevStateMap.put(FadeKind.FOCUS, comp.hasFocus());
		this.focusLoopFadeInstanceId = -1;
	}

	/**
	 * Tracks a single change to the model.
	 * 
	 * @param fadeKind
	 *            Fade animation kind.
	 * @param newValue
	 *            New value of the relevant attribute of the model.
	 * @param toRepaintParent
	 *            If <code>true</code>, the component parent will be
	 *            repainted at every animation cycle. If <code>false</code>,
	 *            the component itself will be repainted at every animation
	 *            cycle.
	 */
	protected void trackModelChange(FadeKind fadeKind, boolean newValue,
			boolean toRepaintParent) {
		// special handling of cell renderers and editors - no
		// animation there
		if (LafWidgetRepository.getRepository().getFadeIgnoreManager()
				.toIgnoreAnimations(this.comp))
			return;

		if (this.toIgnore.contains(fadeKind)) {
			if (this.callback != null) {
				this.callback.fadeEnded(fadeKind);
			} else {
				this.comp.repaint();
			}
			return;
		}
		// Component currComp = this.comp;
		// while (currComp != null) {
		// for (Iterator it = FadeStateListener.ignoreAnimationsOn.iterator();
		// it
		// .hasNext();) {
		// Class ignoreAnimationClazz = (Class) it.next();
		// if (ignoreAnimationClazz.isAssignableFrom(currComp.getClass()))
		// return;
		// }
		// currComp = currComp.getParent();
		// }
		try {
			if (this.prevStateMap.containsKey(fadeKind)) {
				boolean prevValue = this.prevStateMap.get(fadeKind);
				if (prevValue == newValue)
					return;
			} else {
				if (!newValue) {
					return;
				} else {
					if (fadeKind == FadeKind.SELECTION) {
						if (this.callback != null)
							this.callback.fadeEnded(FadeKind.SELECTION);
						return;
					}
				}
			}
			FadeTracker.getInstance().trackFade(this.comp, fadeKind, newValue,
					toRepaintParent, this.callback);
		} finally {
			this.prevStateMap.put(fadeKind, newValue);
		}
	}

	/**
	 * Registers listeners on all model changes.
	 */
	public void registerListeners() {
		this.registerListeners(false);
	}

	/**
	 * Registers listeners on all model changes.
	 * 
	 * @param toRepaintParent
	 *            If <code>true</code>, the component parent will be
	 *            repainted at every animation cycle. If <code>false</code>,
	 *            the component itself will be repainted at every animation
	 *            cycle.
	 */
	public void registerListeners(final boolean toRepaintParent) {
		this.modelListener = new ChangeListener() {
			public void stateChanged(ChangeEvent e) {
				if (FadeStateListener.this.buttonModel == null)
					return;
				FadeStateListener.this.trackModelChange(FadeKind.SELECTION,
						FadeStateListener.this.buttonModel.isSelected(),
						toRepaintParent);
				FadeStateListener.this.trackModelChange(FadeKind.ARM,
						FadeStateListener.this.buttonModel.isArmed(),
						toRepaintParent);
				FadeStateListener.this.trackModelChange(FadeKind.ROLLOVER,
						FadeStateListener.this.buttonModel.isRollover(),
						toRepaintParent);
				FadeStateListener.this.trackModelChange(FadeKind.ENABLE,
						FadeStateListener.this.buttonModel.isEnabled(),
						toRepaintParent);
				FadeStateListener.this.trackModelChange(FadeKind.PRESS,
						FadeStateListener.this.buttonModel.isPressed(),
						toRepaintParent);
			}
		};
		if (this.buttonModel != null) {
			this.buttonModel.addChangeListener(this.modelListener);
		}

		this.focusListener = new FocusListener() {
			public void focusGained(FocusEvent e) {
				FadeStateListener.this.trackModelChange(FadeKind.FOCUS, true,
						toRepaintParent);

				initiateFocusFadeLoop();
			}

			public void focusLost(FocusEvent e) {
				FadeStateListener.this.trackModelChange(FadeKind.FOCUS, false,
						toRepaintParent);
				if (focusLoopFadeInstanceId >= 0) {
					FadeTracker.getInstance().cancelFadeInstance(
							focusLoopFadeInstanceId);
				}
			}
		};
		this.comp.addFocusListener(this.focusListener);

		if (this.comp.hasFocus()) {
			this.initiateFocusFadeLoop();
		}
	}

	/**
	 * Initiates the looping animation on focus ring.
	 */
	protected void initiateFocusFadeLoop() {
		FadeTrackerCallback loopCallback = (callback != null) ? callback
				: new FadeTrackerAdapter() {
					public void fadeEnded(FadeKind fadeKind) {
						comp.repaint();
					}

					public void fadePerformed(FadeKind fadeKind,
							float fadeCycle10) {
						comp.repaint();
					}
				};
		if (FadeConfigurationManager.getInstance().fadeAllowed(
				FadeKind.FOCUS_LOOP_ANIMATION, this.comp)) {
			focusLoopFadeInstanceId = FadeTracker.getInstance()
					.trackFadeLooping(FadeKind.FOCUS_LOOP_ANIMATION,
							LafWidgetUtilities.getAnimationKind(this.comp),
							this.comp, null, false, loopCallback, -1, false);
		}
	}

	/**
	 * Unregisters all listeners on model changes.
	 */
	public void unregisterListeners() {
		if (this.buttonModel != null) {
			this.buttonModel.removeChangeListener(this.modelListener);
		}
		this.comp.removeFocusListener(this.focusListener);
	}
}
