/* DominationNumber.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard;

import java.util.ArrayList;
import java.util.Arrays;
import org.grinvin.Edge;
import org.grinvin.GraphBundleView;
import org.grinvin.invariants.IntegerValue;
import org.grinvin.invariants.computers.AbstractInvariantComputer;

/**
 * computes the domination number of the graph.
 */
public class DominationNumber extends AbstractInvariantComputer {
        
    public String getInvariantId() {
        return "org.grinvin.invariants.DominationNumber";
    }

    public String getVersion() {
        return "1.0.2"; //already two bugs fixed :)
    }
    
    private boolean debug = false;
    
    public IntegerValue compute(GraphBundleView bundle) {
        int[][] adjlist = bundle.adjacencyList();
        int n = adjlist.length;
        if(n==0)
            return IntegerValue.undefinedValue(this);
        if(n==1)
            return new IntegerValue(1, this);
        //check to see if this is a complete graph
        if(bundle.getGraph().getNumberOfEdges()==n*(n-1)/2)
            return new IntegerValue(1, this);
        
        boolean[] dominatingSet = new boolean[n];
        Arrays.fill(dominatingSet, false);
        
        //find underbound and upperbound
        //==============================
        
        //bubble sort vertices according to degree before starting
        int[] vertices = sortVertices(adjlist);
        
        int sum = 0;
        int minimumSize = 0;
        while(sum<n)
            sum += (adjlist[vertices[minimumSize++]].length+1);
        
        int bestSize = n; //upperbound
        int currentSize = 0;
        
        if(debug){
            System.out.println("found bounds:");
            System.out.println("Lowerbound: " + minimumSize);
            System.out.println("upperbound: " + bestSize);
        }
        
        //mark forbidden vertices
        //=======================
        
        //initial dominationCapacity = degree + 1
        int[] dominationCapacity = new int[n];
        for(int i = 0; i<n; i++)
            if(adjlist[i].length==0){
                //add isolated vertices to dominating set
                dominationCapacity[i]=0;
                dominatingSet[i]=true;
                currentSize++;
            } else
                dominationCapacity[i]=adjlist[i].length+1;
        
        if(debug)
            System.out.println("calculated capacity");
        
        //remove vertices who have a neighbour with a neighborhood that is a superset of this vertex's neighborhood
        //check every edge
        int[] obligation = new int[n];
        
        for(Edge e : bundle.getGraph().edges()){
            int v1 = e.getFirstEndpoint().getIndex();
            int v2 = e.getSecondEndpoint().getIndex();
            if(adjlist[v1].length < adjlist[v2].length){
                v1 = e.getSecondEndpoint().getIndex();
                v2 = e.getFirstEndpoint().getIndex();
            } else if(adjlist[v1].length == adjlist[v2].length && obligation[v2]==1){
                v1 = e.getSecondEndpoint().getIndex();
                v2 = e.getFirstEndpoint().getIndex();
            }
            boolean hasSubNeighborhood = true;
            if(debug){
                System.out.println("v1: " + v1);
                System.out.println("v2: " + v2);
                System.out.println("neighbours: " + Arrays.toString(adjlist[v2]));
            }
            for(int neighbour : adjlist[v2]){
                if(neighbour!=v1){
                    boolean inV1Neighborhood = false;
                    int i = 0;
                    while(!inV1Neighborhood && i<adjlist[v1].length)
                        inV1Neighborhood = (adjlist[v1][i++] == neighbour);
                    hasSubNeighborhood = hasSubNeighborhood && inV1Neighborhood;
                }
            }
            if(hasSubNeighborhood){
                //dominationCapacity[v2]=0;
                obligation[v2]=-1;
                if(obligation[v1]!=-1)
                    obligation[v1]=1;
            }
        }

        int[] initDominationCapacity = new int[n];
        System.arraycopy(dominationCapacity, 0, initDominationCapacity, 0, n);
        
        if(debug)
            System.out.println("Marked forbidden");

        
        //
        int[] dominationCapacityFrequency = new int[adjlist[vertices[0]].length+2];
        for(int i = 0; i < n; i++)
            dominationCapacityFrequency[dominationCapacity[i]]++;
        //dominating set when dominationCapacityFrequency[0]==n
        
        if(debug)
            printInfo(-1, dominatingSet, dominationCapacity, dominationCapacityFrequency, currentSize, bestSize);
        
        //add obligated vertices: these are certain to be obligated
        for(int i = 0; i < n; i++)
            if(adjlist[i].length==1 && obligation[adjlist[i][0]]!=-1 && !dominatingSet[adjlist[i][0]]){
                addToDominatingSet(adjlist[i][0], adjlist, dominatingSet, dominationCapacity, dominationCapacityFrequency, obligation, initDominationCapacity);
                currentSize++;
            }
        
        if(debug)
            System.out.println("Start recursion");
        if(debug)
            printInfo(-1, dominatingSet, dominationCapacity, dominationCapacityFrequency, currentSize, bestSize);
        bestSize = findDominatingSet(adjlist, minimumSize, bestSize, currentSize, dominatingSet, dominationCapacity, dominationCapacityFrequency, obligation, initDominationCapacity);
        
        return new IntegerValue(bestSize, this);
    }
    
    // recursivly finds a dominating set
    private int findDominatingSet(int[][] adjlist, int minimumSize, int bestSize, int currentSize, boolean[] dominatingSet, int[] dominationCapacity, int[] dominationCapacityFrequency, int[] obligation, int[] initDominationCapacity) {
        if(currentSize >= bestSize)
            // we are only interested in a set that is smaller than the current best
            return bestSize;
        if(dominationCapacityFrequency[0]==adjlist.length){
            if(debug)
                System.out.println("========Found dominating set===========");
            return currentSize;
        }
        if(bestSize==minimumSize)
            return bestSize;
        
        //first add obligated vertices
       // int[] obligatedVertices = addObligatedVertices(adjlist, dominatingSet, dominationCapacity, dominationCapacityFrequency, obligation, currentSize, initDominationCapacity);
        int[] obligatedVertices = new int[0];
        
        if(obligatedVertices.length!=0){
            bestSize = findDominatingSet(adjlist, minimumSize, bestSize, currentSize+obligatedVertices.length, dominatingSet, dominationCapacity, dominationCapacityFrequency, obligation, initDominationCapacity);
            //now remove obligated vertices before returning
            for(int vertex : obligatedVertices)
                removeFromDominatingSet(vertex, adjlist, dominatingSet, dominationCapacity, dominationCapacityFrequency, obligation, initDominationCapacity);            
        } else {
            //no obligated vertices
            for(int i = dominationCapacityFrequency.length-1; i > 0; i--){
                if(dominationCapacityFrequency[i]!=0){
                    int counter = 0;
                    int vertex = 0;
                    while(counter<dominationCapacityFrequency[i] && vertex < dominationCapacity.length){
                        if(dominationCapacity[vertex]==i && obligation[vertex]!=-1){
                            addToDominatingSet(vertex, adjlist, dominatingSet, dominationCapacity, dominationCapacityFrequency, obligation, initDominationCapacity);
                            if(debug)
                                printInfo(vertex, dominatingSet, dominationCapacity, dominationCapacityFrequency, currentSize+1, bestSize);
                            bestSize = findDominatingSet(adjlist, minimumSize, bestSize, currentSize+1, dominatingSet, dominationCapacity, dominationCapacityFrequency, obligation, initDominationCapacity);
                            removeFromDominatingSet(vertex, adjlist, dominatingSet, dominationCapacity, dominationCapacityFrequency, obligation, initDominationCapacity);
                            counter++;
                        }
                        if(debug){
                            System.out.println("Currently looking at vertecis with capacity " + i);
                            System.out.println("Domination Capacity: " + Arrays.toString(dominationCapacity));
                            System.out.println("Obligation: " + Arrays.toString(obligation));
                        }
                        vertex++;
                    }
                }
            }            
        }
        if(dominationCapacityFrequency[0]==adjlist.length)
            return currentSize;
        else
            return bestSize;
    }

    private int[] sortVertices(int[][] adjlist) {
        int[] vertices = new int[adjlist.length];
        for(int i=0; i < adjlist.length; i++)
            vertices[i] = i;
        for(int i=0; i < vertices.length; i++)
            for(int j=vertices.length-1; j>i; j--)
                if(adjlist[vertices[j]].length > adjlist[vertices[j-1]].length){
            vertices[j] = vertices[j] + vertices[j-1];
            vertices[j-1] = vertices[j] - vertices[j-1];
            vertices[j] = vertices[j] - vertices[j-1];
                }
        return vertices;
    }
    
    private boolean isDominated(int vertex, int[][] adjlist, boolean[] dominatingSet){
        if(dominatingSet[vertex])
            return true;
        for(int v : adjlist[vertex])
            if(dominatingSet[v])
                return true;
        return false;
    }

    private void addToDominatingSet(int vertex, int[][] adjlist, boolean[] dominatingSet, int[] dominationCapacity, int[] dominationCapacityFrequency, int[] obligation, int[] initDominationCapacity){
        if(debug){
            System.out.println("Domination Capacity: " + Arrays.toString(dominationCapacity));
            System.out.println("Adding vertex " + vertex + " to dominating set.");
        }
        if(dominatingSet[vertex])
            return;
        dominatingSet[vertex]=true;
        updateDominationCapacity(vertex, adjlist, dominatingSet, dominationCapacity, dominationCapacityFrequency, obligation, initDominationCapacity);
        for(int neighbour : adjlist[vertex]){
            updateDominationCapacity(neighbour, adjlist, dominatingSet, dominationCapacity, dominationCapacityFrequency, obligation, initDominationCapacity);
            for(int nextNeighbour : adjlist[neighbour])
                updateDominationCapacity(nextNeighbour, adjlist, dominatingSet, dominationCapacity, dominationCapacityFrequency, obligation, initDominationCapacity);
        }
    }
    
    private void removeFromDominatingSet(int vertex, int[][] adjlist, boolean[] dominatingSet, int[] dominationCapacity, int[] dominationCapacityFrequency, int[] obligation, int[] initDominationCapacity){
        if(debug){
            System.out.println("Domination Capacity: " + Arrays.toString(dominationCapacity));
            System.out.println("Removing vertex " + vertex + " from dominating set.");
        }
        if(!dominatingSet[vertex])
            return;
        dominatingSet[vertex] = false;
        updateDominationCapacity(vertex, adjlist, dominatingSet, dominationCapacity, dominationCapacityFrequency, obligation, initDominationCapacity);
        for(int neighbour : adjlist[vertex]){
            updateDominationCapacity(neighbour, adjlist, dominatingSet, dominationCapacity, dominationCapacityFrequency, obligation, initDominationCapacity);
            for(int nextNeighbour : adjlist[neighbour])
                updateDominationCapacity(nextNeighbour, adjlist, dominatingSet, dominationCapacity, dominationCapacityFrequency, obligation, initDominationCapacity);
        }
    }

    private void updateDominationCapacity(int vertex, int[][] adjlist, boolean[] dominatingSet, int[] dominationCapacity, int[] dominationCapacityFrequency, int[] obligation, int[] initDominationCapacity){
        int count = isDominated(vertex, adjlist, dominatingSet) ? 1 : 0;
        for(int neighbour : adjlist[vertex])
            if(isDominated(neighbour, adjlist, dominatingSet))
                count++;
        //System.out.println("Vertex " + vertex + ": " +initDominationCapacity[vertex] + "-" + count);
        setDominationCapacity(initDominationCapacity[vertex]-count, vertex, dominationCapacity, dominationCapacityFrequency, obligation);
    }
    
    private void changeDominationCapacity(int change, int vertex, int[] dominationCapacity, int[] dominationCapacityFrequency, int[] obligation){
        if(debug)
            System.out.println("changing capacity of vertex " + vertex + "(currently: " + dominationCapacity[vertex] + ")");
        //if(change<0 && dominationCapacity[vertex]==0)
        //    return;
        if(obligation[vertex]<0)
            return;
        //if(dominationCapacity[vertex]+change>=dominationCapacityFrequency.length)
        //    return;
        dominationCapacityFrequency[dominationCapacity[vertex]>=0 ? dominationCapacity[vertex] : 0]--;
        dominationCapacity[vertex]+=change;
        dominationCapacityFrequency[dominationCapacity[vertex]>=0 ? dominationCapacity[vertex] : 0]++;
    }
    
    private void setDominationCapacity(int newDC, int vertex, int[] dominationCapacity, int[] dominationCapacityFrequency, int[] obligation){
        if(newDC>=dominationCapacityFrequency.length)
            return;
        //if(obligation[vertex]<0)
        //    newDC=0;
        dominationCapacityFrequency[dominationCapacity[vertex]>=0 ? dominationCapacity[vertex] : 0]--;
        dominationCapacity[vertex]=newDC;
        dominationCapacityFrequency[dominationCapacity[vertex]>=0 ? dominationCapacity[vertex] : 0]++;
    }
    
    //not dominated neighbours of vertices with domination capacity 2
    private int[] addObligatedVertices(int[][] adjlist, boolean[] dominatingSet, int[] dominationCapacity, int[] dominationCapacityFrequency, int[] obligation, int currentSize, int[] initDominationCapacity){
        if(debug)  
            System.out.println("---------------Start obligated vertex-----------------");
        int counter = 0;
        int vertex = 0;
        ArrayList<Integer> temp = new ArrayList<Integer>();
        for(int i=0; i<dominationCapacity.length; i++)
            if(dominationCapacity[i]==2 && !isDominated(i, adjlist, dominatingSet))
                for(int neighbour : adjlist[i])
                    if(!(dominationCapacity[neighbour]==2 && neighbour<i) && !isDominated(neighbour, adjlist, dominatingSet) && !temp.contains(Integer.valueOf(neighbour)) && obligation[neighbour]!=-1)
                        temp.add(Integer.valueOf(neighbour));
        int[] addedVertices = new int[temp.size()];
        for(int i = 0; i < addedVertices.length; i++)
            addedVertices[i] = temp.get(i).intValue();
        for(int v : addedVertices)
            addToDominatingSet(v, adjlist, dominatingSet, dominationCapacity, dominationCapacityFrequency, obligation, initDominationCapacity);
        if(debug){
            printInfo(addedVertices, dominatingSet, dominationCapacity, dominationCapacityFrequency, currentSize + addedVertices.length);
            System.out.println("---------------End obligated vertex-----------------");
        }
        return addedVertices;
    }
    
    private int notDominatedVertices(){
        
        return -1;
    }

    private void printInfo(int vertex, boolean[] dominatingSet, int[] dominationCapacity, int[] dominationCapacityFrequency, int currentSize, int bestSize) {
        System.out.println("Added vertex " + vertex + " to set.");
        System.out.print("Current set: ");
        int size=0;
        for(int i = 0; i<dominatingSet.length; i++)
            if(dominatingSet[i]){
                System.out.print(i + " ");
                size++;
            }
        System.out.println();
        System.out.println("Best size: " + bestSize);
        System.out.println("Current size: " + currentSize);
        System.out.println("Current Actual size: " + size);
        System.out.println("Domination Capacity: " + Arrays.toString(dominationCapacity));
        System.out.println("Domination Capacity Frequency: " + Arrays.toString(dominationCapacityFrequency));
        System.out.println();
    }
    
    private void printInfo(int[] vertices, boolean[] dominatingSet, int[] dominationCapacity, int[] dominationCapacityFrequency, int currentSize) {
        System.out.println("Added vertices " + Arrays.toString(vertices) + " to set.");
        System.out.print("Current set: ");
        int size=0;
        for(int i = 0; i<dominatingSet.length; i++)
            if(dominatingSet[i]){
                System.out.print(i + " ");
                size++;
            }
        System.out.println();
        System.out.println("Current size: " + currentSize);
        System.out.println("Current Actual size: " + size);
        System.out.println("Domination Capacity: " + Arrays.toString(dominationCapacity));
        System.out.println("Domination Capacity Frequency: " + Arrays.toString(dominationCapacityFrequency));
        System.out.println();
    }
}
