/* ChromaticIndex.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.standard;

import org.grinvin.Edge;
import org.grinvin.GraphBundleView;
import org.grinvin.invariants.IntegerValue;
import org.grinvin.invariants.computers.AbstractInvariantComputer;

/**
 * computes the chromatic index of the graph.
 */
public class ChromaticIndex extends AbstractInvariantComputer {
    
    public IntegerValue compute(GraphBundleView bundle) {
        //first find the maximum degree
        int[][] adjlist = bundle.adjacencyList();
        int n = adjlist.length;
        if(n==0)
            return IntegerValue.undefinedValue(this);
        int maxdeg = 0;
        int vertexWithMaxDegree = -1;
        for (int i = 0; i < n; i++) {
            if (adjlist[i].length > maxdeg) {
                maxdeg = adjlist[i].length;
                vertexWithMaxDegree = i;
            }
        }
        
        //array that tracks which colors are used at each vertex
        boolean[][] graph = new boolean[n][maxdeg];
        //color the edges surrounding a vertex with maximum degree
        //and build an array with edges
        for(int i=0; i < maxdeg; i++)
            graph[vertexWithMaxDegree][i]=true;
        Edge[] edges = new Edge[bundle.getGraph().getNumberOfEdges() - maxdeg];
        int index = 0;
        int color=0;
        for(Edge e : bundle.getGraph().edges())
            if(e.getFirstEndpoint().getIndex()==vertexWithMaxDegree)
                graph[e.getSecondEndpoint().getIndex()][color++]=true;
            else if(e.getSecondEndpoint().getIndex()==vertexWithMaxDegree)
                graph[e.getFirstEndpoint().getIndex()][color++]=true;
            else
                edges[index++]=e;
        //start at first edge and find a coloring with maxdeg colors
        int currentEdge=0;
        if(overfull(edges.length, n, maxdeg))
            return new IntegerValue(maxdeg + 1, this);
        if(colorEdge(graph, currentEdge, edges, maxdeg))
            return new IntegerValue(maxdeg, this);
        else
            return new IntegerValue(maxdeg + 1, this);
    }
    
    // receives a partially edge colored graph and tries to color the next edge
    // returns false if no coloring is possible
    private boolean colorEdge(boolean[][] graph, int currentEdge, Edge[] edges, int maxdeg) {
        if(currentEdge == edges.length)
            return true; //all edges colored with maxdeg colors.
        for(int i=0; i < maxdeg; i++){
            if(!graph[edges[currentEdge].getFirstEndpoint().getIndex()][i] &&
                    !graph[edges[currentEdge].getSecondEndpoint().getIndex()][i]) {
                //color edge
                graph[edges[currentEdge].getFirstEndpoint().getIndex()][i] = true;
                graph[edges[currentEdge].getSecondEndpoint().getIndex()][i] = true;
                if(colorEdge(graph, currentEdge+1, edges, maxdeg)) {
                    return true;
                } else {
                    graph[edges[currentEdge].getFirstEndpoint().getIndex()][i] = false;
                    graph[edges[currentEdge].getSecondEndpoint().getIndex()][i] = false;
                }
            }
        }
        return false;
    }
    
    private boolean overfull(int nrOfEdges, int nrOfVertices, int maxdeg){
        return (nrOfVertices%2==1) && ((nrOfVertices - 1)/2*maxdeg<nrOfEdges);
    }
    
    public String getInvariantId() {
        return "org.grinvin.invariants.ChromaticIndex";
    }
}
