/* BalabanIndex.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.computers.chemical;

import org.grinvin.Edge;
import org.grinvin.GraphBundleView;
import org.grinvin.GraphView;
import org.grinvin.invariants.RealValue;
import org.grinvin.invariants.computers.AbstractInvariantComputer;

/**
 * computes the Balaban index of the graph.
 */
public class BalabanIndex extends AbstractInvariantComputer {
    
    public RealValue compute(GraphBundleView bundle) {
        GraphView graph = bundle.getGraph();
        int[][] distanceMatrix = bundle.distanceMatrix();
        int numberOfVertices = graph.getNumberOfVertices();
        int numberOfEdges = graph.getNumberOfEdges();
        int cyclomaticNumber = numberOfEdges - numberOfVertices + 1;
        
        //calculate the vertex distance degrees
        //and check for connectedness
        boolean connected = true;
        int[] vertexDistanceDegree = new int[numberOfVertices];
        for(int i = 0; i < numberOfVertices; i++)
            for(int j = 0; j < numberOfVertices; j++){
                connected = connected && !(distanceMatrix[i][j]==0 && j != i);
                vertexDistanceDegree[i] += distanceMatrix[i][j];
            }
        
        if(!connected)
            return RealValue.undefinedValue(this);

        //calculate Balaban index
        double balabanIndex = 0;
        for (Edge e: graph.edges())
            balabanIndex += 1/Math.sqrt(vertexDistanceDegree[e.getFirstEndpoint().getIndex()] * vertexDistanceDegree[e.getSecondEndpoint().getIndex()]);
        balabanIndex *= numberOfEdges/(cyclomaticNumber + 1);
        return new RealValue(balabanIndex, this);
    }
    
    public String getInvariantId() {
        return "org.grinvin.invariants.BalabanIndex";
    }
}
