/* TuranGraphFactory.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2007 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.factories.graphs.standard;

import org.grinvin.Embedding;
import org.grinvin.Graph;
import org.grinvin.Vertex;
import org.grinvin.factories.FactoryParameterException;
import org.grinvin.factories.graphs.AbstractGraphFactory;

/**
 * Factory that creates grids with given number of rows and columns.
 */
public class TuranGraphFactory extends AbstractGraphFactory {
    
    //
    @Override protected void createGraph(Graph graph, Embedding embedding) {
        embedding.setDimension(2);
        int n = ((Integer)values[0]).intValue();
        int r = ((Integer)values[1]).intValue();
        int t = n % r;
        Vertex[] vertices = new Vertex[n];
        
        //We start by creating and placing the vertices.
        //There are r partitions. Every partition is placed
        //on an edge of a regular r-gon.
        
        //first r-t partitions of size (n - t)/r.
        int part_size = (n - t)/r;
        
        //when all partitions have the same order we use
        //the distance of the first vertex to the center
        //to scale. When there are partitions of order
        //(n - t)/r + 1 we use the distance of the last
        //vertex (i.e. a vertex in a partition of order
        //(n - t)/r + 1) to the center.
        double scale = (t==0 ? Math.sqrt(
                (1 + ((Math.cos(2*Math.PI/r) - 1)/(part_size + 5)) * 3)*
                (1 + ((Math.cos(2*Math.PI/r) - 1)/(part_size + 5)) * 3) +
                ((Math.sin(2*Math.PI/r)/(part_size + 5)) * 3)*
                ((Math.sin(2*Math.PI/r)/(part_size + 5)) * 3)
                ) : Math.sqrt(
                (Math.cos(2*(r-1)*Math.PI/r) + ((1 - Math.cos(2*(r-1)*Math.PI/r))/(part_size + 1 + 5)) * 3)*
                (Math.cos(2*(r-1)*Math.PI/r) + ((1 - Math.cos(2*(r-1)*Math.PI/r))/(part_size + 1 + 5)) * 3) +
                (Math.sin(2*(r-1)*Math.PI/r) + ((-Math.sin(2*(r-1)*Math.PI/r))/(part_size + 1 + 5)) * 3)*
                (Math.sin(2*(r-1)*Math.PI/r) + ((-Math.sin(2*(r-1)*Math.PI/r))/(part_size + 1 + 5)) * 3)
                ));
        for(int i = 0; i < r - t; i++) {
            double angle1 = 2 * i * Math.PI / r;
            double angle2 = 2 * (i + 1) * Math.PI / r;
            double x1 = Math.cos(angle1);
            double x2 = Math.cos(angle2);
            double y1 = Math.sin(angle1);
            double y2 = Math.sin(angle2);
            for(int j = 0; j < part_size; j++) {
                double[] coords = new double[2];
                coords[0]=(x1 + ((x2 - x1)/(part_size + 5)) * (j + 3))/scale;
                coords[1]=(y1 + ((y2 - y1)/(part_size + 5)) * (j + 3))/scale;
                vertices[i*part_size + j] = graph.addNewVertex(null);
                embedding.setCoordinates(vertices[i*part_size + j], coords);
            }
        }
        //then t partitions of size (n - t)/r + 1.
        part_size = (n-t)/r + 1;
        for(int i = 0; i < t; i++) {
            double angle1 = 2 * (i + r - t) * Math.PI / r;
            double angle2 = 2 * (i + r - t + 1) * Math.PI / r;
            double x1 = Math.cos(angle1);
            double x2 = Math.cos(angle2);
            double y1 = Math.sin(angle1);
            double y2 = Math.sin(angle2);
            for(int j = 0; j < part_size; j++) {
                double[] coords = new double[2];
                coords[0]=(x1 + ((x2 - x1)/(part_size + 5)) * (j + 3))/scale;
                coords[1]=(y1 + ((y2 - y1)/(part_size + 5)) * (j + 3))/scale;
                vertices[(r-t)*(n-t)/r + i*part_size + j] = graph.addNewVertex(null);
                embedding.setCoordinates(vertices[(r-t)*(n-t)/r + i*part_size + j], coords);
            }
        }
        
        // Now we connect the partitions with each other.
        part_size = (n - t)/r;
        for(int i = 0; i < r - t; i++)
            for(int j = 0; j < part_size; j++)
                for(int k = (i + 1)*part_size; k < n; k++)
                    graph.addNewEdge(vertices[i*part_size + j],vertices[k],null);
        part_size = (n-t)/r + 1;
        for(int i = 0; i < t - 1; i++)
            for(int j = 0; j < part_size; j++)
                for(int k = (r-t)*(n-t)/r + (i + 1)*part_size; k < n; k++)
                    graph.addNewEdge(vertices[(r-t)*(n-t)/r + i*part_size + j],vertices[k],null);
        
    }
    
    //
    @Override protected void checkParameters(Object[] values) throws FactoryParameterException {
        super.checkParameters(values);
        int n = ((Integer)values[0]).intValue();
        int r = ((Integer)values[1]).intValue();
        if ( r < 3 || n < r)
            throw new FactoryParameterException("r has to be bigger than 2 and n should be bigger than or equal to r.");
    }
    
}
