/* GraphListSaver.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.io.list;

import java.io.IOException;
import java.io.OutputStream;
import java.net.URI;
import java.util.logging.Logger;

import org.grinvin.io.LoaderSaverHelper;
import org.grinvin.io.SectionSaver;
import org.grinvin.list.graphs.GraphList;
import org.grinvin.list.graphs.GraphListElement;

import org.jdom.DocType;
import org.jdom.Element;

/**
 * Helper methods for saving graph list elements, lists of graphs
 * and lists of invariants to disk.
 */
public final class GraphListSaver {
    
    //
    private static final String PUBLIC_ID = "-//GrInvIn IO//GraphList 1.1//EN";

    //
    private static final Logger LOGGER = Logger.getLogger("org.grinvin.io");
    
    // do not instantiate
    private GraphListSaver() {}
    
    /**
     * Save a list of graphs as an XML file. The XML file will only contain URL-references
     * to the corresponding graph bundles, and not the graph bundle contents. These should
     * be saved separately.<p>
     * If the parameter <tt>ilist</tt> is not null, also the invariants in that
     * list are stored into the XML file.
     * Creates a file with contents of the form
     * <pre>
     *    &lt;graphlistelements name="..."&gt;
     *        &lt;graphlistelement graphURI="..."/&gt;
     *        ...
     *        &lt;invariant invariantId="..."/&gt;
     *        ...
     *    &lt;/graphlistelements&gt;
     * </pre>
     * <b>Note:</b> Graphs without an URI ar note saved
     * @param list Graph list to be stored
     * @param ilist Optional invariant list to be stored into the same file (can be null)
     * @param file File into which this list should be stored
     */
    public static void save(GraphList list, OutputStream out) throws IOException {
        Element element = new Element("graphlist");
        for(GraphListElement e : list) {
            URI uri = e.getURI();
            if (uri != null)
                element.addContent(new Element("graphlistelement").setAttribute("uri", uri.toString()));
            
        }
        LoaderSaverHelper.outputXML(element,
                new DocType("graphlist", PUBLIC_ID, "http://downloads.grinvin.org/dtds/io/graphlist-1.1.dtd"),
                out);
    }
    
    /**
     * Save a (session) graph list into a (workspace) directory. Uses the uri
     * of the graph list to determine the destination file name. If no URI was
     * assigned, a new URI is created. Note that the graph list elements themselves
     * need to be saved separately.<p>
     * @throws IllegalArgumentException if the URI of the list is non null and
     * not a session URI
     * @see #saveIntoWorkspace(GraphListElement gle, File directory)
     */
    public static void saveIntoWorkspace(GraphList list, SectionSaver ssaver) throws IOException {
        saveContentsIntoWorkspace(list, ssaver);
        String name = LoaderSaverHelper.prepareSessionURI(list, "graphlists", "ggl");
        save(list, ssaver.createSection(name));
    }
    
    private static void saveContentsIntoWorkspace(GraphList list, SectionSaver ssaver) throws IOException {
        for (GraphListElement gle : list)
            GraphListElementSaver.saveIntoWorkspace(gle, ssaver);
    }
    
}
