/* RealValue.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.invariants.values;

import java.util.ResourceBundle;

import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantComputer;
import org.grinvin.invariants.InvariantType;

/**
 * Invariant type which allows real values, including positive and negative infinity
 * and undefined values. Identified by the string <code>real</code>.
 */
public class RealValue extends NumericValue {
    
    //
    private static final String BUNDLE_NAME = "org.grinvin.invariants.resources";
   
    /**
     * Return an object of this type representing positive infinity.
     * @param computer Invariant computer which calculated this value.
     */
    public static RealValue positiveInfinity (InvariantComputer computer) {
        return new RealValue (Double.POSITIVE_INFINITY, computer);
    }
    
    /**
     * Return an object of this type representing negative infinity.
     * @param computer Invariant computer which calculated this value.
     */
    public static RealValue negativeInfinity (InvariantComputer computer) {
        return new RealValue (Double.NEGATIVE_INFINITY, computer);
    }
    
    /**
     * Return an object of this type representing an undefined value.
     * @param computer Invariant computer which calculated this value.
     */
    public static RealValue undefinedValue (InvariantComputer computer) {
        return new RealValue (Double.NaN, computer);
    }
    
    /**
     * Create an object of this type with the given value which
     * is computed by the given computer.
     * @see #positiveInfinity(InvariantComputer)
     * @see #negativeInfinity(InvariantComputer)
     * @see #undefinedValue(InvariantComputer)
     */
    public RealValue (double value, InvariantComputer computer) {
        super (value, InvariantType.REAL, computer);
    }
    
    RealValue(double value, Invariant invariant, String computerId, String computerVersion) {
        super(value, InvariantType.REAL, invariant, computerId, computerVersion);
    }
    
    /**
     * Converts a real number to a value with internationalized
     * names for 'undefined' and 'infinity'.
     */
    public static String toText (double value) {
        if(Double.isNaN(value))
            return ResourceBundle.getBundle(BUNDLE_NAME).getString("undefined");
        else if (Double.isInfinite(value))
            return (value < 0 ? "-" : "") + 
                    ResourceBundle.getBundle(BUNDLE_NAME).getString("infinity");
        else
            return Double.toString(value);
    }
    
    //
    public String toText() {
        return toText (this.value);
    }
        
}
