/* InlineGraphViewer.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.help;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.net.URI;
import javax.swing.JPanel;
import javax.swing.UIManager;

import org.grinvin.list.graphs.GraphListElementManager;

/**
 * Component for displaying a {@link GraphViewer} in a HTML page.
 * The image of the graph is centered horizontally on the page.
 */
public class InlineGraphViewer extends JPanel {
    
    private URI uri;

    private GraphViewer viewer;
    
    private final static String editorPropertyPrefix = "EditorPane" + ".";
    
    /**
     * Creates a new instance of InlineGraphViewer.
     */
    public InlineGraphViewer(){
        viewer = new GraphViewer();
        setLayout(new BorderLayout());
        add(viewer, BorderLayout.CENTER);
        setOpaque(false);
    }
    
    public InlineGraphViewer(String graph, String backgroundColor, String size, String caption){
        this();
        setGraph(graph);
        if(backgroundColor!=null)
            setBackgroundColor(backgroundColor);
        if(size!=null)
            setDisplaySize(size);
        if(caption!=null)
            setCaption(caption);
    }
    
    /**
     * Sets the background Color.
     *
     * @param backgroundColor a string containing the six characters representing the color in hexadecimal notation: RRGGBB.
     */
    public void setBackgroundColor(String backgroundColor) {
        try {
            Color color = Color.decode("0X" + backgroundColor);
            viewer.setOpaque(true);
            viewer.setBackground(color);
        } catch (NumberFormatException e) {
            viewer.setOpaque(false);
            viewer.setBackground(UIManager.getColor(editorPropertyPrefix + "background"));
        }
    }
    
    /**
     * Returns the background Color
     */
    public String getBackgroundColor() {
        return Integer.toHexString(viewer.getBackground().getRGB());
    }
    
    /**
     * Sets the graph.
     *
     * @param graph a string representing the URI of the graph to be displayed.
     */
    public void setGraph(String graph){
        uri = URI.create(graph);
        viewer.setElement(GraphListElementManager.getInstance().getGraphListElement(uri));
    }
    
    /**
     * Returns the URI of the displayed graph as a string.
     */
    public String getGraph(){
        return uri.toString();
    }
    
    public void setDisplaySize(String displaySize){
        int size = Integer.parseInt(displaySize);
        viewer.setPreferredSize(new Dimension(size, size));
        viewer.setMinimumSize(new Dimension(size, size));
    }
    
    public String getDisplaySize(){
        return Integer.toString(viewer.getPreferredSize().width);
    }
    
    public void setCaption(String caption){
        viewer.setText(caption);
    }
    
    public String getCaption(){
        return viewer.getText();
    }
}
