/* SelectRectangle.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gred.mouse;

import be.ugent.caagt.swirl.mouse.NullMouseHandler;

import java.awt.BasicStroke;
import javax.swing.JComponent;
import javax.swing.SwingUtilities;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Paint;
import java.awt.Rectangle;
import java.awt.Stroke;
import java.awt.event.MouseEvent;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Collection;

import org.grinvin.graphs.Element;
import org.grinvin.gred.GraphPanel;
import org.grinvin.util.GrinvinUtilities;

/**
 * Mouse handler which allows elements on a {@link GraphPanel} to be
 * selected by means of a 'rubber band' rectangle. Consumes all first button
 * pressed events, also when pressed on a vertex or edge.
 */
public class SelectRectangle extends NullMouseHandler {

    // create a singleton
    private SelectRectangle () { }
    
    //
    private static final SelectRectangle SINGLETON = new SelectRectangle();
    
    /**
     * Return the singleton object of this type.
     */
    public static SelectRectangle getHandler () {
        return SINGLETON;
    }
    
    /**
     * Convenience method that returns a rectangle that contains both 
     * event locations.
     */
    private Rectangle getRectangle(MouseEvent draggedEvent, MouseEvent pressedEvent) {
        
        int draggedX = draggedEvent.getX(); 
        int draggedY = draggedEvent.getY(); 
        int pressedX = pressedEvent.getX();
        int pressedY = pressedEvent.getY();
        
        if (draggedX > pressedX)
            if (draggedY > pressedY)
                return  new Rectangle(pressedX, pressedY, draggedX - pressedX, draggedY - pressedY);
            else
                return  new Rectangle(pressedX, draggedY, draggedX - pressedX, pressedY - draggedY);
        else
            if (draggedY > pressedY)
                return  new Rectangle(draggedX, pressedY, pressedX - draggedX, draggedY - pressedY);
            else
                return  new Rectangle(draggedX, draggedY, pressedX - draggedX, pressedY - draggedY);
    }
   
    /**
     * Consume the event when pressed.
     */
    @Override public void doMousePressed(JComponent parent, MouseEvent mouseEvent) {
        if (SwingUtilities.isLeftMouseButton (mouseEvent))
            mouseEvent.consume ();
    }
    
    /**
     * Clear the selection when the mouse was clicked (but not ctrl-clicked)
     */
    @Override public void doMouseClicked (JComponent parent, MouseEvent mouseEvent) {
        if (SwingUtilities.isLeftMouseButton (mouseEvent) && !GrinvinUtilities.isMultipleSelectionEvent(mouseEvent)) {
            ((GraphPanel)parent).getContext ().clearSelection ();
            mouseEvent.consume ();
        }
    }
    
    
    /**
     * Performs a drag.
     */
    @Override public void doMouseDragged(JComponent parent, MouseEvent mouseEvent, MouseEvent previousEvent, MouseEvent pressedEvent) {
        parent.repaint(); // TODO: more specific?
    }
    
    /**
     * Sets the selection to the elements within the current rectangle
     * or adds these elements to the selection when 'Control' is down.
     */
    @Override public void doMouseReleased(JComponent parent, MouseEvent mouseEvent, MouseEvent pressedEvent) {
        parent.repaint();
        GraphPanel panel = (GraphPanel)parent;
        Rectangle2D bounds = panel.embeddedRectangle(getRectangle(pressedEvent, mouseEvent));
        Collection<Element> set = panel.addInsideOfRectangle(bounds, new ArrayList<Element>());
        if (GrinvinUtilities.isMultipleSelectionEvent(mouseEvent))
            panel.getContext().addSelection(set);
        else
            panel.getContext().setSelection(set);
    }

    //
    private final Paint color  = new Color(0, 127, 255);
    
    // stroke used to draw the rubber band
    private final Stroke stroke = new BasicStroke (1.0f);

    /**
     * Paints the rubber band rectangle.
     */
    @Override public void paintDragging(JComponent parent, Graphics g, MouseEvent mouseEvent, MouseEvent previousEvent, MouseEvent pressedEvent) {
        Graphics2D g2 = (Graphics2D)g;
        g2.setPaint(color);
        g2.setStroke (stroke);
        g2.draw (getRectangle (pressedEvent, mouseEvent));
    }
    
    @Override public boolean canHandle(JComponent parent) {
        return parent instanceof GraphPanel;
    }

    
}
