/*-
* See the file LICENSE for redistribution information.
*
* Copyright (c) 2002-2005
*      Sleepycat Software.  All rights reserved.
*
* $Id: FileSelector.java,v 1.5 2005/04/20 16:45:23 mark Exp $
*/

package com.sleepycat.je.cleaner;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.dbi.DatabaseId;
import com.sleepycat.je.tree.LN;

/**
 * Keeps track of the status of files for which cleaning is in progres.
 */
class FileSelector {

    /*
     * Each file for which cleaning is in progress is in one of the following
     * collections.  Files numbers migrate from one collection to another as
     * their status changes, in order:
     *
     * toBeCleaned -> cleaned -> checkpointed -> fullyProcessed -> safeToDelete
     *
     * Access to these collections is synchronized to guarantee that the status
     * is atomically updated.
     */

    /*
     * A file is initially to-be-cleaned when it is selected as part of a batch
     * that, when deleted, will bring total utilization down to the minimum
     * configured value.  All files in this collection will be cleaned in list
     * order, which is the order in which they were selected.
     */
    private List toBeCleanedFiles;

    /*
     * A file moves to the cleaned set when all log entries have been read and
     * processed.  However, entries needing migration will be marked with the
     * BIN entry MIGRATE flag, and entries that could not be locked will be
     * in the pending LN set.
     */
    private Set cleanedFiles;

    /*
     * A file moves to the checkpointed set at the end of a checkpoint if it
     * was in the cleaned set at the beginning of the checkpoint.  Because all
     * dirty BINs are flushed during the checkpoints, no files in this set
     * will have entries with the MIGRATE flag set.  However, some entries may
     * be in the pending LN set.
     */
    private Set checkpointedFiles;

    /*
     * A file is moved from the checkpointed set to the fully-processed set
     * when the pending LN set becomes empty.  Since a pending LN was not
     * locked successfully, we don't know its original file.  But we do know
     * that when no pending LNs are present for any file, all log entries in
     * checkpointed files are either obsolete or have been migrated.  However,
     * the parent BINs of the migrated entries may not have been logged yet.
     */
    private Set fullyProcessedFiles;

    /*
     * A file moves to the safe-to-delete set at the end of a checkpoint if it
     * was in the fully-processed set at the beginning of the checkpoint.  All
     * parent BINs of migrated entries have now been logged and the files are
     * safe to delete.
     */
    private Set safeToDeleteFiles;

    /*
     * Pending LNs are stored in a map of {NodeID -> LNInfo}.  These are either
     * 1- LNs that could not be locked, or 2- LNs that were in BIN entries with
     * the MIGRATE flag set and which were not processed during a checkpoint
     * because the BIN was logged by a split.
     */
    private Map pendingLNs;

    /*
     * If during a checkpoint there are no pending LNs, we can move cleaned
     * files to safe-delete files at the end of the checkpoint.  This is an
     * optimization that allows deleting files more quickly when possible. In
     * particular this impacts the checkpoint during environment close, since
     * no locks are held during that checkpoint; this optimization allows us to
     * delete all cleaned files after the final checkpoint.
     */
    private boolean pendingLNsDuringCheckpoint;

    FileSelector() {
        toBeCleanedFiles = new ArrayList();
        cleanedFiles = new HashSet();
        checkpointedFiles = new HashSet();
        fullyProcessedFiles = new HashSet();
        safeToDeleteFiles = new HashSet();
        pendingLNs = new HashMap();
    }

    /**
     * Returns the best file that qualifies for cleaning, or null if no file
     * qualifies.  This method is not thread safe and should only be called
     * from the cleaner thread.
     *
     * @param forceCleaning is true to always select a file, even if its
     * utilization is above the minimum utilization threshold.
     *
     * @param lowUtilizationFiles is a returned set of files that are below the
     * minimum utilization threshold.
     *
     * @param maxBatchFiles is the maximum number of files to be selected at
     * one time, or zero if there is no limit.
     *
     * @return a copy of the selected list of files, in the order to be
     * cleaned.
     */
    List selectFilesForCleaning(UtilizationProfile profile,
                                boolean forceCleaning,
                                Set lowUtilizationFiles,
                                int maxBatchFiles)
        throws DatabaseException {

        /*
         * Add files until we reach the theoretical minimum utilization
         * threshold.
         */
        while (true) {

            if (maxBatchFiles > 0) {
                synchronized (this) {
                    if (toBeCleanedFiles.size() >= maxBatchFiles) {
                        break;
                    }
                }
            }

            Long fileNum = profile.getBestFileForCleaning
                (this, forceCleaning, lowUtilizationFiles);

            if (fileNum == null) {
                break;
            }

            synchronized (this) {
                toBeCleanedFiles.add(fileNum);
            }
        }

        /* Return a copy of the selected files. */
        synchronized (this) {
            return new ArrayList(toBeCleanedFiles);
        }
    }

    /**
     * Returns whether the file is in any stage of the cleaning process.
     */
    synchronized boolean isFileCleaningInProgress(Long file) {
        return toBeCleanedFiles.contains(file) ||
               cleanedFiles.contains(file) ||
               checkpointedFiles.contains(file) ||
               fullyProcessedFiles.contains(file) ||
               safeToDeleteFiles.contains(file);
    }

    /**
     * When cleaning is complete, move the file from the to-be-cleaned set to
     * the cleaned set.
     */
    synchronized void addCleanedFile(Long fileNum) {
        cleanedFiles.add(fileNum);
        toBeCleanedFiles.remove(fileNum);
    }

    /**
     * Returns a copy of the cleaned and fully-processed files at the time a
     * checkpoint starts.
     */
    synchronized Set[] getFilesAtCheckpointStart() {

        pendingLNsDuringCheckpoint = !pendingLNs.isEmpty();

        Set[] files = new Set[2];

        files[0] = (cleanedFiles.size() > 0) ?
            (new HashSet(cleanedFiles)) : null;

        files[1] = (fullyProcessedFiles.size() > 0) ?
            (new HashSet(fullyProcessedFiles)) : null;

        return (files[0] != null || files[1] != null) ? files : null;
    }

    /**
     * When a checkpoint is complete, moves the previously cleaned and
     * fully-processed files to the checkpointed and safe-to-delete sets.
     */
    synchronized void updateFilesAtCheckpointEnd(Set[] files) {

        if (files != null) {

            Set previouslyCleanedFiles = files[0];
            if (previouslyCleanedFiles != null) {
                if (pendingLNsDuringCheckpoint) {
                    checkpointedFiles.addAll(previouslyCleanedFiles);
                } else {
                    safeToDeleteFiles.addAll(previouslyCleanedFiles);
                }
                cleanedFiles.removeAll(previouslyCleanedFiles);
            }

            Set previouslyProcessedFiles = files[1];
            if (previouslyProcessedFiles != null) {
                safeToDeleteFiles.addAll(previouslyProcessedFiles);
                fullyProcessedFiles.removeAll(previouslyProcessedFiles);
            }

            updateProcessedFiles();
        }
    }

    /**
     * Adds the given LN info to the pending LN set.
     */
    synchronized void addPendingLN(LN ln, DatabaseId dbId,
                                   byte[] key, byte[] dupKey) {
        assert ln != null;

        pendingLNs.put
            (new Long(ln.getNodeId()),
             new LNInfo(ln, dbId, key, dupKey));

        pendingLNsDuringCheckpoint = true;
    }

    /**
     * Returns an array of LNInfo for LNs that could not be migrated in a
     * prior cleaning attempt, or null if no LNs are pending.
     */
    synchronized LNInfo[] getPendingLNs() {

        if (pendingLNs.size() > 0) {
            LNInfo[] lns = new LNInfo[pendingLNs.size()];
            pendingLNs.values().toArray(lns);
            return lns;
        } else {
            return null;
        }
    }

    /**
     * Removes the LN for the given node ID from the pending LN set.
     */
    synchronized void removePendingLN(long nodeId) {

        pendingLNs.remove(new Long(nodeId));
        updateProcessedFiles();
    }

    /**
     * Returns a copy of the safe-to-delete files.
     */
    synchronized Set copySafeToDeleteFiles() {
        if (safeToDeleteFiles.size() == 0) {
            return null;
        } else {
            return new HashSet(safeToDeleteFiles);
        }
    }

    /**
     * Removes file from the safe-to-delete set after the file itself has
     * finally been deleted.
     */
    synchronized void removeDeletedFile(Long fileNum) {
        safeToDeleteFiles.remove(fileNum);
    }

    /**
     * If there are no pending LNs outstanding, move the checkpointed files to
     * the fully-processed set.  The check for pending LNs and the copying of
     * the checkpointed files must be done atomically in a synchronized block.
     * All methods that call this method are synchronized.
     */
    private void updateProcessedFiles() {
        if (pendingLNs.isEmpty()) {
            fullyProcessedFiles.addAll(checkpointedFiles);
            checkpointedFiles.clear();
        }
    }
}
