/*  This file is part of the KDE libraries
    Copyright (C) 1999 Waldo Bastian (bastian@kde.org)
    Modified by Toan T Nguyen (nguyenthetoan@gmail.com) to allow for
    listbox selection return

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; version 2
    of the License.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/
#ifndef _KXMESSAGEBOX_H_
#define _KXMESSAGEBOX_H_

#include <qstring.h>
#include <qmessagebox.h>

#include <kguiitem.h>
#include <kstdguiitem.h>

class QWidget;
class QStringList;
class KConfig;
class KDialogBase;

 /**
  * Easy message dialog box.
  *
  * Provides convenience functions for some i18n'ed standard dialogs.
  *
  * The text in message boxes is wrapped automatically. The text may either
  * be plaintext or richtext. If the text is plaintext, a newline-character
  * may be used to indicate the end of a paragraph.
  *
  * @author Waldo Bastian (bastian@kde.org)
  */
class KxmameMessageBox
{
public:
  /**
   * Button types.
   **/
 enum ButtonCode
 {
   Ok = 1,
   Cancel = 2,
   Yes = 3,
   No = 4,
   Continue = 5
 };

 enum DialogType
 {
   QuestionYesNo = 1,
   WarningYesNo = 2,
   WarningContinueCancel = 3,
   WarningYesNoCancel = 4,
   Information = 5,
   // Reserved for: SSLMessageBox = 6
   Sorry = 7,
   Error = 8,
   QuestionYesNoCancel = 9
 };

 enum OptionsType
 {
   Notify = 1,        ///< Emit a KNotifyClient event
   AllowLink = 2,     ///< The message may contain links.
   Dangerous = 4,     ///< The action to be confirmed by the dialog is a potentially destructive one
   PlainCaption = 8,  /**< Do not use KApplication::makeStdCaption()
   		           @since 3.4
		       */
   NoExec = 16        /**< Do not call exec() in createKMessageBox()
			   @since 3.4
		       */
 };

static void warning(QWidget *parent,
		const QString &text,
		const QString &caption = QString::null,
		int options = Notify);

static void warningListWId(WId parent_id,
		const QString &text,
		const QStringList &strlist,
		const QString &caption = QString::null,
		int options = Notify);

static int warningContinueCancel(QWidget *parent,
		const QString &text,
		const QString &caption = QString::null,
		const KGuiItem &buttonContinue = KStdGuiItem::cont(),
		const QString &dontAskAgainName = QString::null,
		int options = Notify);
/**
  * Display a "warning" dialog with a listbox to show information to the user.
  *
  * @param parent  If @p parent is 0, then the message box becomes an
  *                application-global modal dialog box. If @p parent is a
  *                widget, the message box becomes modal relative to parent.
  * @param text    Message string.
  * @param strlist List of strings to be written in the listbox. If the
  *                list is empty, it doesn't show any listbox, working
  *                as warningContinueCancel.
  * @param caption Message box title. The application name is added to
  *                the title. The default title is i18n("Warning").
  * @param buttonContinue The text for the first button.
  * @param dontAskAgainName If provided, a checkbox is added with which
  *                further confirmation can be turned off.
  *                The string is used to lookup and store the setting
  *                in the applications config file.
  *                The setting is stored in the "Notification Messages" group.
  *                If @p dontAskAgainName starts with a ':' then the setting
  *                is stored in the global config file.
  *
  * @param options  see OptionsType
  *
  * The second button always has the text "Cancel".
  *
  * @return  @p Continue is returned if the Continue-button is pressed.
  *          @p Cancel is returned if the Cancel-button is pressed.
  *
  * To be used for questions like "You are about to Print. Are you sure?"
  * the continueButton should then be labeled "Print".
  *
  * The default button is buttonContinue. Pressing "Esc" selects "Cancel".
  */
 static int warningContinueCancelList(int *listitemSelected, QWidget *parent,
                         const QString &text,
                         const QStringList &strlist,
                         const QString &caption = QString::null,
                         const KGuiItem &buttonContinue = KStdGuiItem::cont(),
                         const QString &dontAskAgainName = QString::null,
                         int options = Notify);

 /**
  * This function accepts the window id of the parent window, instead
  * of QWidget*. It should be used only when necessary.
  * @since 3.2
  */
 static int warningContinueCancelListWId(int *listitemSelected, WId parent_id,
                         const QString &text,
                         const QStringList &strlist,
                         const QString &caption = QString::null,
                         const KGuiItem &buttonContinue = KStdGuiItem::cont(),
                         const QString &dontAskAgainName = QString::null,
                         int options = Notify);


    /**
     * Create content and layout of a standard dialog
     *
     * @param dialog  The parent dialog base
     * @param icon    Which predefined icon the message box shall show.
     * @param text    Message string.
     * @param strlist List of strings to be written in the listbox.
     *                If the list is empty, it doesn't show any listbox
     * @param ask     The text of the checkbox. If empty none will be shown.
     * @param checkboxReturn The result of the checkbox. If it's initially
     *                true then the checkbox will be checked by default.
     * @param options  see OptionsType
     * @param details Detailed message string.
     * @since 3.3
     * @return A KDialogBase button code, not a KMessageBox button code,
     *         based on the buttonmask given to the constructor of the
     *         @p dialog (ie. will return KDialogBase::Yes [256] instead of 
     *         KMessageBox::Yes [3]). Will return KMessageBox::Cancel
     *         if the message box is queued for display instead of 
     *         exec()ed immediately or if the option NoExec is set.
     * @note   The @p dialog that is passed in is deleted by this
     *         function. Do not delete it yourself.
     */
 static int createKxmameMessageBox(int *listitemSelected, KDialogBase *dialog, QMessageBox::Icon icon,
                             const QString &text, const QStringList &strlist,
                             const QString &ask, bool *checkboxReturn,
                             int options, const QString &details=QString::null);

    /**
     * Create content and layout of a standard dialog
     *
     * @param dialog  The parent dialog base
     * @param icon    A QPixmap containing the icon to be displayed in the
     *                dialog next to the text.
     * @param text    Message string.
     * @param strlist List of strings to be written in the listbox.
     *                If the list is empty, it doesn't show any listbox
     * @param ask     The text of the checkbox. If empty none will be shown.
     * @param checkboxReturn The result of the checkbox. If it's initially
     *                true then the checkbox will be checked by default.
     * @param options  see OptionsType
     * @param details Detailed message string.
     * @param notifyType The type of notification to send when this message
     *                is presentend.
     * @since 3.3
     * @return A KDialogBase button code, not a KMessageBox button code,
     *         based on the buttonmask given to the constructor of the
     *         @p dialog (ie. will return KDialogBase::Yes [256] instead of 
     *         KMessageBox::Yes [3]). Will return KMessageBox::Cancel
     *         if the message box is queued for display instead of 
     *         exec()ed immediately or if the option NoExec is set.
     * @note   The @p dialog that is passed in is deleted by this
     *         function. Do not delete it yourself.
     */
 static int createKxmameMessageBox(int *listitemSelected, KDialogBase *dialog, QPixmap icon,
                             const QString &text, const QStringList &strlist,
                             const QString &ask, bool *checkboxReturn,
                             int options, const QString &details=QString::null,
                             QMessageBox::Icon notifyType=QMessageBox::Information);

};

#endif
