/*
 *			GPAC - MPEG-4 Systems C Development Kit
 *
 *			Copyright (c) Jean Le Feuvre 2000-2003 
 *					All rights reserved
 *
 *  This file is part of GPAC / RTP input plugin
 *
 *  GPAC is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *   
 *  GPAC is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *   
 *  You should have received a copy of the GNU General Public License
 *  along with GNU Make; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. 
 *		
 */

#include "rtp_in.h"




M4Err RP_SetupSDP(RTPClient *rtp, SDPInfo *sdp, RTPStream *stream)
{
	M4Err e;
	SDPMedia *media;
	Float Start, End;
	u32 i;
	char *sess_ctrl;
	X_Attribute *att;
	RTSPRange *range;
	RTPStream *ch;

	Start = 0.0;
	End = -1.0;

	sess_ctrl = NULL;
	range = NULL;

	for (i=0; i<ChainGetCount(sdp->Attributes); i++) {
		att = ChainGetEntry(sdp->Attributes, i);
		//session-level control string. Keep it in the current session if any
		if (!strcmp(att->Name, "control") && att->Value) sess_ctrl = att->Value;
		//NPT range only for now
		else if (!strcmp(att->Name, "range") && !range) range = RTSP_ParseRange(att->Value);
	}
	if (range) {
		Start = range->start;
		End = range->end;
		RTSP_DeleteRange(range);
	}
	
	//setup all streams
	for (i=0; i<ChainGetCount(sdp->media_desc); i++) {
		media = ChainGetEntry(sdp->media_desc, i);	
		ch = RP_NewStream(rtp, media, sdp, stream);
		//do not generate error if the channel is not created, just assume
		//1 - this is not an MPEG-4 configured channel -> not needed
		//2 - this is a 2nd describe and the channel was already created
		if (!ch) continue;

		e = RP_AddStream(rtp, ch, sess_ctrl);
		if (e) {
			RP_DeleteStream(ch);
			return e;
		}

		if (!ch->has_range) {
			ch->range_start = Start;
			ch->range_end = End;
			if (End != -1) ch->has_range = 1;
		}
	}
	return M4OK;
}

/*load iod from data:application/mpeg4-iod;base64*/
M4Err RP_SDPLoadIOD(RTPClient *rtp, char *iod_str)
{
	char *buf64, buf[2000];
	u32 size, size64;

	if (rtp->session_iod) return M4ServiceError;
	/*the only IOD format we support*/
	iod_str += 1;
	if (strnicmp(iod_str, "data:application/mpeg4-iod;base64", strlen("data:application/mpeg4-iod;base64")))
		return M4UnsupportedURL;

	buf64 = strstr(iod_str, ",");
	if (!buf64) return M4InvalidURL;
	buf64 += 1;
	size64 = strlen(buf64) - 1;

	size = Base64_dec(buf64, size64, buf, 2000);
	if (!size) return M4ServiceError;

	rtp->session_iod_len = size;
	rtp->session_iod = malloc(sizeof(char)*size);
	memcpy(rtp->session_iod, buf, size);
	return M4OK;
}

M4Err RP_EmulateIOD(RTPClient *rtp, u32 expect_type)
{
	ObjectDescriptor *the_od, *od;
	ESDescriptor *esd, *an_esd;
	LPODCODEC codec;
	ObjectDescriptorUpdate *odU;
	RTPStream *a_str, *v_str, *s_str;
	M4Err e;
	BitStream *bs;
	char *data, buffer[1024], buffer64[1024];
	u32 data_size, size64;
	u32 count;
	u32 i, bifsID;
	u32 w, h;
	count = ChainGetCount(rtp->channels);
	if (count>2) return M4NotSupported;

	if (expect_type==NM_OD_INTERACT) return M4NotSupported;
	if (expect_type==NM_OD_BIFS) return M4NotSupported;

	a_str = v_str = NULL;
	for (i=0; i<ChainGetCount(rtp->channels); i++) {
		RTPStream *ch = ChainGetEntry(rtp->channels, i);
		if (!a_str && ch->sl_map.StreamType==M4ST_AUDIO) a_str = ch;
		else if (!v_str && ch->sl_map.StreamType==M4ST_VISUAL) v_str = ch;
	}
	if (!v_str && !a_str) return M4InvalidURL;

	s_str = NULL;
	if (expect_type==NM_OD_VIDEO) {
		if (!v_str) return M4InvalidURL;
		s_str = v_str;
	} else if (expect_type==NM_OD_AUDIO) {
		if (!a_str) return M4InvalidURL;
		s_str = a_str;
	}

	/*single OD generation*/
	if (s_str) {
		the_od = (ObjectDescriptor *) OD_NewDescriptor(ObjectDescriptor_Tag);
		the_od->objectDescriptorID = 1;
		an_esd = OD_NewESDescriptor(0);
		an_esd->slConfig->timestampResolution = s_str->clock_rate;
		an_esd->slConfig->useRandomAccessPointFlag = 1;
		an_esd->slConfig->useTimestampsFlag = 1;
		an_esd->ESID = s_str->ES_ID;
		an_esd->decoderConfig->streamType = s_str->sl_map.StreamType;
		an_esd->decoderConfig->objectTypeIndication = s_str->sl_map.ObjectTypeIndication;
		if (s_str->sl_map.config) {
			an_esd->decoderConfig->decoderSpecificInfo->data = malloc(sizeof(char) * s_str->sl_map.configSize);
			memcpy(an_esd->decoderConfig->decoderSpecificInfo->data, s_str->sl_map.config, sizeof(char) * s_str->sl_map.configSize);
			an_esd->decoderConfig->decoderSpecificInfo->dataLength = s_str->sl_map.configSize;
		}
		ChainAddEntry(the_od->ESDescriptors, an_esd);
		e = OD_EncDesc((Descriptor *) the_od, &rtp->session_iod, &rtp->session_iod_len);
		OD_DeleteDescriptor((Descriptor**)&the_od);
		return e;
	}


	e = M4OK;
	bifsID = 0;
	if (a_str) bifsID = a_str->ES_ID;
	if (v_str && (bifsID<v_str->ES_ID) ) bifsID = v_str->ES_ID;
	bifsID++;
	the_od = (ObjectDescriptor *) OD_NewDescriptor(ObjectDescriptor_Tag);
	the_od->objectDescriptorID = 1;
	
	/*generate BIFS*/
	w = h = 10;
	if (v_str) {
		/*MPEG-4 video*/
		if ((v_str->sl_map.ObjectTypeIndication==0x20) && v_str->sl_map.config) {
			M4VDecoderSpecificInfo dsi;
			M4V_GetConfig(v_str->sl_map.config, v_str->sl_map.configSize, &dsi);
			w = dsi.width;
			h = dsi.height;
		}
		/*H263: use our private DSI*/
		else if (v_str->rtptype==RTP_HINT_H263) {
			BitStream *bs = NewBitStream(v_str->sl_map.config, v_str->sl_map.configSize, BS_READ);
			BS_ReadInt(bs, 32);
			w = BS_ReadInt(bs, 32);
			h = BS_ReadInt(bs, 32);
			DeleteBitStream(bs);
		}
	}
	esd = OD_NewESDescriptor(0);
	esd->slConfig->timestampResolution = 1000;
	esd->slConfig->useRandomAccessPointFlag = 1;
	esd->slConfig->useTimestampsFlag = 1;
	bs = NewBitStream(NULL, 0, BS_WRITE);
	BS_WriteInt(bs, 0, 10);
	BS_WriteInt(bs, 1, 1);
	BS_WriteInt(bs, 1, 1);
	BS_WriteInt(bs, 1, 1);
	BS_WriteInt(bs, w, 16);
	BS_WriteInt(bs, h, 16);
	esd->decoderConfig->streamType = M4ST_BIFS;
	esd->decoderConfig->objectTypeIndication = 0x01;
	BS_Align(bs);
	BS_GetContent(bs, (unsigned char **) &esd->decoderConfig->decoderSpecificInfo->data, &esd->decoderConfig->decoderSpecificInfo->dataLength);
	DeleteBitStream(bs);
	esd->OCRESID = esd->ESID = bifsID;
	if (a_str && v_str) {
		data = (char *) ISMA_BIFS_AV;
		data_size = 16;
	} else if (a_str) {
		data = (char *) ISMA_BIFS_AUDIO;
		data_size = 8;
	} else if (v_str) {
		data = (char *) ISMA_BIFS_VIDEO;
		data_size = 11;
	}
	size64 = Base64_enc(data, data_size, buffer64, 1024);
	buffer64[size64] = 0;
	sprintf(buffer, "data:application/mpeg4-bifs-au;base64,%s", buffer64);
	esd->URLString = strdup(buffer);
	ChainAddEntry(the_od->ESDescriptors, esd);
	
	/*generate OD*/
	esd = OD_NewESDescriptor(0);
	esd->slConfig->timestampResolution = 1000;
	esd->slConfig->useRandomAccessPointFlag = 1;
	esd->slConfig->useTimestampsFlag = 1;
	esd->decoderConfig->streamType = M4ST_OD;
	esd->decoderConfig->objectTypeIndication = 0x01;
	esd->ESID = bifsID + 1;
	esd->OCRESID = bifsID;
	codec = OD_NewCodec(OD_WRITE);
	odU = (ObjectDescriptorUpdate *) OD_NewCommand(ODUpdate_Tag);
	if (v_str) {
		od = (ObjectDescriptor *) OD_NewDescriptor(ObjectDescriptor_Tag);
		od->objectDescriptorID = ISMA_VIDEO_OD_ID;
		an_esd = OD_NewESDescriptor(0);
		an_esd->slConfig->timestampResolution = v_str->clock_rate;
		an_esd->slConfig->useRandomAccessPointFlag = 1;
		an_esd->slConfig->useTimestampsFlag = 1;
		an_esd->ESID = v_str->ES_ID;
		if (!an_esd->ESID) v_str->ES_ID = an_esd->ESID = bifsID + 2;
		an_esd->OCRESID = bifsID;
		an_esd->decoderConfig->streamType = v_str->sl_map.StreamType;
		an_esd->decoderConfig->objectTypeIndication = v_str->sl_map.ObjectTypeIndication;
		if (v_str->sl_map.config) {
			an_esd->decoderConfig->decoderSpecificInfo->data = malloc(sizeof(char) * v_str->sl_map.configSize);
			memcpy(an_esd->decoderConfig->decoderSpecificInfo->data, v_str->sl_map.config, sizeof(char) * v_str->sl_map.configSize);
			an_esd->decoderConfig->decoderSpecificInfo->dataLength = v_str->sl_map.configSize;
		}
		ChainAddEntry(od->ESDescriptors, an_esd);
		ChainAddEntry(odU->objectDescriptors, od);
	}
	if (a_str) {
		od = (ObjectDescriptor *) OD_NewDescriptor(ObjectDescriptor_Tag);
		od->objectDescriptorID = ISMA_AUDIO_OD_ID;
		an_esd = OD_NewESDescriptor(0);
		an_esd->slConfig->timestampResolution = a_str->clock_rate;
		an_esd->slConfig->useRandomAccessUnitsOnlyFlag = 1;
		an_esd->slConfig->useTimestampsFlag = 1;
		an_esd->ESID = a_str->ES_ID;
		if (!an_esd->ESID) a_str->ES_ID = an_esd->ESID = bifsID + 3;
		an_esd->OCRESID = bifsID;
		an_esd->decoderConfig->streamType = a_str->sl_map.StreamType;
		an_esd->decoderConfig->objectTypeIndication = a_str->sl_map.ObjectTypeIndication;
		if (a_str->sl_map.config) {
			an_esd->decoderConfig->decoderSpecificInfo->data = malloc(sizeof(char) * a_str->sl_map.configSize);
			memcpy(an_esd->decoderConfig->decoderSpecificInfo->data, a_str->sl_map.config, sizeof(char) * a_str->sl_map.configSize);
			an_esd->decoderConfig->decoderSpecificInfo->dataLength = a_str->sl_map.configSize;
		}
		ChainAddEntry(od->ESDescriptors, an_esd);
		ChainAddEntry(odU->objectDescriptors, od);
	}
	OD_AddCommand(codec, (ODCommand *) odU);
	OD_EncodeAU(codec);
	data = NULL;
	OD_GetEncodedAU(codec, &data, &data_size);
	OD_DeleteCodec(codec);

	size64 = Base64_enc(data, data_size, buffer64, 1024);
	buffer64[size64] = 0;
	sprintf(buffer, "data:application/mpeg4-od-au;base64,%s", buffer64);
	esd->URLString = strdup(buffer);
	ChainAddEntry(the_od->ESDescriptors, esd);
	free(data);

	/*finally encode IOD*/
	e = OD_EncDesc((Descriptor *) the_od, &rtp->session_iod, &rtp->session_iod_len);
	OD_DeleteDescriptor((Descriptor**)&the_od);
	return e;
}

void RP_LoadSDP(RTPClient *rtp, char *sdp_text, u32 sdp_len, RTPStream *stream)
{
	M4Err e;
	u32 i;
	SDPInfo *sdp;
	Bool is_isma_1;
	char *iod_str;
	X_Attribute *att;

	is_isma_1 = 0;
	iod_str = NULL;
	sdp = NewSDPInfo();
	e = SDP_Parse(sdp, sdp_text, sdp_len);

	if (e == M4OK) e = RP_SetupSDP(rtp, sdp, stream);

	/*root SDP, attach service*/
	if (! stream) {
		/*look for IOD*/
		if (e==M4OK) {
			for (i=0; i<ChainGetCount(sdp->Attributes); i++) {
				att = ChainGetEntry(sdp->Attributes, i);
				if (!iod_str && !strcmp(att->Name, "mpeg4-iod") ) iod_str = att->Value;
				if (!is_isma_1 && !strcmp(att->Name, "isma-compliance") ) {
					if (!stricmp(att->Value, "1,1.0,1")) is_isma_1 = 1;
				}
			}

			/*force iod reconstruction with ISMA to use proper clock dependencies*/
			if (is_isma_1) iod_str = NULL;

			/*folks at QT have weird notions of MPEG-4 systems, they use hardcoded IOD 
			with AAC ESDescriptor even when streaming AMR...*/
			if (iod_str) {
				for (i=0; i<ChainGetCount(rtp->channels); i++) {
					RTPStream *ch = ChainGetEntry(rtp->channels, i);
					if (ch->rtptype==RTP_HINT_AMR) {
						iod_str = NULL;
						break;
					}
				}
			}
			
			if (!iod_str) {
				e = RP_EmulateIOD(rtp, NM_OD_SCENE);
			} else {
				e = RP_SDPLoadIOD(rtp, iod_str);
			}
		}

		if (e) {
			rtp->status = NM_Unavailable;
		} else {
			rtp->status = NM_Connected;
		}
		/*attach service*/
		NM_OnConnect(rtp->service, NULL, e);
	}
	/*channel SDP */
	else {
		if (e) {
			NM_OnConnect(rtp->service, stream->channel, e);
			stream->status = NM_Unavailable;
		} else {
			/*connect*/
			RP_SetupChannel(stream, NULL);
		}
	}

	if (sdp) SDP_Delete(sdp);
}

